package org.lsst.ccs.gconsole.plugins.tracer.filters;

import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusAgentInfo;
import org.lsst.ccs.bus.messages.StatusAlert;
import org.lsst.ccs.bus.messages.StatusCommandDictionary;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateBundle;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.plugins.tracer.FilteredMessage;
import org.lsst.ccs.gconsole.plugins.tracer.MessageFilter;
import org.lsst.ccs.gconsole.plugins.tracer.Tracer;

/**
 * Message filter that selects messages based on their class.
 *
 * @author onoprien
 */
public class MessageClassSelector implements MessageFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final Class<?> clazz;

// -- Life cycle : -------------------------------------------------------------
    
    private MessageClassSelector(Class<?> clazz) {
        this.clazz = clazz;
    }
    
// -- Filtering : --------------------------------------------------------------
    
    @Override
    public FilteredMessage apply(FilteredMessage filteredMessage) {
        return clazz.isInstance(filteredMessage.getMessage()) ? filteredMessage : null;
//        return clazz.isAssignableFrom(filteredMessage.getMessage().getClass()) ? filteredMessage : null;
    }
    
// -- Creators : ---------------------------------------------------------------
    
    @Create(category = Tracer.CATEGORY,
            name = "Heart Beat",
            path = "Built-In/Message type/HeartBeat",
            description = "Accepts only heartbeat messages. No modifications.")
    static public Tracer heartBeat() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new MessageClassSelector(StatusHeartBeat.class));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "AgentInfo",
            path = "Built-In/Message type/AgentInfo",
            description = "Accepts only StatusAgentInfo messages. No modifications.")
    static public Tracer agentInfo() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new MessageClassSelector(StatusAgentInfo.class));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "StateBundle",
            path = "Built-In/Message type/StateBundle",
            description = "Accepts only StatusStateBundle messages. No modifications.")
    static public Tracer stateBundle() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new MessageClassSelector(StatusStateBundle.class));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "Alert",
            path = "Built-In/Message type/Alert",
            description = "Accepts only alert-related status messages. No modifications.")
    static public Tracer alert() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new MessageClassSelector(StatusAlert.class));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "Command Dictionary",
            path = "Built-In/Message type/CommandDictionary",
            description = "Accepts only status messages that contain a command dictionary. No modifications.")
    static public Tracer commandDictionary() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new MessageClassSelector(StatusCommandDictionary.class));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "Data Dictionary",
            path = "Built-In/Message type/DataDictionary",
            description = "Accepts only status messages that contain a data dictionary. No modifications.")
    static public Tracer dataDictionary() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new MessageClassSelector(StatusDataProviderDictionary.class));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "State Change",
            path = "Built-In/Message type/StateChangeNotification",
            description = "Accepts only state change messages. No modifications.")
    static public Tracer stateChangeNotification() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new MessageClassSelector(StatusStateChangeNotification.class));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "Configuration Info",
            path = "Built-In/Message type/ConfigurationInfo",
            description = "Accepts only configuration information status messages. No modifications.")
    static public Tracer configurationInfo() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new MessageClassSelector(StatusConfigurationInfo.class));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "LOG bus",
            path = "Built-In/Bus/LOG",
            description = "Accepts only LOG bus messages. No modifications.")
    static public Tracer logBus() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new MessageClassSelector(LogMessage.class));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "STATUS bus",
            path = "Built-In/Bus/STATUS",
            description = "Accepts only STATUS bus messages. No modifications.")
    static public Tracer statusBus() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new MessageClassSelector(StatusMessage.class));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "COMMAND bus",
            path = "Built-In/Bus/COMMAND",
            description = "Accepts only COMMAND bus messages. No modifications.")
    static public Tracer commandBus() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new MessageClassSelector(CommandMessage.class));
        return tracer;
    }

}

