package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Component;
import java.awt.Dialog;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.util.concurrent.CancellationException;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.PersistableAgentChannelsFilter;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.services.persist.Persistable;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;

/**
 * Dialog to select monitoring data view type and filter.
 * <p>
 * Currently implemented to look at these sources:<ul>
 * <li>Objects of type {@code AgentChannelsFilter} in console lookup.
 * <li>Strings with ID "AgentChannelsFilter" and "MonitorView" in console lookup.
 </ul>
 * @author onoprien
 */
final class MonitorPageDialog extends JDialog {

// -- Fields : -----------------------------------------------------------------
    
    private final String defaultName = "Monitor";
    private final PersistenceService service = Console.getConsole().getSingleton(PersistenceService.class);
    private String viewCategory, filterCategory;
    
    private PersistableMonitorView view;
    private PersistableAgentChannelsFilter filter;
    private final MonitorPage.Descriptor descriptor;
    private MonitorPage page;
    
    private JTextField nameField, viewField, filterField;

// -- Life cycle : -------------------------------------------------------------
    
    private MonitorPageDialog(MonitorPage.Descriptor before, String title, Component parent) {
        super(parent == null ? null : SwingUtilities.getWindowAncestor(parent), 
                title == null ? "Open new monitoring page" : title,
                Dialog.ModalityType.APPLICATION_MODAL);
        
        String viewName = "";
        String filterName = "";
        descriptor = before;
        Persistable.Descriptor desc = descriptor.getView();
        if (desc != null && desc.getCategory() != null) {
            viewCategory = desc.getCategory();
            try {
                view = (PersistableMonitorView) service.make(desc);
                String name = view.getName();
                viewName = (name == null || name.trim().isEmpty()) ? view.getClass().getSimpleName() : name;
            } catch (RuntimeException x) {
                view = null;
            }
        } else {
            viewCategory = MonitorView.CATEGORY;
            view = (PersistableMonitorView) service.make(MonitorView.CATEGORY, "Built-In/Default");
            viewName = view.getName();
        }
        desc = descriptor.getFilter();
        if (desc != null && desc.getCategory() != null) {
            filterCategory = desc.getCategory();
            try {
                filter = (PersistableAgentChannelsFilter) service.make(desc);
                String name = filter.getName();
                filterName = (name == null || name.trim().isEmpty()) ? filter.getClass().getSimpleName() : name;
            } catch (RuntimeException x) {
                filter = null;
            }
        } else {
            filterCategory = AgentChannelsFilter.CATEGORY;
            filter = (PersistableAgentChannelsFilter) service.make(AgentChannelsFilter.CATEGORY, "Built-In/Unfiltered");
            filterName = filter.getName();
        }
        
        GridBagLayout gridbag = new GridBagLayout();
        setLayout(gridbag);
        
        GridBagConstraints c = new GridBagConstraints();
        c.insets = new Insets(3, 8, 3, 8);
        
        c.anchor = GridBagConstraints.EAST;
        
        c.gridx = 0;
        c.gridy = 0;
        JLabel label = new JLabel("Page name:");
        gridbag.setConstraints(label, c);
        add(label);
        c.gridy = 1;
        label = new JLabel("View type:");
        gridbag.setConstraints(label, c);
        add(label);
        c.gridy = 2;
        label = new JLabel("Filter:");
        gridbag.setConstraints(label, c);
        add(label);
        
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        
        c.gridx = 1;
        c.gridy = 0;
        nameField = new JTextField(descriptor.getName() == null ? defaultName : descriptor.getName(), 50);
        gridbag.setConstraints(nameField, c);
        add(nameField);
        
        c.gridy = 1;
        viewField = new JTextField(viewName, 50);
        viewField.setEditable(false);
        gridbag.setConstraints(viewField, c);
        add(viewField);
        
        c.gridy = 2;
        filterField = new JTextField(filterName, 50);
        filterField.setEditable(false);
        gridbag.setConstraints(filterField, c);
        add(filterField);
        
        c.fill = GridBagConstraints.NONE;
        c.weightx = 0.;
        
        c.gridx = 2;
        c.gridy = 1;
        JButton button = new JButton("Choose...");
        button.setToolTipText("Select view");
        button.addActionListener(e -> {
            PersistableMonitorView v;
            try {
                if (view == null) {
                    v = (PersistableMonitorView) service.make(descriptor.getView(), "Select view", parent, viewCategory);
                } else {
                    v = (PersistableMonitorView) service.edit(view, "Select view", parent);
                }
            } catch (RuntimeException x) {
                v = null;
            }
            if (v != null) {
                view = v;
                descriptor.setView(view.save());
                String name = view.getName();
                if (name == null || name.trim().isEmpty()) {
                    name = view.getClass().getSimpleName();
                }
                viewField.setText(name);
            }
        });
        gridbag.setConstraints(button, c);
        add(button);
        
        c.gridy = 2;
        button = new JButton("Choose...");
        button.setToolTipText("Select filter");
        button.addActionListener(e -> {
            PersistableAgentChannelsFilter f;
            try {
                if (filter == null) {
                    f = (PersistableAgentChannelsFilter) service.make(descriptor.getFilter(), "Select filter", parent, filterCategory);
                } else {
                    f = (PersistableAgentChannelsFilter) service.edit(filter, "Select filter", parent);
                }
            } catch (RuntimeException x) {
                f = null;
            }
            if (f != null) {
                filter = f;
                descriptor.setFilter(filter.save());
                String name = filter.getName();
                if (name == null || name.trim().isEmpty()) {
                    name = filter.getClass().getSimpleName();
                }
                filterField.setText(name);
            }
        });
        gridbag.setConstraints(button, c);
        add(button);
        
        c.gridx = 0;
        c.gridy = 3;
        c.anchor = GridBagConstraints.SOUTHEAST;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.weighty = 1.;
        c.insets = new Insets(16, 8, 4, 8);
        
        JPanel box = new JPanel();
        box.setLayout(new GridLayout(1,2,16,5));
        button = new JButton("Cancel");
        button.addActionListener(e -> {
            view = null;
            filter = null;
            page = null;
            dispose();
        });
        box.add(button);
        button = new JButton("OK");
        button.addActionListener(e -> {
            if (done()) {
                dispose();
            }
        });
        box.add(button);
        gridbag.setConstraints(box, c);
        add(box);
    }
    
    static MonitorPage edit(MonitorPage page, String title, Component parentComponent) {
        MonitorPage.Descriptor before = page == null ? new MonitorPage.Descriptor() : page.save();
        MonitorPageDialog dialog = new MonitorPageDialog(before, title, parentComponent);
        dialog.setSize(dialog.getPreferredSize());
        dialog.pack();
        dialog.setLocationRelativeTo(parentComponent);
        dialog.setVisible(true);
        if (dialog.page == null) throw new CancellationException();
        return dialog.page;
    }

    
// -- Local methods : ----------------------------------------------------------
    
    private boolean done() {

        if (view == null) {
            Console.getConsole().error("Please select view type");
            return false;
        }
        if (filter == null) {
            Console.getConsole().error("Please select filter");
            return false;
        }
        String name = nameField.getText();
        if (!name.trim().isEmpty()) {
            descriptor.setName(name);
        }
        try {
            page = new MonitorPage(view, filter, name);
            String pageCategory = descriptor == null ? null : descriptor.getCategory();
            page.getDescriptor().setCategory(pageCategory == null ? MonitorPage.CATEGORY : pageCategory);
//            page = new MonitorPage(descriptor); // FIXME: can I reuse existing view and filter? If not, do I need them?
        } catch (RuntimeException x) {
            Console.getConsole().error("Cannot create motinor page", x);
            return false;
        }
        return true;
    }

}
