package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.beans.Transient;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.AbstractAction;
import javax.swing.AbstractCellEditor;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.SwingConstants;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeWillExpandListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.ExpandVetoException;
import javax.swing.tree.TreeCellEditor;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.services.persist.Persistable;

/**
 * {@link MonitorView} that displays monitoring data as a tree of tables.
 * Unlike the original {@code TreeView}, this class is optimized for working with a large number of channels.
 * All access to instances of this class, including construction, should happen on the EDT.
 *
 * @author onoprien
 */
public class LazyTreeView extends AbstractMonitorView {

// -- Fields : -----------------------------------------------------------------
    
    static public final String CREATOR_PATH = "Built-In/Basic/Tree";
    static public final String CREATOR_PATH_CONF = "Built-In/Basic/Tree, Configurable";
    
    private Descriptor descriptor;
    
    private final RootNode root;
    DefaultTreeModel treeModel;
    private final Tree tree;
    private final JScrollPane scrollPane;
    
    private boolean doScroll = true;

// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = MonitorView.CATEGORY,
            name = "Lazy Tree View",
            path = CREATOR_PATH,
            description = "Monitoring data view that displays its data in a tree of tables.")
    public LazyTreeView() {
        descriptor = new Descriptor();
        root = new RootNode();
        treeModel = new DefaultTreeModel(root);
        tree = new Tree(treeModel);
        tree.addTreeWillExpandListener(new TreeWillExpandListener() {
            @Override
            public void treeWillExpand(TreeExpansionEvent event) throws ExpandVetoException {
                InterNode node = (InterNode) event.getPath().getLastPathComponent();
                node.getDescriptor().setExpanded(true);
                if (node.isLazy()) {
                    node.rebuild();
                }
            }
            @Override
            public void treeWillCollapse(TreeExpansionEvent event) throws ExpandVetoException {
                InterNode node = (InterNode) event.getPath().getLastPathComponent();
                node.getDescriptor().setExpanded(false);
            }
        });
        scrollPane = new JScrollPane(tree);
        root.build(Collections.emptyList());
    }
    
    @Create(category = MonitorView.CATEGORY,
            name = "Lazy Tree View",
            path = CREATOR_PATH_CONF,
            description = "Monitoring data view that displays its data in a tree of tables.")
    public LazyTreeView(
            @Par(def = Par.NULL, desc = "List of fields to display. If not specified, the set of fields provided by the filter is used. Available standard fields: VALUE, UNITS, LOW_WARN, LOW_ALARM, ALERT_LOW, HIGH_WARN, HIGH_ALARM, ALERT_HIGH, DESCR") List<String> fields
    ) {
        this();
        if (fields != null) {
            descriptor.setFields(fields.toArray(new String[0]));
        }
    }
    

    @Override
    protected void resetChannels() {
        root.setLoading(true);
        updateDescriptor(root);
        root.clear();
        root.build(data.entrySet());
        root.setLoading(false);
        treeModel.reload();
        root.restoreExpansion();
    }


// -- Getters and setters : ----------------------------------------------------
    
    /**
     * Returns the graphical component maintained by this view.
     * @return Graphical component to be displayed by the GUI.
     */
    @Override
    public JComponent getPanel() {
        return scrollPane;
    }
    
    
// -- Customized JTree : -------------------------------------------------------

    final class Tree extends JTree implements MonitorDisplay {
        
        Tree(TreeModel model) {
            super(model);
            setRootVisible(true);
            setShowsRootHandles(true);
            setRowHeight(0);
            setCellRenderer(new TreeRenderer());
            setCellEditor(new TreeEditor());
            addMouseMotionListener(new MouseMotionListener() {
                @Override
                public void mouseMoved(MouseEvent e) {
                    if (tree.getRowForLocation(e.getX(), e.getY()) != -1) {
                        tree.startEditingAtPath(tree.getPathForLocation(e.getX(), e.getY()));
                    }
                }
                @Override
                public void mouseDragged(MouseEvent e) {
                }
            });
            setEditable(true);
            getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
            setScrollsOnExpand(false);
        } 

        @Override
        public int getScrollableUnitIncrement(Rectangle visibleRect, int orientation, int direction) {
            int out = super.getScrollableUnitIncrement(visibleRect, orientation, direction);
            if (orientation == SwingConstants.VERTICAL) {
                int limit = visibleRect.height/8;
                if (out > limit) return limit;
            }
            return out;
        }
        
        @Override
        public void saveData(OutputStream out, String mimeType) {
            List<AgentChannel> channels = data.values().stream().map(handle -> handle.getChannel()).filter(channel -> channel != null).collect(Collectors.toList());
            LsstMonitorPlugin.saveData(out, mimeType, channels, fields);
        }

        @Override
        public void scrollPathToVisible(TreePath path) {
            if (doScroll) super.scrollPathToVisible(path);
        }

        @Override
        public void startEditingAtPath(TreePath path) {
            doScroll = false;
            super.startEditingAtPath(path);
            doScroll = true;
        }
        
    }

    class TreeRenderer extends DefaultTreeCellRenderer {
        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel, boolean expanded, boolean leaf, int row, boolean hasFocus) {
            if (value instanceof DisplayNode) {
                DisplayNode node = (DisplayNode) value;
                return node.getComponent();
            } else {
                return super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
            }
        }
    }

    class TreeEditor extends AbstractCellEditor implements TreeCellEditor {

        private final TreeRenderer renderer;
        private Object editorValue;
        
        private final JPopupMenu popup;
        private final Action compactAct, sectionAct;
        private final EnumMap<InterNodeDescriptor.DisplayMode, AbstractAction> modeAct;
        private final EnumMap<InterNodeDescriptor.Sort, AbstractAction> sortAct;

        public TreeEditor() {
            
            renderer = new TreeRenderer();
            
            popup = new JPopupMenu();
            JMenu menu = new JMenu("Display mode...");
            JMenuItem menuItem;
            
            modeAct = new EnumMap<>(InterNodeDescriptor.DisplayMode.class);
            for (InterNodeDescriptor.DisplayMode mode : InterNodeDescriptor.DisplayMode.values()) {
                AbstractAction act = new AbstractAction(mode.toString()) {
                    @Override
                    public void actionPerformed(ActionEvent e) {
                        if (editorValue instanceof InterNode) {
                            InterNode node = (InterNode) editorValue;
                            modeAct.values().forEach( act -> act.putValue(AbstractAction.SELECTED_KEY, act.equals(this)));
                            node.getDescriptor().setDisplayMode(mode);
                            node.rebuild();
                        }
                    }
                };
                modeAct.put(mode, act);
                menuItem = new JCheckBoxMenuItem(act);
                menu.add(menuItem);
            }
            popup.add(menu);
            
            menu = new JMenu("Order...");
            sortAct = new EnumMap<>(InterNodeDescriptor.Sort.class);
            for (InterNodeDescriptor.Sort mode : InterNodeDescriptor.Sort.values()) {
                AbstractAction act = new AbstractAction(mode.toString()) {
                    @Override
                    public void actionPerformed(ActionEvent e) {
                        if (editorValue instanceof InterNode) {
                            InterNode node = (InterNode) editorValue;
                            InterNodeDescriptor.Sort selectedMode = null;
                            for (Map.Entry<InterNodeDescriptor.Sort, AbstractAction> ee : sortAct.entrySet()) {
                                AbstractAction action = ee.getValue();
                                boolean isChosen = (action == this);
                                action.putValue(AbstractAction.SELECTED_KEY, isChosen);
                                if (isChosen) selectedMode = ee.getKey();
                            }
                            node.getDescriptor().setSort(selectedMode);
                            node.rebuild();
                        }
                    }
                };
                sortAct.put(mode, act);
                menuItem = new JCheckBoxMenuItem(act);
                menu.add(menuItem);
            }
            popup.add(menu);
            
            compactAct = new AbstractAction("Compact") {
                @Override
                public void actionPerformed(ActionEvent e) {
                    if (editorValue instanceof InterNode) {
                        InterNode node = (InterNode) editorValue;
                        Boolean v = (Boolean) getValue(AbstractAction.SELECTED_KEY);
                        node.getDescriptor().setCompact(Boolean.TRUE.equals(v));
                        node.rebuild();
                    }
                }
            };
            menuItem = new JCheckBoxMenuItem(compactAct);
            popup.add(menuItem);
            sectionAct = new AbstractAction("Sectioned") {
                @Override
                public void actionPerformed(ActionEvent e) {
                    if (editorValue instanceof InterNode) {
                        InterNode node = (InterNode) editorValue;
                        Boolean v = (Boolean) getValue(AbstractAction.SELECTED_KEY);
                        node.getDescriptor().setSection(Boolean.TRUE.equals(v));
                        node.rebuild();
                    }
                }
            };
            menuItem = new JCheckBoxMenuItem(sectionAct);
            popup.add(menuItem);
            renderer.addMouseListener(new MouseAdapter() {
                public void mousePressed(MouseEvent e) {
                    maybeShowPopup(e);
                }
                public void mouseReleased(MouseEvent e) {
                    maybeShowPopup(e);
                }
                private void maybeShowPopup(MouseEvent e) {
                    if (e.isPopupTrigger()) {
                        popup.show(e.getComponent(), e.getX(), e.getY());
                    }
                }
            });

        }

        @Override
        public Component getTreeCellEditorComponent(JTree tree, Object value, boolean isSelected, boolean expanded, boolean leaf, int row) {
            editorValue = value;
            if (editorValue instanceof InterNode) {
                InterNode node = (InterNode) editorValue;
                compactAct.putValue(AbstractAction.SELECTED_KEY, node.getDescriptor().isCompact());
                sectionAct.putValue(AbstractAction.SELECTED_KEY, node.getDescriptor().isSection());
                for (Map.Entry<InterNodeDescriptor.DisplayMode, AbstractAction> e : modeAct.entrySet()) {
                    e.getValue().putValue(AbstractAction.SELECTED_KEY, e.getKey().equals(node.getDescriptor().getDisplayMode()));
                }
                for (Map.Entry<InterNodeDescriptor.Sort, AbstractAction> e : sortAct.entrySet()) {
                    e.getValue().putValue(AbstractAction.SELECTED_KEY, e.getKey().equals(node.getDescriptor().getSort()));
                }
            }
            return renderer.getTreeCellRendererComponent(tree, value, isSelected, expanded, leaf, row, true);
        }

        /** At this point the component is added to the tree (undocumented!) but tree's internal cleanup might not yet be done. */
        @Override
        public boolean shouldSelectCell(EventObject anEvent) {
            return false;
        }

        @Override
        public Object getCellEditorValue() {
            return editorValue;
        }
    }


// -- Tree node classes : ------------------------------------------------------
    
    /**  Tree node. */
    private class Node extends DefaultMutableTreeNode {
        
        Node(String name) {
            super(name);
        }
        
        /** Destroys the tree below this node. */
        void clear() {
            Enumeration<TreeNode> en = children();
            while (en.hasMoreElements()) {
                ((Node)en.nextElement()).clear();
            }
            removeAllChildren();
        }
        
        /** Returns a bean describing the current state of this node, or {@code null} if this node is in the default state. */
        Serializable save() {
            return null;
        }
        
        /** Returns a slash-separated path to this node, excluding the root. */
        protected String getPathString() {
            StringJoiner sj = new StringJoiner("/");
            for (TreeNode node : getPath()) {
                if (node != root) {
                    sj.add(node.toString());
                }
            }
            return sj.toString();
        }
    }
    
    /** Intermediate node - its path is a partial path of one or more channels. */
    private class InterNode extends Node {
        
        // Fields:
        
        protected final String LAZY = "-"; // Name for placeholder nodes indicating unloaded branch
        
        protected InterNodeDescriptor desc; // current settings for this node
        protected Collection<Map.Entry<String,DisplayChannel>> data; // list of path:channel below this node, in original order
       
        // Life cycle:
        
        InterNode(String name) {
            super(name);
        }
        
        /** Builds a tree below this node. */
        void build(Collection<Map.Entry<String,DisplayChannel>> data) {
            
            this.data = data;
            String pathKey = getPathString();
            TreeMap<String, Serializable> nodes = LazyTreeView.this.getDescriptor().getNodes();
            Serializable sz = nodes == null ? null : nodes.get(pathKey);
            if (sz instanceof InterNodeDescriptor) {
                desc = (InterNodeDescriptor) sz;
            } else {
                desc = new InterNodeDescriptor();
                if (this == root) {
                    desc.setExpanded(true);
                }
            }
            
            // create a dummy node and stop here if this node is not expanded
            
            if (!desc.isExpanded()) {
                add(new Node(LAZY));
                return;
            }
            
            // preprocess data: make sections, sort, etc.
            
            ArrayList<Map.Entry<String,DisplayChannel>> processedData;
            if (desc.isSection()) {
                processedData = new ArrayList<>(data.size());
                for (Map.Entry<String,DisplayChannel> e : data) {
                    String processedKey = e.getKey().replaceFirst("/+", "//");
                    processedData.add(new AbstractMap.SimpleImmutableEntry<>(processedKey, e.getValue()));
                }
            } else {
                processedData = new ArrayList<>(data);
            }
            
            InterNodeDescriptor.Sort sortAlgorithm = desc.getSort();
            if (!InterNodeDescriptor.Sort.NONE.equals(sortAlgorithm)) {
                processedData.sort(sortAlgorithm);
            }
            
            // create child nodes
            
            if (desc.isFlat()) {  // everything goes into a single SectionedTable
                
                String key = getPathString();
                if (!key.isEmpty()) key += "/";
                key += SectionedTable.class.getName();
                Serializable disDesc = nodes == null ? null : nodes.get(key);
                SectionedTable.Descriptor descriptor = (!(disDesc instanceof SectionedTable.Descriptor)) ? null : (SectionedTable.Descriptor)disDesc;
                SectionedTable tableModel = SectionedTable.getInstance(processedData, desc.isCompact() ? compactFields : fields, descriptor);
                DisplayNode dNode = new MonitorTableNode(tableModel);
                tableModel.setListener(dNode);
                add(dNode);
                
            } else {
                
                ArrayList<Map.Entry<String,DisplayChannel>> leaves = new ArrayList<>(processedData.size());
                ArrayList<Map.Entry<String,DisplayChannel>> mesh = new ArrayList<>(processedData.size());
                LinkedHashMap<String, ArrayList<Map.Entry<String,DisplayChannel>>> branches = new LinkedHashMap<>();
                
                for (Map.Entry<String,DisplayChannel> e : processedData) {
                    String[] pathSegments = e.getKey().split("/+");
                    if (pathSegments.length < 2) {
                        leaves.add(e);
                    } else if (desc.isMesh() && (pathSegments.length == 2 || pathSegments.length == 3)) {
                        if (desc.isFlip()) {
                            String dp = pathSegments.length == 2 ? pathSegments[1]+"/"+pathSegments[0] : pathSegments[0]+"/"+pathSegments[2]+"/"+pathSegments[1];
                            e = new AbstractMap.SimpleImmutableEntry<>(dp, e.getValue());
                        }
                        mesh.add(e);
                    } else {
                        ArrayList<Map.Entry<String,DisplayChannel>> branch = branches.get(pathSegments[0]);
                        if (branch == null) {
                            branch = new ArrayList<>();
                            branches.put(pathSegments[0], branch);
                        }
                        StringJoiner sj = new StringJoiner("/");
                        for (int i=1; i<pathSegments.length; i++) {
                            sj.add(pathSegments[i]);
                        }
                        branch.add(new AbstractMap.SimpleImmutableEntry<>(sj.toString(), e.getValue()));
                    }
                }
                
                if (!leaves.isEmpty()) {
                    SectionedTable tableModel = SectionedTable.getInstance(leaves, desc.isCompact() ? compactFields : fields, null);
                    DisplayNode dNode = new MonitorTableNode(tableModel);
                    tableModel.setListener(dNode);
                    add(dNode);
                }
                
                if (!mesh.isEmpty()) {
                    MeshTable tableModel = MeshTable.getInstance(mesh, desc.isCompact() ? compactFields : fields);
                    DisplayNode dNode = new MonitorTableNode(tableModel);
                    tableModel.setListener(dNode);
                    add(dNode);
                }
                
                if (!branches.isEmpty()) {
                    for (Map.Entry<String, ArrayList<Map.Entry<String,DisplayChannel>>> e : branches.entrySet()) {
                        InterNode child = new InterNode(e.getKey());
                        add(child);
                        e.getValue().trimToSize();
                        child.build(e.getValue());
                    }
                }
               
            }
            
        }
        
        /** Trigger rebuilding the tree below this node. Make sure this node is expanded. */
        protected void rebuild() {
            if (data != null && !data.isEmpty()) {
                desc.setExpanded(true);
                Collection<Map.Entry<String,DisplayChannel>> data = this.data;
                updateDescriptor(this);
                clear();
                build(data);
                treeModel.nodeStructureChanged(this);
                restoreExpansion();
            }
        }
        
        @Override
        void clear() {
            super.clear();
            data = null;
        }
        
        @Override
        InterNodeDescriptor save() {
            return desc.getFlags() == 0 ? null : desc.clone();
        }
        
        void restoreExpansion() {
            Enumeration<TreeNode> en = depthFirstEnumeration();
            while (en.hasMoreElements()) {
                TreeNode node = en.nextElement();
                if (node instanceof InterNode) {
                    InterNode n = (InterNode) node;
                    if (n.desc.isExpanded()) {
                        tree.expandPath(new TreePath(n.getPath()));
                    }
                }
            }
        }
        
        // Setters/getters :
        
        boolean isLazy() {
            return children != null && this.children.size() == 1 && this.children.get(0).toString().equals(LAZY);
        }
        
        public InterNodeDescriptor getDescriptor() {
            return desc;
        }

    }
    
    public static final class InterNodeDescriptor implements Serializable, Cloneable {
        
        public static enum DisplayMode {TREE, FLAT, MESH, FLIPPED_MESH}
        
        /**
         * Enumeration of sorting algorithms supported by {@code InterNode}.
         */
        public static enum Sort implements Comparator<Map.Entry<String,DisplayChannel>> {
            
            /** No sorting - channels are displayed or passed to children in the order in which they were supplied by the parent. */
            NONE {
                @Override
                public int compare(Map.Entry<String, DisplayChannel> o1, Map.Entry<String, DisplayChannel> o2) {
                    throw new UnsupportedOperationException("This value indicates no sorting");
                }
            },
            
            /** Alphabetic sorting by display path. */
            ALPHABETIC {
                @Override
                public int compare(Map.Entry<String, DisplayChannel> e1, Map.Entry<String, DisplayChannel> e2) {
                    return e1.getKey().compareTo(e2.getKey());
                }
            };

            @Override
            abstract public int compare(Map.Entry<String, DisplayChannel> o1, Map.Entry<String, DisplayChannel> o2);
        }
        
        static final int FLAT = 0b1;
        static final int MESH = 0b10;
        static final int FLIP = 0b100;
        static final int COMPACT = 0b1000;
        static final int SECTION = 0b10000;
        static final int EXPANDED = 0b100000;
        static final int SORT = 0b1000000;

        private int flags;
        
        public InterNodeDescriptor() {
        }

        public int getFlags() {
            return flags;
        }
        
        @Transient
        public DisplayMode getDisplayMode() {
            if (isFlat()) {
                return DisplayMode.FLAT;
            } else if (isMesh()) {
                return isFlip() ? DisplayMode.FLIPPED_MESH : DisplayMode.MESH;
            } else {
                return DisplayMode.TREE;
            }
        }

        @Transient
        public boolean isFlat() {
            return (flags & FLAT) > 0;
        }

        @Transient
        public boolean isMesh() {
            return (flags & MESH) > 0;
        }

        @Transient
        public boolean isFlip() {
            return (flags & FLIP) > 0;
        }

        @Transient
        public boolean isCompact() {
            return (flags & COMPACT) > 0;
        }

        @Transient
        public boolean isSection() {
            return (flags & SECTION) > 0;
        }
        
        @Transient
        public boolean isExpanded() {
            return (flags & EXPANDED) > 0;
        }
        
        @Transient
        public Sort getSort() {
            return (flags & SORT) > 0 ? Sort.ALPHABETIC : Sort.NONE;
        }

        public void setFlags(int flags) {
            this.flags = flags;
        }
        
        @Transient
        public void setDisplayMode(DisplayMode value) {
            switch (value) {
                case TREE:
                    setFlat(false);
                    setMesh(false);
                    setFlip(false);
                    break;
                case FLAT:
                    setFlat(true);
                    setMesh(false);
                    setFlip(false);
                    break;
                case MESH:
                    setFlat(false);
                    setMesh(true);
                    setFlip(false);
                    break;
                 case FLIPPED_MESH:
                    setFlat(false);
                    setMesh(true);
                    setFlip(true);
                    break;
           }
        }

        @Transient
        public void setFlat(boolean value) {
            setBits(value, FLAT);
        }

        @Transient
        public void setMesh(boolean value) {
            setBits(value, MESH);
        }

        @Transient
        public void setFlip(boolean value) {
            setBits(value, FLIP);
        }

        @Transient
        public void setCompact(boolean value) {
            setBits(value, COMPACT);
        }

        @Transient
        public void setSection(boolean value) {
            setBits(value, SECTION);
        }
        
        @Transient
        public void setExpanded(boolean value) {
            setBits(value, EXPANDED);
        }
        
        @Transient
        public void setSort(Sort value) {
            setBits(Sort.ALPHABETIC.equals(value), SORT);
        }
        
        private void setBits(boolean value, int mask) {
            if (value) {
                flags |= mask;
            } else {
                flags &= ~mask;
            }            
        }

        @Override
        protected InterNodeDescriptor clone() {
            try {
                return (InterNodeDescriptor) super.clone();
            } catch (CloneNotSupportedException x) {
                return null; // never
            }
        }
        
    }
    
    private final class RootNode extends InterNode {
        
        RootNode() {
            super("");
            setLoading(false);
        }

        @Override
        InterNodeDescriptor save() {
            return desc.getFlags() == InterNodeDescriptor.EXPANDED ? null : desc.clone();
        }
        
        void setLoading(boolean loading) {
            super.setUserObject(loading ? "Loaging..." : "Camera Control System");
            if (treeModel != null) treeModel.nodeChanged(this);
        }

        @Override
        void restoreExpansion() {
            if (!desc.isExpanded()) {
                tree.collapsePath(new TreePath(getPath()));
            }
            super.restoreExpansion();
        }
        
    }
    
    /**
     * Leaf node that displays data. Currently, {@code MonitorTableNode} is the only implementation.
     */
    protected abstract class DisplayNode extends Node implements MonitorTable.Listener {
        
        DisplayNode() {
            super("display");
        }
        
        DisplayNode(String name) {
            super(name);
        }
        
        abstract JComponent getComponent();

        @Override
        public void stateChanged(MonitorTable.Event e) {
            if (parent == null) return;
            switch (e.getReason()) {
                case CELLS:
                    treeModel.nodeChanged(this);
                    break;
                case TABLE:
                    InterNode p = (InterNode)parent;
                    p.rebuild();
                    tree.startEditingAtPath(new TreePath(p.getPath()));
                    break;
            }
            
        }
        
    }
    
    /**
     * Tree node that displays a monitor table.
     * FIXME: Do I still need separate tables for renderer and editor?
     */
    protected class MonitorTableNode extends DisplayNode {
        
        private final MonitorTable model;
        private final JPanel panel;
        
        MonitorTableNode(MonitorTable tableModel) {
            this(tableModel, tableModel.getClass().getName());
        }
        
        MonitorTableNode(MonitorTable tableModel, String name) {
            
            super(name);
            model = tableModel;
            
            if (formatter != null) {
                model.setFormat(formatter);
            }

            JComponent table = model.getTable();
            panel = new JPanel(new BorderLayout());
            panel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0), BorderFactory.createLineBorder(Color.BLACK)));
            if (model.showHeader()) {
                panel.add(((JTable)table).getTableHeader(), BorderLayout.NORTH);
            }
            panel.add(table, BorderLayout.CENTER);
        }
        
        @Override
        JComponent getComponent() {
            return panel;
        }
        
        @Override
        void clear() {
            model.destroy();
            super.clear();
        }

        @Override
        Serializable save() {
            return model.save();
        }
        
    }

    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * JavaBean that contains information required to re-create this view in its current state.
     */
    static public class Descriptor extends AbstractMonitorView.Descriptor {
        
        private TreeMap<String, Serializable> nodes;

        public TreeMap<String, Serializable> getNodes() {
            return nodes;
        }

        public void setNodes(TreeMap<String, Serializable> nodes) {
            this.nodes = nodes;
        }

        @Override
        public Descriptor clone() {
            Descriptor desc = (Descriptor) super.clone();
            if (desc.nodes != null) {
                desc.nodes = new TreeMap<>(desc.nodes); // FIXME
            }
            return desc;
        }
        
    }
    
    /**
     * Returns a descriptor that contains information required to re-create this view in its current state.
     * @return View descriptor, or {@code null} is this view does not support saving.
     */
    @Override
    public Descriptor save() {
        getDescriptor().setNodes(null);
        updateDescriptor(root);
        return descriptor.clone();
    }
    
    /**
     * Restores this view to the state described by the provided descriptor, to the extent possible.
     * @param descriptor View descriptor.
     */
    @Override
    public void restore(Persistable.Descriptor descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = ((Descriptor) descriptor).clone();
            root.clear();
            root.build(data.entrySet());
            treeModel.nodeStructureChanged(root);
            root.restoreExpansion();
        }
    }

    /**
     * Returns a reference to the descriptor maintained by this view.
     * @return Descriptor maintained by this view.
     */
    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }
    
    private void updateDescriptor(Node node) {
        
        TreeMap<String, Serializable> nodes = getDescriptor().getNodes();
        if (nodes == null) {
            nodes = new TreeMap<>();
            getDescriptor().setNodes(nodes);
        }
        
        String path = node.getPathString();
        Serializable desc = node.save();
        if (desc == null) {
            nodes.remove(path);
        } else {
            nodes.put(path, desc);
        }
        
        if (node != root) path += "/";
        Enumeration<TreeNode> en = node.children();
        while (en.hasMoreElements()) {
            updateDescriptor(((Node)en.nextElement()), path, nodes);
        }
    }
    
    private void updateDescriptor(Node node, String path, TreeMap<String, Serializable> map) {
            Serializable desc = node.save();
            path += node.toString();
            if (desc == null) {
                map.remove(path);
            } else {
                map.put(path, desc);
            }
            path += "/";
            Enumeration<TreeNode> en = node.children();
            while (en.hasMoreElements()) {
                updateDescriptor(((Node)en.nextElement()), path, map);
            }
    }
    
}
