package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.BorderLayout;
import java.util.*;
import javax.swing.Box;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.base.Const;

/**
 * {@link MonitorView} that divides channels into groups and displays one group at a time, using combo boxes to select the group.
 *
 * @author onoprien
 */
public class ComboGroupView extends GroupView {

// -- Fields : -----------------------------------------------------------------
    
    static public final String CREATOR_PATH = "Built-In/Grouped/Combo";
    
    private JPanel root;
    private Box comboPanel;
    private ArrayList<JComboBox<String>> combos;
    private JScrollPane viewsPane;
    private boolean adjusting;

// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = MonitorView.CATEGORY,
            name = "Group Combo",
            path = CREATOR_PATH,
            description = "Monitoring data view that divides channels into groups, and displays one group at a time. Combo boxes are used to select the group to display, and tree views are used for each group.")
    public ComboGroupView(
            @Par(def = "1", desc = "Depth of group hierarchy (number of combo boxes, maximum 6). If 0, the display path is expected in \"group/.../name//rest/of/.../path\" format, and the depth is determined by the number of slash-separated segments in \"group/.../name\". If positive N, the fist N segments of the display path ware used as a group name.") int depth,
            @Par(def = Par.NULL, desc = "List of labels for combo boxes. If not specified, no labels are used.") List<String> labels
    )
    {
        super();
        descriptor = new Descriptor();
        descriptor.setName("Group Combo");
        if (depth < 0 || depth > 6) {
            throw new IllegalArgumentException("Illegal group tree depth, must be between 0 and 6.");
        }
        descriptor.setDepth(depth);
        if (labels != null && !labels.isEmpty()) {
            ((Descriptor)descriptor).setLabels(labels.toArray(new String[0]));
        }
    }

    @Override
    public JComponent getPanel() {
        if (root == null) {
            root = new JPanel(new BorderLayout());
            comboPanel = Box.createHorizontalBox();
            root.add(comboPanel, BorderLayout.NORTH);
            comboPanel.add(Box.createRigidArea(Const.HDIM));
            combos = new ArrayList<>(depth);
            depthChanged();
            comboPanel.add(Box.createHorizontalGlue());
            viewsPane = new JScrollPane();
            root.add(viewsPane, BorderLayout.CENTER);
            selectGroup(null);
        }
        return root;
    }

    
// -- Overriding GroupView : ---------------------------------------------------

    @Override
    protected void selectGroup(String group) {
        if (root == null) return;
        super.selectGroup(group);
    }
    
    
// -- Implementing GroupView : -------------------------------------------------
    
    @Override    
    protected void groupsAdded(List<String> groups) {
        if (root == null) return;
        depthChanged();
        selectGroup(currentGroup);
    }
    
    @Override
    protected void groupsRemoved(List<String> groups) {
        if (root == null) return;
        depthChanged();
        selectGroup(currentGroup);
    }

    /** On group tree depth change, adjust the number of combo boxes to match. */
    private void depthChanged() {
        if (root != null) {
            int oldDepth = combos.size();
            if (depth > oldDepth) {
                for (int level = oldDepth; level < depth; level++) {
                    JComboBox<String> cb = new JComboBox<>();
                    cb.setEnabled(false);
                    int lev = level;
                    cb.addActionListener(e -> {
                        comboBoxAction(lev, ((String)cb.getSelectedItem()));
                    });
                    combos.add(cb);
                    int index = 2*level;
                    String[] labels = getDescriptor().getLabels();
                    JLabel label = new JLabel(labels != null && labels.length > level ? " "+ labels[level] +" " : "  "); 
                    comboPanel.add(label, ++index);
                    comboPanel.add(cb, ++index);
                }
            } else if (depth < oldDepth) {
                for (int level = oldDepth; level > depth; ) {
                    combos.remove(--level);
                    int index = 2*level + 1;
                    comboPanel.remove(index);
                    comboPanel.remove(index);
                }
            }
        }
    }

    /** On group selection, re-populate all combo boxes. */
    @Override
    protected void groupSelected() {
        adjusting = true;
        if (Objects.equals(currentGroup, "")) {
            combos.forEach(cb -> {
                cb.removeAllItems();
                cb.setSelectedItem(null);
                cb.setEnabled(false);
            });
            viewsPane.setViewportView(new MonitorDisplay.EMPTY());
        } else {
            String[] gg = getPath(currentGroup);
            String g = null;
            for (int i=0; i<gg.length; i++) {
                combos.get(i).setModel(new DefaultComboBoxModel<>(getSubgroups(g)));
                combos.get(i).setSelectedItem(gg[i]);
                combos.get(i).setEnabled(true);
                g = g == null ? gg[i] : g +"/"+ gg[i];
            }
            if (gg.length < depth) {
                String[] sg = getSubgroups(g);
                if (sg.length == 0) {
                    combos.get(gg.length).removeAllItems();
                    combos.get(gg.length).setSelectedItem(null);
                    combos.get(gg.length).setEnabled(false);
                } else {
                    combos.get(gg.length).setModel(new DefaultComboBoxModel<>(sg));
                    combos.get(gg.length).setSelectedItem("");
                    combos.get(gg.length).setEnabled(true);
                }
            }
            for (int i=gg.length+1; i<depth; i++) {
                combos.get(i).removeAllItems();
                combos.get(i).setSelectedItem(null);
                combos.get(i).setEnabled(false);
            }
            AbstractMonitorView view = getView(currentGroup);
            JComponent panel = view.getPanel();
            viewsPane.setViewportView(panel);
        }
        adjusting = false;
    }

    
// -- Local methods : ----------------------------------------------------------
    
    private void comboBoxAction(int level, String groupSegment) {
        if (adjusting) return;
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < level; i++) {
            sb.append((String) (combos.get(i).getSelectedItem())).append("/");
        }
        sb.append(groupSegment);
        selectGroup(sb.toString());
    }

// -- Saving/Restoring : -------------------------------------------------------
    
    @Override
    public Descriptor getDescriptor() {
        return (Descriptor) super.getDescriptor();
    }
        
    static public class Descriptor extends GroupView.Descriptor {

        private String[] labels;

        public String[] getLabels() {
            return labels;
        }

        public void setLabels(String[] labels) {
            this.labels = labels;
        }

        @Override
        public Descriptor clone() {
            return (Descriptor) super.clone();
        }
        
    }
    
}
