package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.io.Serializable;
import java.util.*;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTextPane;
import javax.swing.KeyStroke;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultCaret;
import javax.swing.text.Position;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyleContext;
import javax.swing.text.StyledDocument;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.gconsole.agent.command.CommandCode;
import org.lsst.ccs.gconsole.agent.command.CommandHandle;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.services.persist.Savable;

/**
 * Component that displays commands output.
 *
 * @author onoprien
 */
public final class OutputPanel extends JTextPane implements Savable, HasPopupItems {

// -- Fields : -----------------------------------------------------------------
        
    private final LsstCommandBrowserPlugin plugin = Console.getConsole().getSingleton(LsstCommandBrowserPlugin.class);
    
    private final LinkedList<Record> records = new LinkedList<>();
    private final int MAX_RECORDS = 100;
    long recordID = 0L;
    
    private final Style attBase, attPlane, attCom, attTime, attRed;
    private final Action actClear, actFontLarger, actFontSmaller;
    private final CommandHandle util = new CommandHandle() {}; // empty command handle for access to its utility methods
    
    private Descriptor descriptor = new Descriptor();

// -- Life cycle : -------------------------------------------------------------
    
    public OutputPanel() {
        
        // JTextPane settings
        
        setEditable(false);
        setCaretPosition(0);
        try {
        } catch (ClassCastException x) {
            ((DefaultCaret)getCaret()).setUpdatePolicy(DefaultCaret.ALWAYS_UPDATE);
        }
        
        // Styles

        StyledDocument doc = getStyledDocument();
        Style def = StyleContext.getDefaultStyleContext().getStyle(StyleContext.DEFAULT_STYLE);

        attBase = doc.addStyle("attBase", def);
        StyleConstants.setFontSize(attBase, 15);

        attPlane = doc.addStyle("attPlane", attBase);
        StyleConstants.setFontFamily(attPlane, Font.MONOSPACED);
        
        attCom = doc.addStyle("attCom", attBase);
        StyleConstants.setFontFamily(attCom, Font.MONOSPACED);
        StyleConstants.setBold(attCom, true);
        StyleConstants.setForeground(attCom, new Color(0,0,100));
        
        attTime = doc.addStyle("attTime", attBase);
        StyleConstants.setForeground(attTime, new Color(110,110,110));

        attRed = doc.addStyle("attRed", attBase);
        StyleConstants.setForeground(attRed, Color.red);
        
        // Actions
        
        KeyStroke ks;
        String name;
        
        name = "Clear";
        ks = KeyStroke.getKeyStroke(KeyEvent.VK_C, KeyEvent.ALT_DOWN_MASK);
        actClear = new AbstractAction(name) {
            @Override
            public void actionPerformed(ActionEvent e) {
                StyledDocument doc = OutputPanel.this.getStyledDocument();
                try {
                    doc.remove(0, doc.getLength());
                } catch (BadLocationException x) {
                }
                records.clear();
            }
        };
        actClear.putValue(Action.ACCELERATOR_KEY, ks);
        getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(ks, name);
        getActionMap().put(name, actClear);
        
        name = "Larger";
        ks = KeyStroke.getKeyStroke(KeyEvent.VK_EQUALS, KeyEvent.ALT_DOWN_MASK);
        actFontLarger = new AbstractAction(name) {
            @Override
            public void actionPerformed(ActionEvent e) {
                int fs = StyleConstants.getFontSize(attBase);
                setFontSize(fs + 1);                
            }
        };
        actFontLarger.putValue(Action.ACCELERATOR_KEY, ks);
        getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(ks, name);
        getActionMap().put(name, actFontLarger);

        name = "Smaller";
        ks = KeyStroke.getKeyStroke(KeyEvent.VK_MINUS, KeyEvent.ALT_DOWN_MASK);
        actFontSmaller = new AbstractAction(name) {
            @Override
            public void actionPerformed(ActionEvent e) {
                int fs = StyleConstants.getFontSize(attBase);
                if (fs > 6) {
                    setFontSize(fs - 1);
                }
            }
        };
        actFontSmaller.putValue(Action.ACCELERATOR_KEY, ks);
        getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(ks, name);
        getActionMap().put(name, actFontSmaller);
    }
    
// -- Operations : -------------------------------------------------------------
    
    public long addCommand(String command) {
        Record record = new Record(command);
        record.add();
        records.add(record);
        if (records.size() > MAX_RECORDS) {
            Record r = records.pollFirst();
            r.remove();
        }
        setCaretPosition(getStyledDocument().getLength());
        return record.id;
    }
    
    public void addAck(long id) {
        Iterator<Record> it = records.descendingIterator();
        while (it.hasNext()) {
            Record r = it.next();
            if (r.id == id) {
                r.addAck();
                break;
            }
        }
        setCaretPosition(getStyledDocument().getLength());
    }
    
    public void addResponse(long id, CommandCode code, Object response) {
        Iterator<Record> it = records.descendingIterator();
        while (it.hasNext()) {
            Record r = it.next();
            if (r.id == id) {
                r.addResponse(code, response);
                break;
            }
        }
        setCaretPosition(getStyledDocument().getLength());
    }
    
    
// -- Pop-up menu : ------------------------------------------------------------

    @Override
    public JPopupMenu modifyPopupMenu(JPopupMenu menu, Component cmpnt, Point point) {
        menu.add(new JMenuItem(actClear));
        JMenu fontMenu = new JMenu("Font size");
        menu.add(fontMenu);
        fontMenu.add(new JMenuItem(actFontLarger));
        fontMenu.add(new JMenuItem(actFontSmaller));
        return menu;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void setFontSize(int size) {
        setButtonSize(this, size);
        StyleConstants.setFontSize(attBase, size);
        descriptor.setFontSize(size);
    }
    
    private void setButtonSize(Component comp, int size) {
        if (comp instanceof JButton) {
            comp.setPreferredSize(new Dimension(comp.getPreferredSize().width, size));
        } else if (comp instanceof Container) {
            Component[] cc = ((Container)comp).getComponents();
            for (Component c : cc) {
                setButtonSize(c, size);
            }
        }
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    public class Record {
        
        final long id;
        final String command;
        CommandCode code;
        Object response;
        Position posFirst, posLast;
        String timing;
        long sendTime;
        
        Record(String command) {
            id = recordID++;
            this.command = command;
        }
        
        void add() {
            StyledDocument doc = OutputPanel.this.getStyledDocument();
            int i = doc.getLength();
            try {
                if (plugin.isDisplayTiming()) {
                    sendTime = System.currentTimeMillis();
                    timing = " [ack: ... / total: ... ] ms\n";
                    doc.insertString(i, "> " + command + "  ", attCom);
                    doc.insertString(doc.getLength(), timing, attTime);
                } else {
                    sendTime = 0L;
                    timing = " ...\n";
                    doc.insertString(i, "> " + command + "  " + timing, attCom);
                }
                posFirst = doc.createPosition(i);
                posLast = doc.createPosition(doc.getLength()-1);
            } catch (BadLocationException x) {
            }
        }
        
        void remove() {
            try {
                int start = posFirst.getOffset();
                int length = posLast.getOffset() - start + 1;
                OutputPanel.this.getStyledDocument().remove(start, length);
            } catch (BadLocationException x) {
            }
        }
        
        void collaps() {
            throw new UnsupportedOperationException("Not yet implemented.");
        }
        
        void expand() {
            throw new UnsupportedOperationException("Not yet implemented.");
        }
        
        void addAck() {
            if (sendTime > 0L) {
                StyledDocument doc = OutputPanel.this.getStyledDocument();
                try {
                    int i = posLast.getOffset() - timing.length();
                    doc.remove(i, timing.length());
                    long time = System.currentTimeMillis() - sendTime;
                    timing = timing.replace("ack: ...", "ack: " + Long.toString(time));
                    doc.insertString(posLast.getOffset(), timing, attTime);
                } catch (BadLocationException x) { // never
                }
            }
        }
        
        void addResponse(CommandCode code, Object response) {
            this.code = code;
            this.response = response;
            StyledDocument doc = OutputPanel.this.getStyledDocument();
            try {
                int i = posLast.getOffset() - timing.length();
                doc.remove(i, timing.length());
                if (sendTime > 0L) {
                    long time = System.currentTimeMillis() - sendTime;
                    timing = timing.replace("... ] ms\n", Long.toString(time) +" ] ms");
                    doc.insertString(posLast.getOffset(), timing, attTime);
                }
                showResponse();
            } catch (BadLocationException x) { // never
            }
        }
        
        private void showResponse() throws BadLocationException {
            StyledDocument doc = OutputPanel.this.getStyledDocument();
            int i = posLast.getOffset();
            doc.insertString(i++, "\n", attCom);
            switch (code) {
                case SUCCESS:
                    if (response == null) {
                        doc.insertString(i, "null", attPlane);
                    } else {
                        Class<?> respClass = response.getClass();
                        if (respClass.isArray()) {
                            try {
                                response = Arrays.deepToString((Object[]) response);
                            } catch (ClassCastException x) {
                                if (respClass == byte[].class) {
                                    response = Arrays.toString((byte[]) response);
                                } else if (respClass == short[].class) {
                                    response = Arrays.toString((short[]) response);
                                } else if (respClass == int[].class) {
                                    response = Arrays.toString((int[]) response);
                                } else if (respClass == long[].class) {
                                    response = Arrays.toString((long[]) response);
                                } else if (respClass == char[].class) {
                                    response = Arrays.toString((char[]) response);
                                } else if (respClass == float[].class) {
                                    response = Arrays.toString((float[]) response);
                                } else if (respClass == double[].class) {
                                    response = Arrays.toString((double[]) response);
                                } else if (respClass == boolean[].class) {
                                    response = Arrays.toString((boolean[]) response);
                                }
                            }
                        } else if (response instanceof List<?> && response.toString().length() > 40) {
                            StringBuilder sb = new StringBuilder("[\n    ");
                            List<String> ss = ((List<?>)response).stream().map(o -> Objects.toString(o)).collect(Collectors.toList());
                            sb.append(String.join(",\n    ", ss));
                            sb.append("\n]");
                            response = sb.toString();
                        } else if (response instanceof Map<?,?> && response.toString().length() > 40) {
                            StringBuilder sb = new StringBuilder("{\n    ");
                            List<String> ss = ((Map<?,?>)response).entrySet().stream().map(e -> Objects.toString(e.getKey()) +"="+ Objects.toString(e.getValue())).collect(Collectors.toList());
                            sb.append(String.join(",\n    ", ss));
                            sb.append("\n}");
                            response = sb.toString();
                        } else if (response.equals("")) {
                            response = "\"\"";
                        }
                        doc.insertString(i, response.toString(), attPlane);
                    }
                    break;
                case SEND_FAIL:
                    if (response instanceof Exception) {
                        Exception x = (Exception) response;
                        String message = util.shorten(x.getMessage()) +" ";
                        doc.insertString(i, message, attRed);
                        insertButton(i + message.length(), x.getMessage(), x);
                    } else {
                        doc.insertString(i, "Could not send the command: "+ response, attRed);
                    }
                    break;
                case NACK:
                    StringBuilder sb = new StringBuilder("Command rejected");
                    if (response instanceof CommandNack) {
                        CommandNack nack = (CommandNack) response;
                        String message = util.shorten(nack.getReason());
                        if (message != null && !message.isEmpty()) {
                            sb.append(": ").append(message);
                        }
                    }
                    doc.insertString(i, sb.toString(), attRed);
                    break;
                case TIMEOUT:
                    doc.insertString(i, util.shorten(((TimeoutException)response).getMessage()), attRed);
                    break;
                case CANCEL:
                    doc.insertString(i, "Cancelled.", attRed);
                    break;
                case EXEC_FAIL:
                    if (response instanceof Exception) {
                        Exception x = (Exception) response;
                        String message = util.shorten(x.getMessage()) +" ";
                        doc.insertString(i, message, attRed);
                        insertButton(i + message.length(), x.getMessage(), x);
                    } else {
                        doc.insertString(i, "Command execution failed.", attRed);
                    }
                    break;
            }
        }
        
        private void insertButton(int offset, String message, Exception exception) throws BadLocationException {
            SimpleAttributeSet attButton = new SimpleAttributeSet(attBase);
            StyleConstants.setAlignment(attButton, StyleConstants.ALIGN_CENTER);
            JButton button = new JButton(" ");
            button.setToolTipText("More information ...");
            button.setPreferredSize(new Dimension(button.getPreferredSize().width, StyleConstants.getFontSize(attBase)));
            button.setAlignmentY(.8f);
            button.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
            button.setMargin(new Insets(0, 0, 0, 0));
            button.addActionListener(e -> {
                StringBuilder sb = new StringBuilder("Command:\n");
                sb.append(command).append("\n");
                sb.append("failed during execution.\n\n");
                if (message != null) {
                    sb.append(message);
                }
                Console.getConsole().error(sb.toString(), exception);
            });
            StyleConstants.setComponent(attButton, button);
            StyledDocument doc = OutputPanel.this.getStyledDocument();
            doc.insertString(offset, " ", attButton);
        }
    }

    
// -- Saving/restoring : -------------------------------------------------------    
    
    @Override
    public void restore(Serializable descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
        } else if (descriptor == null) {
            this.descriptor = new Descriptor();
        }
        setFontSize(this.descriptor.getFontSize());
    }

    @Override
    public Descriptor save() {
        return descriptor.clone();
    }
    
    static public class Descriptor implements Serializable, Cloneable {

        private int fontSize = 15;

        public int getFontSize() {
            return fontSize;
        }

        public void setFontSize(int fontSize) {
            this.fontSize = fontSize;
        }
        
        @Override
        public Descriptor clone() {
            try {
                return (Descriptor) super.clone();
            } catch (CloneNotSupportedException x) {
                throw new RuntimeException(); // never
            }
        }
        
    }

}
