package org.lsst.ccs.gconsole.jas3;

import java.awt.Dimension;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTextField;
import org.freehep.jas.services.PreferencesTopic;
import org.lsst.ccs.gconsole.base.Console;

/**
 *
 * @author onoprien
 */
public class PreferencePage  implements PreferencesTopic {

// -- Fields : -----------------------------------------------------------------
    
    private final Console console;
    private final String[] path;
    private final ArrayList<Setter> setters = new ArrayList<>(1);

// -- Life cycle : -------------------------------------------------------------
    
    public PreferencePage(Console console, String[] path) {
        this.console = console;
        this.path = path;
    }
    
    synchronized public void add(String group, String format) {
        setters.add(new Setter(group,format));
        setters.trimToSize();
    }

// -- Implementing PreferencesTopic --------------------------------------------

    @Override
    public String[] path() {
        return path;
    }

    @Override
    synchronized public JComponent component() {
        try {
            return new GUI();
        } catch (IllegalArgumentException x) {
            return new JLabel("  "+ x.getMessage() +"  ");
        }
    }

    @Override
    public boolean apply(JComponent panel) {
        try {
            return ((GUI) panel).apply();
        } catch (ClassCastException x) {
            return true;
        }
    }


// -- Local classes : ----------------------------------------------------------
    
    private class GUI extends Box {
    
        private final int HSPACE = 10;
        private final int VSPACE = 5;
        private final Pattern ptrn = Pattern.compile("\\$\\{[^}]+\\}");
        
        private final HashMap<String,Widget> data = new HashMap<>();
        
        GUI() {
            super(BoxLayout.Y_AXIS);
            setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
            
            HashMap<String,Box> groups = new HashMap<>();
            
            for (Setter setter : setters) {
                
                Box line = Box.createHorizontalBox();
                if (setter.group == null) {
                    add(line);
                    add(Box.createRigidArea(new Dimension(0,VSPACE)));
                } else {
                    Box groupBox = groups.get(setter.group);
                    if (groupBox == null) {
                        groupBox = Box.createVerticalBox();
                        groupBox.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createTitledBorder(setter.group), BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
                        groups.put(setter.group, groupBox);
                        add(Box.createRigidArea(new Dimension(0,VSPACE)));
                        add(groupBox);
                        add(Box.createRigidArea(new Dimension(0,VSPACE)));
                    }
                    groupBox.add(line);
                    groupBox.add(Box.createRigidArea(new Dimension(0,VSPACE)));
                }
                
                Matcher mat = ptrn.matcher(setter.format);
                int i=0;
                while (mat.find()) {
                    String s = setter.format.substring(i, mat.start());
                    if (!s.isEmpty()) {
                        line.add(new JLabel(s));
                        line.add(Box.createRigidArea(new Dimension(HSPACE,0)));
                    }
                    i = mat.end();
                    String qualifier = null;
                    String key = setter.format.substring(mat.start()+2, mat.end()-1);
                    int k = key.indexOf("#");
                    if (k != -1) {
                        qualifier = key.substring(k+1);
                        key = key.substring(0, k);
                    }
                    Widget pd = createWidget(key, qualifier);
                    line.add(pd.getComponent());
                    line.add(Box.createRigidArea(new Dimension(HSPACE,0)));
                    Widget old = data.put(key, pd);
                    if (old != null) throw new IllegalArgumentException("Multiple setters for property: "+ key);
                }
                String s = setter.format.substring(i, setter.format.length());
                if (!s.isEmpty()) {
                    line.add(new JLabel(s));
                    line.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
                }

                line.add(Box.createHorizontalGlue());
            }
            
            add(Box.createVerticalGlue());
        }
        
        boolean apply() {
            try {
                Map<String,Object> newValues = data.entrySet().stream().collect(Collectors.toMap(e->e.getKey(), e->e.getValue().getValue()));
                data.values().forEach(widget -> widget.save());
                console.setProperties(newValues);
                return true;
            } catch (IllegalArgumentException x) {
                return false;
            }
        }
        
    }
    
    private class Setter {
        Setter(String group, String format) {
            this.group = group;
            this.format = format;
        }
        String group;
        String format;
    }
    
    
// -- Widgets for setting specific types of properties : -----------------------
    
    private Widget createWidget(String key, String qualifier) {
        
        Object value = console.getProperty(key);
        if (value == null) throw new IllegalArgumentException("No such property: " + key);
        Class<?> c = value.getClass();
        HashMap<String, String> tokens = new HashMap<>();
        if (qualifier != null && !qualifier.isEmpty()) {
            String[] ss = qualifier.split("\\#");
            for (String s : ss) {
                int i = s.indexOf("=");
                if (i == -1) {
                    tokens.put(s, null);
                } else {
                    tokens.put(s.substring(0, i), s.substring(i + 1));
                }
            }
        }

        if (Boolean.class.isAssignableFrom(c)) {
            return new BooleanWidget(value);
        } else if (Integer.class.isAssignableFrom(c)) {
            return new IntegerWidget(value);
        } else if (Long.class.isAssignableFrom(c)) {
            return new LongWidget(value);
        } else if (Double.class.isAssignableFrom(c)) {
            return new DoubleWidget(value);
        } else if (String.class.isAssignableFrom(c)) {
            try {
                int n = Integer.parseInt(tokens.get("history"));
                return new StringWithHistoryWidget(key, (String)value, n);
            } catch (NullPointerException | IllegalArgumentException | ClassCastException x) {
                return new StringWidget(value);
            }
        } else if (Enum.class.isAssignableFrom(c)) {
            return new EnumWidget(value);
        } else {
            throw new IllegalArgumentException("Unsupported preference type: " + c.getName());
        }
        
    }
    
    static abstract class Widget {
        abstract Object getValue();
        abstract JComponent getComponent();
        void save() {}
    }
    
    private class BooleanWidget extends Widget {
        
        final JCheckBox checkBox;
        
        BooleanWidget(Object value) {
            checkBox = new JCheckBox();
            checkBox.setSelected((Boolean)value);
        }

        @Override
        Object getValue() {
            return checkBox.isSelected();
        }

        @Override
        JComponent getComponent() {
            return checkBox;
        }
    }
    
    private class StringWidget extends Widget {
        
        final JTextField textField;
        
        StringWidget(Object value) {
            textField = new JTextField(value.toString());
            textField.setMaximumSize(new Dimension(Integer.MAX_VALUE, textField.getPreferredSize().height));
        }

        @Override
        String getValue() {
            return textField.getText();
        }

        @Override
        JComponent getComponent() {
            return textField;
        }
    }
    
    static private class IntegerWidget extends Widget {
        
        final JTextField textField;
        
        IntegerWidget(Object value) {
            textField = new JTextField(value.toString(), 11);
            textField.setMaximumSize(textField.getPreferredSize());
        }

        @Override
        Object getValue() {
            return Integer.valueOf(textField.getText());
        }

        @Override
        JComponent getComponent() {
            return textField;
        }
    }
    
    private class LongWidget extends Widget {
        
        final JTextField textField;
        
        LongWidget(Object value) {
            textField = new JTextField(value.toString(), 20);
            textField.setMaximumSize(textField.getPreferredSize());
        }

        @Override
        Object getValue() {
            return Long.valueOf(textField.getText());
        }

        @Override
        JComponent getComponent() {
            return textField;
        }
    }
    
    private class DoubleWidget extends Widget {
        
        final JTextField textField;
        
        DoubleWidget(Object value) {
            textField = new JTextField(value.toString(), 12);
            textField.setMaximumSize(textField.getPreferredSize());
        }

        @Override
        Object getValue() {
            return Double.valueOf(textField.getText());
        }

        @Override
        JComponent getComponent() {
            return textField;
        }
    }
    
    private class EnumWidget extends Widget {
        
        final JComboBox combo;
        
        EnumWidget(Object value) {
            Enum enumValue = (Enum)value;
            enumValue.getDeclaringClass().getEnumConstants();
            
            combo = new JComboBox(enumValue.getDeclaringClass().getEnumConstants());
            combo.setSelectedItem(enumValue);
            combo.setMaximumSize(combo.getPreferredSize());
        }

        @Override
        Object getValue() {
            return combo.getSelectedItem();
        }

        @Override
        JComponent getComponent() {
            return combo;
        }
    }
    
    private class StringWithHistoryWidget extends Widget {
        
        final String historyKey;
        final int keep;
        final JComboBox combo;
        final ArrayDeque history;
        
        StringWithHistoryWidget(String key, String value, int keep) {
            historyKey = key + "$history$";
            this.keep = keep;
            console.addProperty(historyKey, new String[0]);
            String[] hh = (String[]) console.getProperty(historyKey);
            if (hh == null) hh = new String[0];
            history = new ArrayDeque<>(hh.length+2);
            for (String h : hh) {
                if (!value.equals(h)) {
                    history.add(h);
                }
            }
            history.addFirst(value);
            combo = new JComboBox(history.toArray(new String[0]));
            combo.setMaximumSize(combo.getPreferredSize());
            combo.setEditable(true);
        }

        @Override
        String getValue() {
            return (String) combo.getSelectedItem();
        }

        @Override
        JComponent getComponent() {
            return combo;
        }

        @Override
        void save() {
            String value = getValue();
            if (!history.contains(value)) {
                history.addFirst(value);
            }
            while (history.size() > keep) {
                history.removeLast();
            }
            console.setProperty(historyKey, history.toArray(new String[0]));
        }
   }

    
// -- Testing : ----------------------------------------------------------------

    public static void main(String... args) {
        Pattern p = Pattern.compile("\\$\\{[^{}]+\\}");
        Matcher m = p.matcher("pre1${par1}pre2${par2}post");
        while (m.find()) {
            System.out.println("start "+ m.start() +" end "+ m.end() +" group "+ m.group() +" count "+ m.groupCount());
        }
        System.out.println(m.appendTail(new StringBuffer()));
    }

}
