package org.lsst.ccs.gconsole.base;

import java.awt.AWTEvent;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.AWTEventListener;
import java.util.concurrent.TimeUnit;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;
import org.freehep.jas.services.PreferencesTopic;
import org.lsst.ccs.gconsole.util.ThreadUtil;
import org.lsst.ccs.gconsole.util.swing.BoxedComboBox;
import org.lsst.ccs.gconsole.util.swing.BoxedSpinner;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;

/**
 * Auxiliary class that handles killing idle consoles.
 *
 * @author onoprien
 */
class SelfDestructor implements AWTEventListener {

// -- Fields : -----------------------------------------------------------------
    
    private final String PROP_BEFORE = "org.lsst.ccs.gui.self-destruct.before";
    private final String PROP_AFTER = "org.lsst.ccs.gui.self-destruct.after";
    
    private int beforeCountdown, afterCountdown; // seconds
    private long lastTouch;
    private PeriodicTask task;
    private JLabel timeLeftLabel;
    

// -- Life cycle : -------------------------------------------------------------
    
    SelfDestructor() {
        Console c = Console.getConsole();
        c.addProperty(PROP_BEFORE, 49 * 3600);
        c.addProperty(PROP_AFTER, 25 * 3600);
        c.getConsoleLookup().add(new Pref());
    }
    
    void restart() {
        ThreadUtil.invokeLater(() -> {
            Console c = Console.getConsole();
            beforeCountdown = (int) c.getProperty(PROP_BEFORE);
            afterCountdown = (int) c.getProperty(PROP_AFTER);
            Toolkit.getDefaultToolkit().removeAWTEventListener(this);
            if (task != null) {
                task.cancel(true);
                task = null;
            }
            timeLeftLabel = null;
            if (beforeCountdown > 0) {
                Toolkit.getDefaultToolkit().addAWTEventListener(this, AWTEvent.KEY_EVENT_MASK + AWTEvent.MOUSE_EVENT_MASK /* + AWTEvent.MOUSE_MOTION_EVENT_MASK*/);
                lastTouch = System.currentTimeMillis();
                task = Console.getConsole().getScheduler().scheduleAtFixedRate(this::check, 1L, 1L, TimeUnit.MINUTES);
            }
        });
    }
    
    
// -- Listening to user activity : ---------------------------------------------
    
    @Override
    public void eventDispatched(AWTEvent event) {
        if (timeLeftLabel == null) lastTouch = System.currentTimeMillis();
    }


// -- Local methods : ----------------------------------------------------------
    
    private void killConsole() {
        if (task != null) {
            task.cancel(true);
        }
        try {
            Console.getConsole().shutdownAgent();
        } catch (Exception x) {
        }
    }
    
    private void startCountdown() {
        if (timeLeftLabel != null) return;
        if (task != null) {
            task.cancel(true);
            task = null;
        }
        boolean kill = true;
        if (afterCountdown > 0) {
            lastTouch = System.currentTimeMillis();
            timeLeftLabel = new JLabel();
            updateLabel();
            task = Console.getConsole().getScheduler().scheduleAtFixedRate(
                    () -> SwingUtilities.invokeLater(() -> updateLabel()),
                    1L, 1L, TimeUnit.SECONDS);
            Object[] options = {"Keep alive", "Shut down"};
            try {
                int n = JOptionPane.showOptionDialog(Console.getConsole().getWindow(), timeLeftLabel,
                      "Idle CCS console", JOptionPane.YES_NO_OPTION, JOptionPane.PLAIN_MESSAGE, null, options, options[0]);
                if (n == 0) kill = false;
            } catch (RuntimeException x) {
            }
            timeLeftLabel = null;
        }
        if (kill) {
            killConsole();
        } else {
            restart();
        }
    }
    
    private void check() {
        SwingUtilities.invokeLater(() -> {
            if ((System.currentTimeMillis() - lastTouch) > (beforeCountdown * 1000L)) {
                startCountdown();
            }
        });
    }
    
    private void updateLabel() {
        if (timeLeftLabel == null) return;
        int remainSeconds = afterCountdown - (int) ((System.currentTimeMillis() - lastTouch)/1000L);
        if (remainSeconds > 0) {
            StringBuilder sb = new StringBuilder("<html><center>This console will be shut down due to inactivity in<p>");
            int hours = remainSeconds / 3600;
            if (hours > 0) sb.append(hours).append(" hours ");
            remainSeconds %= 3600;
            int minutes = remainSeconds / 60;
            if (minutes > 0) sb.append(minutes).append(" minutes ");
            remainSeconds %= 60;
            sb.append(remainSeconds).append(" seconds.");
            timeLeftLabel.setText(sb.toString());
        } else if (remainSeconds > -2) {
            timeLeftLabel.setText("Shutting down...");
        } else {
            killConsole();
        }
    }
    
    
// -- Preferences : ------------------------------------------------------------
    
    private class Pref implements PreferencesTopic {

        @Override
        public String[] path() {
            return new String[] {"LSST"};
        }

        @Override
        public JComponent component() {
            PrefGUI panel = new PrefGUI();
            Console c = Console.getConsole();
            int beforeCountdown = (int) c.getProperty(PROP_BEFORE);
            int afterCountdown = (int) c.getProperty(PROP_AFTER);
            panel.set(beforeCountdown, afterCountdown);
            return panel;
        }

        @Override
        public boolean apply(JComponent jc) {
            try {
                PrefGUI panel = (PrefGUI) jc;
                int[] seconds = panel.get();
                Console c = Console.getConsole();
                c.setProperty(PROP_BEFORE, seconds[0]);
                c.setProperty(PROP_AFTER, seconds[1]);
                restart();
                return true;
            } catch (RuntimeException x) {
                return false;
            }
        }
        
    }
    
    static private final class PrefGUI extends JPanel {
        
        private final String DAYS = "days";
        private final String HOURS = "hours";
        private final String MINUTES = "minutes";
        
        private JCheckBox killBox, warnBox;
        private JSpinner killSpinner, warnSpinner;
        private JComboBox<String> killCombo, warnCombo;
        private JLabel warnLabel;
        
        PrefGUI() {
            
            this.setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            setBorder(BorderFactory.createTitledBorder("Idle console shutdown"));
            
            Box line = Box.createHorizontalBox();
            add(line);
            line.add(Box.createRigidArea(Const.HDIM));
            killBox = new JCheckBox("Shut down if idle for ");
            line.add(killBox);
            killBox.addActionListener(e -> enableComponents());
            killSpinner = new BoxedSpinner(new SpinnerNumberModel(1, 1, 100, 1));
            line.add(killSpinner);
            line.add(Box.createRigidArea(Const.HDIM));
            killCombo = new BoxedComboBox<>(new String[] {DAYS, HOURS, MINUTES});
            line.add(killCombo);
            line.add(Box.createRigidArea(Const.HDIM));
            line.add(Box.createHorizontalGlue());
            
            line = Box.createHorizontalBox();
            add(line);
            line.add(Box.createRigidArea(new Dimension(3*Const.HSPACE, 0)));
            warnBox = new JCheckBox("Warn ");
            line.add(warnBox);
            warnBox.addActionListener(e -> {
                boolean warningEnabled = warnBox.isSelected();
                warnSpinner.setEnabled(warningEnabled);
                warnCombo.setEnabled(warningEnabled);
            });
            warnSpinner = new BoxedSpinner(new SpinnerNumberModel(1, 1, 100, 1));
            line.add(warnSpinner);
            line.add(Box.createRigidArea(Const.HDIM));
            warnCombo = new BoxedComboBox<>(new String[] {DAYS, HOURS, MINUTES});
            line.add(warnCombo);
            warnLabel = new JLabel(" before shutdown.");
            line.add(warnLabel);
            line.add(Box.createRigidArea(Const.HDIM));
            line.add(Box.createHorizontalGlue());
            
            add(Box.createVerticalGlue());
        }
        
        void set(int beforeCountdown, int afterCountdown) {
            set(beforeCountdown, killBox, killSpinner, killCombo);
            set(afterCountdown, warnBox, warnSpinner, warnCombo);
            enableComponents();
        }
        
        int[] get() {
            int beforeCountdown = get(killBox, killSpinner, killCombo);
            int afterCountdown = get(warnBox, warnSpinner, warnCombo);
            return new int[]{beforeCountdown, afterCountdown};
        }
        
        private void enableComponents() {
            if (killBox.isSelected()) {
                killSpinner.setEnabled(true);
                killCombo.setEnabled(true);
                warnBox.setEnabled(true);
                boolean warningEnabled = warnBox.isSelected();
                warnSpinner.setEnabled(warningEnabled);
                warnCombo.setEnabled(warningEnabled);
                warnLabel.setEnabled(true);
            } else {
                killSpinner.setEnabled(false);
                killCombo.setEnabled(false);
                warnBox.setEnabled(false);
                warnSpinner.setEnabled(false);
                warnCombo.setEnabled(false);
                warnLabel.setEnabled(false);
            }
        }
        
        private void set(int seconds, JCheckBox box, JSpinner spinner, JComboBox combo) {
            boolean enabled = seconds > 0;
            box.setSelected(enabled);
            seconds = Math.abs(seconds);
            seconds /= 60;
            if (seconds % 60 == 0) {
                seconds /= 60;
                if (seconds % 24 == 0) {
                    seconds /= 24;
                    combo.setSelectedItem(DAYS);
                } else {
                    combo.setSelectedItem(HOURS);
                }
            } else {
                combo.setSelectedItem(MINUTES);
            }
            spinner.setValue(seconds);
        }
        
        int get(JCheckBox box, JSpinner spinner, JComboBox combo) {
            int seconds = (int) spinner.getValue();
            String unit = (String) combo.getSelectedItem();
            if (DAYS.equals(unit)) {
                seconds *= 3600 * 24;
            } else if (HOURS.equals(unit)) {
                seconds *= 3600;
            } else {
                seconds *= 60;
            }
            if (!box.isSelected()) seconds = -seconds;
            return seconds;
        }
        
    }
        
    
}
