package org.lsst.ccs.gconsole.plugins.trending.timeselection;

import org.lsst.ccs.gconsole.util.swing.DurationSelector;
import java.awt.Dimension;
import java.awt.Window;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import static org.lsst.ccs.gconsole.base.Const.HSPACE;
import static org.lsst.ccs.gconsole.base.Const.VSPACE;
import org.lsst.ccs.gconsole.util.swing.DateTimeSelector;

/**
 * GUI panel for entering a custom time window.
 *
 * @author onoprien
 */
public final class TimeWindowDialog extends JPanel {

// -- Fields : -----------------------------------------------------------------
    
    private final JPanel startPanel, endPanel;
    private final JLabel beforeLabel, afterLabel, startDuraLabel, endDuraLabel;
    private final DateTimeSelector startChooser, endChooser;
    private final DurationSelector startDuraChooser, endDuraChooser;
    private final ButtonGroup startGroup, endGroup;
    private final JRadioButton startFixedButton, startDuraButton, endFixedButton, endDuraButton, endNowButton;
    private final JTextField nameField;
    private final JCheckBox saveCheck;
    
    private TimeWindow range, outRange;
    private final JButton okButton;   
    private ZoneId zone = ZoneId.systemDefault();
    private long[] current;

    
// -- Life cycle : -------------------------------------------------------------
    
    private TimeWindowDialog(TimeWindow seed) {
        
        range = seed;
        okButton = new JButton("  OK  ");
        setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
        
        // Start panel
        
        startPanel = new JPanel();    
        startPanel.setLayout(new BoxLayout(startPanel, BoxLayout.Y_AXIS));
        startPanel.setAlignmentX(LEFT_ALIGNMENT);
        startPanel.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(" START "),
                BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));

        Box box = Box.createHorizontalBox();
        startGroup = new ButtonGroup();
        startFixedButton = new JRadioButton("Specific time: ");
        startGroup.add(startFixedButton);
        box.add(startFixedButton);
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        startChooser = new DateTimeSelector();
        box.add(startChooser);
        startChooser.setToolTipText("Select date and time for the start of the time window");
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        box.add(Box.createHorizontalGlue());
        startPanel.add(box);

        startPanel.add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));

        box = Box.createHorizontalBox();
        add(box);
        startDuraButton = new JRadioButton("");
        startGroup.add(startDuraButton);
        box.add(startDuraButton);
        startDuraChooser = new DurationSelector();
        box.add(startDuraChooser);
        startDuraChooser.setToolTipText("Select time window duration");
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        beforeLabel = new JLabel("before END.");
//        box.add(beforeLabel);
        box.add(startDuraLabel = new JLabel("before END."));
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        box.add(Box.createHorizontalGlue());
        startPanel.add(box);

        startPanel.add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        add(startPanel);
       
        add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        
        // End panel
        
        endPanel = new JPanel();    
        endPanel.setLayout(new BoxLayout(endPanel, BoxLayout.Y_AXIS));
        endPanel.setAlignmentX(LEFT_ALIGNMENT);
        endPanel.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(" END "),
                BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));

        box = Box.createHorizontalBox();
        endGroup = new ButtonGroup();
        endFixedButton = new JRadioButton("Specific time: ");
        endGroup.add(endFixedButton);
        box.add(endFixedButton);
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        endChooser = new DateTimeSelector();
        box.add(endChooser);
        endChooser.setToolTipText("Select date and time for the end of the time window");
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        box.add(Box.createHorizontalGlue());
        endPanel.add(box);

        endPanel.add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        
        box = Box.createHorizontalBox();
        add(box);
        endDuraButton = new JRadioButton("");
        endGroup.add(endDuraButton);
        box.add(endDuraButton);
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        endDuraChooser = new DurationSelector(1, ChronoUnit.DAYS);
        endDuraChooser.setAmountLimits(1, Integer.MAX_VALUE);
        endDuraChooser.setToolTipText("Select time window duration");
        box.add(endDuraChooser);
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        afterLabel = new JLabel("after START.");
//        box.add(afterLabel);
        box.add(endDuraLabel = new JLabel("after START."));
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        box.add(Box.createHorizontalGlue());
        endPanel.add(box);

        endPanel.add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));

        box = Box.createHorizontalBox();
        add(box);
        endNowButton = new JRadioButton("Current time (re-evaluated on refresh)");
        endGroup.add(endNowButton);
        box.add(endNowButton);
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        box.add(Box.createHorizontalGlue());
        endPanel.add(box);

        endPanel.add(Box.createRigidArea(new Dimension(0, VSPACE)));
        add(endPanel);
       
        add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        
        // Remember As panel
       
        Box saveBox = Box.createVerticalBox();
        saveBox.setAlignmentX(LEFT_ALIGNMENT);
        saveBox.setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createTitledBorder(" Remember As "), 
                    BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
        nameField = new JTextField();
        nameField.setColumns(30);
        saveBox.add(nameField);
        saveBox.add(Box.createRigidArea(new Dimension(0, VSPACE)));
        saveCheck = new JCheckBox("Remember between sessions");
        saveBox.add(saveCheck);
        add(saveBox);
        nameField.addCaretListener(e -> {
            saveCheck.setEnabled(!nameField.getText().trim().isEmpty());
        });
        
        // Set initial state
        
        init();
        
        // Wire for events
        
        startFixedButton.addActionListener(this::onUserAction);
        startDuraButton.addActionListener(this::onUserAction);
        endFixedButton.addActionListener(this::onUserAction);
        endDuraButton.addActionListener(this::onUserAction);
        endNowButton.addActionListener(this::onUserAction);
        startChooser.addListener(this::onUserAction);
        startDuraChooser.addListener(this::onUserAction);
        endChooser.addListener(this::onUserAction);
        endDuraChooser.addListener(this::onUserAction);
    }
    
    /**
     * Opens trending time window creation/editing dialog.
     * 
     * @param parent Parent component.
     * @param seed Time window to be edited, or {@code null} if a new time window should be created.
     * @return Modified or created time window, or {@code null} if the dialog is canceled.
     */
    static public TimeWindow editTimeWindow(JComponent parent, TimeWindow seed) {
        final TimeWindowDialog tsp = new TimeWindowDialog(seed);
        tsp.okButton.addActionListener(e -> {
            tsp.outRange = tsp.getRange();
            Window w = SwingUtilities.getWindowAncestor(tsp);
            if (w != null) w.setVisible(false);
        });
        JButton cancelButton = new JButton("Cancel");
        cancelButton.addActionListener(e -> {
            Window w = SwingUtilities.getWindowAncestor(tsp);
            if (w != null) w.setVisible(false);
        });
        JOptionPane.showOptionDialog(parent, tsp, "Select time range", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE, null, new Object[] {tsp.okButton, cancelButton}, null);
        return tsp.outRange;
    }

    
// -- Local methods : ----------------------------------------------------------
    
    private TimeWindow getRange() {
        if (current == null) return null;
        String name = nameField.getText().trim();
        boolean persistent = !name.isEmpty() && saveCheck.isSelected();
        if (range == null) {
            range = new TimeWindow(name, current[0], current[1], persistent);
        } else {
            range.set(name, current[0], current[1], persistent);
        }
        return range;
    }
    
    private void init() {
    
        TimeWindow window = range == null ? new TimeWindow("", "now - 86400", "now", false) : range;
        
        long[] bounds = window.getBounds();
        long start = bounds[0]/1000L;
        long end = bounds[1]/1000L;
        if (end == 0L) {
            endNowButton.setSelected(true);
            endChooser.set(null);
            endDuraChooser.set(0);
            if (start > 0L) {
                startFixedButton.setSelected(true);
                startChooser.set(toLocal(start));
                startDuraChooser.set(0);
            } else {
                startDuraButton.setSelected(true);
                startChooser.set(null);
                startDuraChooser.set(-start);
            }
        } else if (end > 0L) {
            endFixedButton.setSelected(true);
            endChooser.set(toLocal(end));
            if (start > 0L) {
                startFixedButton.setSelected(true);
                startChooser.set(toLocal(start));
                long dura = end - start;
                startDuraChooser.set(dura);
                endDuraChooser.set(dura);
            } else {
                startChooser.set(toLocal(end + start));
                startDuraButton.setSelected(true);
                startDuraChooser.set(-start);
                endDuraChooser.set(-start);
            }
        } else { // start - specific, end - duration
            startFixedButton.setSelected(true);
            startChooser.set(toLocal(start));
            startDuraChooser.set(-end);
            endDuraButton.setSelected(true);
            endChooser.set(toLocal(start - end));
            endDuraChooser.set(-end);
        }
        
        onUserAction(null);
        
        saveCheck.setSelected(window.isPersistent());
        String name = window.getName();
        if (name != null && !name.isEmpty()) {
            nameField.setText(name);
            saveCheck.setEnabled(true);
        } else {
            saveCheck.setEnabled(false);
        }
    }
    
    private void onUserAction(Object source) {
        
        LocalDateTime start = startChooser.get();
        LocalDateTime end = endChooser.get();
        long duraStart = startDuraChooser.get();
        long duraEnd = endDuraChooser.get();
        String message = null;
        current = new long[] {0L, 0L};
                
        try {

            // synch durations:

            long dura = 0;
            if (duraStart == duraEnd) {
                dura = duraStart;
            } else if (startDuraButton.isSelected()) {
                dura = duraStart;
                endDuraChooser.set(dura);
            } else if (endDuraButton.isSelected()) {
                dura = duraEnd;
                startDuraChooser.set(dura);
            } else {
                startDuraChooser.set(0);
                endDuraChooser.set(0);
            }
            
            // enable/disable controls
            
            if (startFixedButton.isSelected()) {
                startChooser.setEnabled(true);
                startDuraChooser.setEnabled(false);
                beforeLabel.setEnabled(false);
                endDuraButton.setEnabled(true);
                endDuraLabel.setEnabled(true);
            } else {
                startChooser.setEnabled(false);
                startDuraChooser.setEnabled(true);
                beforeLabel.setEnabled(true);
                endDuraButton.setEnabled(false);
                endDuraLabel.setEnabled(false);
            }
            if (endFixedButton.isSelected()) {
                endChooser.setEnabled(true);
                endDuraChooser.setEnabled(false);
                afterLabel.setEnabled(false);
                startDuraButton.setEnabled(true);
                startDuraLabel.setEnabled(true);
            } else if (endDuraButton.isSelected()) {
                endChooser.setEnabled(false);
                endDuraChooser.setEnabled(true);
                afterLabel.setEnabled(true);
                startDuraButton.setEnabled(false);
                startDuraLabel.setEnabled(false);
            } else {
                endChooser.setEnabled(false);
                endDuraChooser.setEnabled(false);
                afterLabel.setEnabled(false);
                startDuraButton.setEnabled(true);
                startDuraLabel.setEnabled(true);
                endChooser.set(null);
            }
            
            // do the rest
            
            if (endNowButton.isSelected()) {
                if (startFixedButton.isSelected()) {  // Fixed start, sliding end
                    startDuraChooser.set(0);
                    endDuraChooser.set(0);
                    if (start == null) {
                        if (dura == 0) {
                            message = "Select START";
                        } else {
                            start = toLocal(System.currentTimeMillis() / 1000L - dura);
                            startChooser.set(start);
                        }
                    }
                    current[0] = toEpochSecond(start);
                } else {  // Sliding start and end
                    if (dura == 0) {
                        if (start == null) {
                            message = "Select duration";
                        } else {
                            start = null;
                            startChooser.set(start);
                            dura = secondsBetween(start, LocalDateTime.now(zone));
                            if (dura < 1) {
                                dura = 0;
                                message = "Select duration";
                            } else {
                                startDuraChooser.set(dura);
                                endDuraChooser.set(dura);
                            }
                        }
                    } else {
                        start = null;
                        startChooser.set(start);
                    }
                    current[0] = -dura;
                }
            } else if (endFixedButton.isSelected() && startFixedButton.isSelected()) { // specific times for START and END
                if (start == null) {
                    if (end == null) {
                        startDuraChooser.set(0);
                        endDuraChooser.set(0);
                        message = "Select START and END";
                    } else {
                        if (dura == 0) {
                            message = "Select START";
                        } else {
                            start = end.minusSeconds(dura);
                            startChooser.set(start);
                        }
                    }
                } else {
                    if (end == null) {
                        if (dura == 0) {
                            message = "Select END";
                        } else {
                            end = start.plusSeconds(dura);
                            endChooser.set(end);
                        }
                    } else {
                        dura = secondsBetween(start, end);
                        if (dura < 1) {
                            dura = 0;
                            message = "Negative duration";
                        }
                        startDuraChooser.set(dura);
                        endDuraChooser.set(dura);
                    }
                }
                if (message == null) {
                    current[0] = toEpochSecond(start);
                    current[1] = toEpochSecond(end);
                }
            } else if (endDuraButton.isSelected()) { // specific time for START and duration for END
                if (start == null) {
                    if (dura == 0) {
                        message = "Select START";
                    } else {
                        if (end == null) {
                            message = "Select START and END";
                        } else {
                            start = end.minusSeconds(dura);
                            startChooser.set(start);
                        }
                    }
                } else {
                    if (dura == 0) {
                        if (end == null) {
                            message = "Select END";
                        } else {
                            dura = secondsBetween(start, end);
                            if (dura < 1) {
                                dura = 0;
                                message = "Negative duration";
                            }
                            startDuraChooser.set(dura);
                            endDuraChooser.set(dura);
                        }
                    } else {
                        end = start.plusSeconds(dura);
                        endChooser.set(end);
                    }
                }
                if (message == null) {
                    current[0] = toEpochSecond(start);
                    current[1] = -dura;
                }
            } else {  // duration for START and specific time for END
                if (end == null) {
                    if (dura == 0) {
                        message = "Select END";
                    } else {
                        if (start == null) {
                            message = "Select START and END";
                        } else {
                            end = start.plusSeconds(dura);
                            endChooser.set(end);
                        }
                    }
                } else {
                    if (dura == 0) {
                        if (start == null) {
                            message = "Select START";
                        } else {
                            dura = secondsBetween(start, end);
                            if (dura < 1) {
                                dura = 0;
                                message = "Negative duration";
                            }
                            startDuraChooser.set(dura);
                            endDuraChooser.set(dura);
                        }
                    } else {
                        start = end.minusSeconds(dura);
                        startChooser.set(start);
                    }
                }
                if (message == null) {
                    current[0] = -dura;
                    current[1] = toEpochSecond(end);
                }
            }
            
            if (start != null && LocalDateTime.now(zone).isBefore(start)) {
                message = "START in the future is not allowed";
            }
        
        } catch (Exception x) {
            message = x.getMessage();
            if (message == null) message = x.toString();
        }
        
        okButton.setToolTipText(message);
        
        if (message == null) {
            okButton.setEnabled(true);
        } else {
            okButton.setEnabled(false);
            current = null;
        }
    }
    
    private LocalDateTime toLocal(long seconds) {
        return LocalDateTime.ofInstant(Instant.ofEpochSecond(seconds), zone);
    }
    
    private long toEpochSecond(LocalDateTime dateTime) {
        return dateTime.atZone(zone).toEpochSecond();
    }
    
    private long secondsBetween(LocalDateTime start, LocalDateTime end) {
        return Duration.between(start, end).toMillis() /1000L;
    }

    
// -- Testing : ----------------------------------------------------------------
    
    static public void main(String... args) {
        
        SwingUtilities.invokeLater(() -> {

            TimeWindow tw = null;
            while (true) {
                tw = TimeWindowDialog.editTimeWindow(null, tw);
                System.out.println(tw);
                if (tw == null) {
                    break;
                }
            }
        });

                
    }
    
//    static public void main(String... args) {
//        LocalDateTime start = LocalDateTime.of(2020, 1, 10, 0, 0, 0);
//        LocalDateTime end = LocalDateTime.of(2020, 1, 10, 1, 1, 0);
//        System.out.println("Millis: "+ Duration.between(start, end).toMillis() +", seconds: "+ Duration.between(start, end).toMillis()/1000L);
//        
//                
//    }
                
}
