package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridLayout;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import org.lsst.ccs.command.DictionaryArgument;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.command.Options;
import org.lsst.ccs.command.StringTokenizer;
import org.lsst.ccs.command.StringTokenizer.Token;
import org.lsst.ccs.command.SupportedOption;
import org.lsst.ccs.command.TokenizedCommand;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.gconsole.base.Const;

/**
 * Graphics component that displays information on a command and accepts arguments input.
 *
 * @author emarin
 */
public final class ArgInputPanel extends JPanel implements ListSelectionListener {
    
// -- Fields : -----------------------------------------------------------------

    private DictionaryCommand command;
    private ArgInput[] argGetters;
    private final JPanel formPane;
    private final JEditorPane commandDesc;
    private JButton sendCmdButton;
    
    private final Map<String,String[]> history = new LinkedHashMap<String,String[]>(4, 0.8f, true) {
        @Override
        protected boolean removeEldestEntry(Map.Entry<String,String[]> eldest) {
            return size() > 50;
        }
    };
    private boolean valuesValid;
    
    private final String OPTIONS_TYPE = "opts";

// -- Life cycle : -------------------------------------------------------------
    
    public ArgInputPanel() {
        setLayout(new BorderLayout());
        
        // Command description panel:
        
        commandDesc = new JEditorPane();
        add(new JScrollPane(commandDesc), BorderLayout.CENTER);
        commandDesc.setBackground(new Color(0xed, 0xec, 0xeb)); //TODO : color must match system colors
        commandDesc.setEditable(false);
        commandDesc.setContentType("text/html");
        commandDesc.setText("      ");
        
        // Argument input panel:
        
        formPane = new JPanel();
        formPane.setBorder(BorderFactory.createCompoundBorder(
                      BorderFactory.createEtchedBorder(), 
                      BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE)));
    }
    
    void setCommandButton(JButton button) {
        sendCmdButton = button;
    }
    
    
// -- Updates : ----------------------------------------------------------------
    
    @Override
    public void valueChanged(ListSelectionEvent lse) {
        CommandListPanel commandListPanel = (CommandListPanel) lse.getSource();
        command = commandListPanel.getSelectedValue();
        clear();
        if (command != null) {
            fill();
        }
        boolean enabled = commandListPanel.isVisibleCommandSelected();
        setEnabled(enabled);
        if (enabled) {
            add(formPane, BorderLayout.SOUTH);
            updateValidity(true);
        } else {
            remove(formPane);
            sendCmdButton.setEnabled(false);
        }
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns a pretty string for displaying the values to the user.
     * The returned string is a valid representation of arguments in a shell command.
     * @return Human-readable representation of argument values.
     */
    public String getValuesAsString() {
        return String.join(" ", retrieveValues());
    }
    
    private List<String> retrieveValues() {
        int n = argGetters.length;
        if (n == 0) {
            return Collections.emptyList();
        }
        List<String> out = new ArrayList<>(n);
        for (int i = 0; i < n; i++) {
            String s = argGetters[i].get();
            String type = argGetters[i].getType();
            if (OPTIONS_TYPE.equals(type)) {
                if (!s.isEmpty()) {
                    out.add(s);
                }
            } else {
                boolean isCollection = type.equals("List") || type.equals("Set") || type.equals("Map");
                List<Token> tt = StringTokenizer.tokenize(s);
                if (command.isVarArgs() && i == n - 1) {
                    if (!tt.isEmpty()) {
                        out.add(s);
                    }
                } else {
                    if (tt.isEmpty()) {
                        if (isCollection) {
                            out.add("[]");
                        } else {
                            out.add("\"" + s + "\"");
                        }
                    } else if (tt.size() == 1) {
                        Token t = tt.get(0);
                        String ts = t.getString();
                        if (isCollection) {
                            if (ts.isEmpty()) {
                                out.add("[]");
                            } else if (ts.charAt(0) == '[') {
                                out.add(ts);
                            } else {
                                out.add("[" + ts + "]");
                            }
                        } else {
                            out.add(s);
                        }
                    } else {
                        if (isCollection) {
                            out.add("[" + s + "]");
                        } else {
                            out.add("\"" + s + "\"");
                        }
                    }
                }
            }
        }
        if (command != null && out != null && !out.isEmpty()) {
            history.put(command.getCommandName(), out.toArray(new String[0]));
        }
        return out;
    }

    
// -- Local methods : ----------------------------------------------------------
    
    private void clear() {
        formPane.removeAll();
        commandDesc.setText("       ");
        valuesValid = false;
        sendCmdButton.setEnabled(valuesValid);
    }
    
    private void fill() {
        
        // Argument input panel:
        
        int nArgs = command.getArguments().length;
        List<SupportedOption> opts = command.getSupportedOptions();
        boolean hasOptions = !opts.isEmpty();
        int nInputs = hasOptions ? nArgs + 1 : nArgs;
        formPane.setLayout(new GridLayout(nInputs, 2));
        argGetters = new ArgInput[nInputs];
        if (hasOptions) {
            argGetters[0] = new ArgInput(opts, "");
            formPane.add(argGetters[0].getLabel());
            formPane.add(argGetters[0].getEditor());
        }
        for (int i = 0; i < nArgs; i++) {
            DictionaryArgument da = command.getArguments()[i];
            boolean isVararg = command.isVarArgs() && i == nArgs-1;
            String defValue = getDefaultValueOfArgument(da);
            if (defValue == null && isVararg) {
                defValue = "";
            }
            ArgInput input = new ArgInput(da, defValue, isVararg);
            argGetters[hasOptions ? i+1 : i] = input;
            formPane.add(input.getLabel());
            formPane.add(input.getEditor());
        }
        String[] hist = history.get(command.getCommandName());
        if (hist != null && hist.length <= nInputs) {
            for (int ii = 0; ii < hist.length; ii++) {
                if (hist[ii] != null) {
                    argGetters[ii].set(hist[ii]);
                }
            }
        }

        // Command description panel:
        
        StringBuilder sb = new StringBuilder("<html>");
        sb.append(command.getType()).append(". Level: ").append(command.getLevel());
        sb.append("<h3>").append(command.getDescription()).append("</h3>");
        commandDesc.setText(sb.toString());
        
        // Refresh display:
        
        updateValidity(true);
        setEnabled(isEnabled());
        repaint();
        validate();
    }
    
    private void updateValidity(boolean argValid) {
        if (valuesValid == argValid) return; // no change
        if (argValid) {
            for (ArgInput argInput : argGetters) {
                if (!argInput.isValueValid()) {
                    return;  // overall state is still invalid
                }
            }
            valuesValid = true; // all argument values are now valid
        } else { // overall state is now invalid
            valuesValid = false;
        }
        sendCmdButton.setEnabled(valuesValid);
    }

    @Override
    public void setEnabled(boolean enabled) {
        super.setEnabled(enabled);
        if (enabled) {
            add(formPane, BorderLayout.SOUTH);
        } else {
            this.remove(formPane);
        }
    }
    
    private String getDefaultValueOfArgument(DictionaryArgument da) {
        String out = da.getDefaultValue();
        if (out != null) {
            switch (out) {
                case Argument.NOT_SET:
                    return null;
                case Argument.NULL:
                    return "null";
            }
        }
        return out;
    }
    
    
// -- Single argument input component : ----------------------------------------
    
    private class ArgInput {
        
        private final DictionaryArgument da;
        private final boolean isVararg;
        private boolean valueValid;
        private final JComponent editor;
        private final JLabel label;
        
        ArgInput(DictionaryArgument da, String defValue, boolean isVararg) {
            this.da = da;
            this.isVararg = isVararg;
            
            label = new JLabel(da.getName());
            String tip = da.getDescription();
            if (tip != null && !tip.isEmpty()) label.setToolTipText(tip);

            List<String> allowedValues = da.getAllowedValues();
            String type = da.getSimpleType();
            if (allowedValues.isEmpty() && "boolean".equals(type)) {
                allowedValues = Arrays.asList(new String[] {"true", "false"});
            }
            if(allowedValues.isEmpty()) {
                HintTextField htf = new HintTextField(isVararg ? type +"..." : type);
                if (defValue == null) {
                    valueValid = false;
                } else {
                    htf.setText(defValue);
                    valueValid = true;
                }
                htf.addCaretListener(e -> validateValue());
                editor = htf;
            } else {
                JComboBox<String> comboBox = new JComboBox<>(allowedValues.toArray(new String[0]));
                defValue = da.getDefaultValue();
                if (defValue != null) {
                    comboBox.setSelectedItem(defValue);
                }
                valueValid = true;
                editor = comboBox;
            }
        }
        
        ArgInput(List<SupportedOption> options, String defValue) {
            da = null;
            isVararg = false;
            label = new JLabel("Options");
            label.setToolTipText("Click the button to enter options");
            valueValid = true;
            editor = new OptionsEditor(options, defValue);
        }
        
        void set(String value) {
            if (editor instanceof HintTextField) {
                ((HintTextField)editor).setText(value);
            } else if (editor instanceof JComboBox) {
                ((JComboBox<String>)editor).setSelectedItem(value);
            } else if (editor instanceof OptionsEditor) {
                ((OptionsEditor)editor).set(value);
            }
            validateValue();
        }
        
        String get() {
            String out;
            if (editor instanceof HintTextField) {
                out = ((HintTextField)editor).getText();
            } else if (editor instanceof JComboBox) {
                out = ((JComboBox<String>)editor).getSelectedItem().toString();
            } else {
                out = ((OptionsEditor)editor).get();
            }
            return out;
        }
        
        String getType() {
            return da == null ? OPTIONS_TYPE : da.getSimpleType();
        }
        
        boolean isValueValid() {
            return valueValid;
        }
        
        // Ideally, validation should use the same algorithm as the command invocation, but there seem to be no clean way to inplement
        // this with the current command utilities. DictionaryArgument does not know Type, but InputConversionEngine needs it.
        private void validateValue() {
            boolean nowValid = true;
            if (editor instanceof HintTextField) {
                String in = ((HintTextField)editor).getText();
                String[] ss = isVararg ? in.split("\\s+") : new String[] {in};
                for (String s : ss) {
                    switch (da.getSimpleType()) {
                        case "int":
                        try {
                            Integer.decode(s);
                        } catch (NumberFormatException x) {
                            nowValid = false;
                        }
                        break;
                        case "long":
                        try {
                            Long.decode(s);
                        } catch (NumberFormatException x) {
                            nowValid = false;
                        }
                        break;
                        case "float":
                        try {
                            Float.parseFloat(s);
                        } catch (NumberFormatException x) {
                            nowValid = false;
                        }
                        break;
                        case "double":
                        try {
                            Double.parseDouble(s);
                        } catch (NumberFormatException x) {
                            nowValid = false;
                        }
                        break;
                    } 
                }
            }
            if (nowValid != valueValid) {
                valueValid = nowValid;
                updateValidity(valueValid);
            }
        }
        
        JComponent getEditor() {
            return editor;
        }
        
        JLabel getLabel() {
            return label;
        }

    }
    
    static private final class OptionsEditor extends JButton {
        
        private final SupportedOption[] supportedOptions;
        private final boolean[] options;
        
        OptionsEditor(List<SupportedOption> opt, String value) {
            int n = opt.size();
            supportedOptions = opt.toArray(new SupportedOption[n]);
            options = new boolean[n];
            set(value);
            addActionListener(e -> edit());
        }
        
        void set(String value) {
            TokenizedCommand tc = new TokenizedCommand("command "+ value);
            Options opt = tc.getOptions();
            Set<String> optSet = opt.getOptions();
            StringBuilder sb = new StringBuilder("-");
            for (int i=0; i<options.length; i++) {
                SupportedOption so = supportedOptions[i];
                if (optSet.contains(so.getSingleLetterName()) || optSet.contains(so.getName())) {
                    sb.append(so.getSingleLetterName());
                    options[i] = true;
                } else {
                    options[i] = false;
                }
            }
            value = sb.length() == 1 ? "" : sb.toString();
            setText(value);
        }
        
        String get() {
            return getText();
        }
        
        private void edit() {
            int n = supportedOptions.length;
            List<JCheckBox> checkBoxList = new ArrayList<>(n);
            JPanel root = new JPanel(new GridLayout(n, 1, Const.HSPACE, Const.VSPACE));
            for (int i=0; i<n; i++) {
                SupportedOption so = supportedOptions[i];
                JCheckBox cb = new JCheckBox(so.getName());
                cb.setSelected(options[i]);
                cb.setToolTipText(so.getDescription());
                checkBoxList.add(cb);
                root.add(cb);
            }
            int out = JOptionPane.showOptionDialog(this, root, "Select options", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE, null, null, null);
            if (out == JOptionPane.OK_OPTION) {
                StringBuilder sb = new StringBuilder("-");
                for (int i=0; i<n; i++) {
                    if (checkBoxList.get(i).isSelected()) {
                        sb.append(supportedOptions[i].getSingleLetterName());
                        options[i] = true;
                    } else {
                        options[i] = false;
                    }
                }
                String value = sb.length() == 1 ? "" : sb.toString();
                setText(value);
            }
        }
    }

}
