package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.*;
import java.util.concurrent.CancellationException;
import java.util.function.Supplier;
import javax.swing.AbstractCellEditor;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.CellEditor;
import javax.swing.DefaultCellEditor;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.WindowConstants;
import javax.swing.border.Border;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableModel;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import static org.lsst.ccs.gconsole.base.Const.HSPACE;
import static org.lsst.ccs.gconsole.base.Const.VSPACE;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.plugins.tracer.FilterStep.Method;
import org.lsst.ccs.gconsole.plugins.tracer.FilterStep.Mode;
import org.lsst.ccs.gconsole.plugins.tracer.FilterStep.Target;
import org.lsst.ccs.gconsole.services.persist.Persistable;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;
import org.openide.awt.ColorComboBox;

/**
 * GUI editor for {@link UserFilter}.
 *
 * @author onoprien
 */
public final class TracerEditor extends JDialog {

// -- Fields : -----------------------------------------------------------------
    
    private final int C_OR = 0;
    private final int C_MODE = 1;
    private final int C_INVERT = 2;
    private final int C_FORMAT = 3;
    private final int C_TARGET = 4;
    private final int C_METHOD = 5;
    private final int C_CODE = 6;
    private final int C_COLOR = 7;
    private final int C_FLAG = 8;
    private final String[] columnNames = {"OR", "Mode", "Invert", "Format", "Target", "Operation", "Parameters", "Color", "Flag"};
    private final Class[] columnClasses = {Boolean.class, FilterStep.Mode.class, Boolean.class, Boolean.class, FilterStep.Target.class, FilterStep.Method.class, String.class, Color.class, FilteredMessage.Flag.class};
    
    private final JTable table;
    private final Model model;
    private String title;
    
    private JButton insertButton;
    private JButton removeButton;
    private JTextField titleField;
    private JButton upButton;
    private JButton downButton;
    private JButton saveButton;
    private JButton saveAsButton;
    private final JButton okButton;
    
    private final Tracer tracer;
    private boolean cancelled = false;


// -- Life cycle : -------------------------------------------------------------
    
    private TracerEditor(Tracer tracer, Window parent) {
        
        super(parent, "Edit Tracer Filter", DEFAULT_MODALITY_TYPE.APPLICATION_MODAL);
        setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                cancelled = true;
            }
        });
        
        this.tracer = tracer;
        
        title = tracer == null ? "Messages" : tracer.getDescriptor().getName();
        model = new Model();
        table = new Table(model);
        table.setFillsViewportHeight(true);
        table.setSelectionMode(ListSelectionModel.SINGLE_INTERVAL_SELECTION);
        table.getSelectionModel().addListSelectionListener(e -> {
            int first = table.getSelectedRow();
            int count = table.getSelectedRowCount();
            insertButton.setEnabled(count > 0);
            removeButton.setEnabled(count > 0);
            upButton.setEnabled(first > 0);
            downButton.setEnabled(count > 0 && first+count < table.getRowCount());
        });
        add(new JScrollPane(table), BorderLayout.CENTER);
        
        // mode
        
        JComboBox combo = new JComboBox();
        for (Mode state : Mode.values()) {
            combo.addItem(state);
        }
        DefaultCellEditor editor = new DefaultCellEditor(combo);
        editor.setClickCountToStart(1);
        table.getColumnModel().getColumn(C_MODE).setCellEditor(editor);

        // target
        
        combo = new JComboBox();
        for (Target target : Target.values()) {
            combo.addItem(target);
        }
        editor = new DefaultCellEditor(combo);
        editor.setClickCountToStart(1);
        table.getColumnModel().getColumn(C_TARGET).setCellEditor(editor);
        
        // method
        
        combo = new JComboBox();
        for (Method method : Method.values()) {
            combo.addItem(method);
        }
        editor = new DefaultCellEditor(combo);
        editor.setClickCountToStart(1);
        table.getColumnModel().getColumn(C_METHOD).setCellEditor(editor);
        
        // code
        
        table.getColumnModel().getColumn(C_CODE).setCellRenderer(new CodeRenderer());
        table.getColumnModel().getColumn(C_CODE).setCellEditor(new CodeEditor());
        
        // color
        
        table.getColumnModel().getColumn(C_COLOR).setCellRenderer(new ColorRenderer());
        table.getColumnModel().getColumn(C_COLOR).setCellEditor(new ColorEditor());
        
        // action
        
//        combo = new JComboBox();
//        for (FilteredMessage.Flag act : FilteredMessage.Flag.values()) {
//            combo.addItem(act);
//        }
//        editor = new DefaultCellEditor(combo);
//        editor.setClickCountToStart(2);
//        table.getColumnModel().getColumn(C_FLAG).setCellEditor(editor);
        
        // top controls
        
        Box controls = Box.createHorizontalBox();
        controls.setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
        add(controls, BorderLayout.NORTH);
        JButton button = new JButton("Add");
        button.setToolTipText("Add step");
        button.addActionListener(e -> {
            CellEditor ce = table.getCellEditor();
            if (ce == null || ce.stopCellEditing()) {
                model.insertRow(model.getRowCount());
            }
        });
        controls.add(button);
        controls.add(Box.createHorizontalStrut(HSPACE));
        insertButton = new JButton("Insert");
        insertButton.setToolTipText("Insert step");
        insertButton.setEnabled(false);
        insertButton.addActionListener(e -> {
            CellEditor ce = table.getCellEditor();
            if (ce == null || ce.stopCellEditing()) {
                int i = table.getSelectedRow();
                model.insertRow(i < 0 ? 0 : i);
            }
        });
        controls.add(insertButton);
        controls.add(Box.createHorizontalStrut(HSPACE));
        removeButton = new JButton("Remove");
        removeButton.setToolTipText("Remove selected step");
        removeButton.setEnabled(false);
        removeButton.addActionListener(e -> {
            CellEditor ce = table.getCellEditor();
            if (ce == null || ce.stopCellEditing()) {
                model.removeRows(table.getSelectedRows());
            }
        });
        controls.add(removeButton);
        controls.add(Box.createRigidArea(Const.HDIM2));
        controls.add(Box.createHorizontalGlue());
        String tt = "Choose tab title for the viewer";
        JLabel label = new JLabel("Title: ");
        label.setToolTipText(tt);
        controls.add(label);
        titleField = new JTextField(title);
        titleField.setToolTipText(tt);
        titleField.addActionListener(e -> title = titleField.getText().trim());
        controls.add(titleField);
        controls.add(Box.createHorizontalGlue());
        controls.add(Box.createRigidArea(Const.HDIM2));
        upButton = new JButton(" Up ");
        upButton.setToolTipText("Move selected step up in sequence");
        upButton.setEnabled(false);
        upButton.addActionListener(e -> move(-1));
        controls.add(upButton);
        controls.add(Box.createHorizontalStrut(HSPACE));
        downButton = new JButton("Down");
        downButton.setToolTipText("Move selected step down in sequence");
        downButton.setEnabled(false);
        downButton.addActionListener(e -> move(1));
        controls.add(downButton);
        
        // bottom controls
        
        controls = Box.createHorizontalBox();
        controls.setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
        add(controls, BorderLayout.SOUTH);
        saveButton = new JButton("Save");
        saveButton.setEnabled(tracer.getDescriptor().getPath() != null);
        saveButton.addActionListener(e -> {
            if (apply()) {
                PersistenceService.getService().save(tracer.save());
            }
        });
        controls.add(saveButton);
        controls.add(Box.createHorizontalStrut(HSPACE));
        saveAsButton = new JButton("Save As...");
        saveAsButton.addActionListener(e -> {
            if (apply()) {
                PersistenceService.getService().saveAs(tracer.save(), "Save", saveAsButton);
            }
        });
        controls.add(saveAsButton);
        controls.add(Box.createHorizontalStrut(2*HSPACE));
        controls.add(Box.createHorizontalGlue());
        JButton applyButton = new JButton("Apply");
        applyButton.addActionListener(e -> apply());
        controls.add(applyButton);
        controls.add(Box.createHorizontalStrut(HSPACE));
        okButton = new JButton("OK");
        okButton.addActionListener(e -> {
            if (apply()) {
                TracerEditor.this.dispose();
            }
        });
        controls.add(okButton);
        controls.add(Box.createHorizontalStrut(HSPACE));
        button = new JButton("Cancel");
        button.addActionListener(e -> TracerEditor.this.cancel());
        controls.add(button);

        table.getColumnModel().getColumn(C_OR).setPreferredWidth(100);
        table.getColumnModel().getColumn(C_MODE).setPreferredWidth(150);
        table.getColumnModel().getColumn(C_INVERT).setPreferredWidth(150);
        table.getColumnModel().getColumn(C_FORMAT).setPreferredWidth(150);
        table.getColumnModel().getColumn(C_TARGET).setPreferredWidth(170);
        table.getColumnModel().getColumn(C_METHOD).setPreferredWidth(170);
        table.getColumnModel().getColumn(C_CODE).setPreferredWidth(800);
        table.getColumnModel().getColumn(C_COLOR).setPreferredWidth(170);
//        table.getColumnModel().getColumn(C_FLAG).setPreferredWidth(170);
        table.clearSelection();
        pack();
        setLocationRelativeTo(getOwner());
    }
    
    /**
     * Opens a dialog and lets the user edit the provided {@code Tracer}.
     * The {@code Tracer} instance returned by this method is always the instance supplied as an argument,
     * but the message filter and descriptor may be replaced or modified.
     * 
     * @param tracer Tracer to be edited.
     * @param parent Parent component.
     * @return Edited Tracer.
     */
    static public Tracer edit(Tracer tracer, Component parent) {
        Window owner;
        try {
            owner = (Window) ((JComponent) parent).getTopLevelAncestor();
        } catch (ClassCastException | NullPointerException x) {
            owner = Console.getConsole().getWindow();
            parent = owner;
        }
        TracerEditor dialog = new TracerEditor(tracer, owner);
        dialog.setPreferredSize(new Dimension(800,600));
        dialog.pack();
        dialog.setLocationRelativeTo(parent);
        dialog.setVisible(true);
        if (dialog.cancelled) {
            throw new CancellationException();
        }
        return tracer;
    }
    
// -- Local methods : ----------------------------------------------------------
    
    private void move(int dir) {
        
        CellEditor ce = table.getCellEditor();
        if (ce != null && !ce.stopCellEditing()) return;
       
        int begSel = table.getSelectedRow();
        int k = table.getSelectedRowCount();
        if (k == 0) return;
        int endSel = begSel+k-1;
        int n = table.getRowCount();
        
        boolean isOrFragment = true;
        for (int i=0; i<k; i++) {
            isOrFragment = isOrFragment && isOR(begSel+i);
        }
        int begNext = dir > 0 ? endSel+1 : begSel-1;
        if (begNext < 0 || begNext >= n) return;
        isOrFragment = isOrFragment && isOR(begNext);
        
        if (isOrFragment) {
            if (dir > 0) {
                model.move(begSel, endSel, begNext, begNext);
                table.getSelectionModel().setSelectionInterval(begSel+1, endSel+1);
            } else {
                model.move(begNext, begNext, begSel, endSel);
                table.getSelectionModel().setSelectionInterval(begSel-1, endSel-1);
            }
        } else {
            if (isOR(begSel)) {
                while (begSel > 0 && isOR(begSel - 1)) {
                    begSel--;
                }
            }
            if (isOR(endSel)) {
                while ((endSel+1) < n && isOR(endSel+1)) {
                    endSel++;
                }
            }
            begNext = dir > 0 ? endSel + 1 : begSel - 1;
            if (begNext < 0 || begNext >= n) return;
            int endNext = begNext;
            if (isOR(endNext)) {
                int bound = dir > 0 ? n : -1;
                while ((endNext+dir != bound) && isOR(endNext+dir)) {
                    endNext += dir;
                }
            }
            k = endNext-begNext+1;
            if (dir > 0) {
                model.move(begSel, endSel, begNext, endNext);
                table.getSelectionModel().setSelectionInterval(begSel+k, endSel+k);
            } else {
                model.move(endNext, begNext, begSel, endSel);
                table.getSelectionModel().setSelectionInterval(begSel-k, endSel-k);
            }
        }

    }
    
    private boolean isOR(int rowIndex) {
        return model.rows.get(rowIndex).or;
    }
    
    private void validateTable() {
        boolean allValid = model.rows.stream().allMatch(step -> step.valid);
        okButton.setEnabled(allValid);
        saveButton.setEnabled(allValid && tracer.getDescriptor().getPath() != null);
        saveAsButton.setEnabled(allValid);
    }
    
    private boolean apply() {
        try {
            MessageFilter filter = model.makeFilter();
            tracer.setFilter(filter);
            title = titleField.getText().trim();
            if (title != null && !title.isEmpty() && !title.equals(tracer.getDescriptor().getName())) {
                tracer.getDescriptor().setName(title);
                if (tracer.getPanel() != null) {
                    Console.getConsole().getPanelManager().set(tracer.getPanel(), Panel.TITLE, title);
                }
            }
            return true;
        } catch (Exception x) {
            return false;
        }
    }
    
    private void cancel() {
        cancelled = true;
        dispose();
    }
    
    private void editCode(Step step, Component parent) {
        switch (step.target) {
            case TEMPLATE:
                String template, pattern;
                if (step.code.length == 2) {
                    template = step.code[0];
                    pattern = step.code[1];
                } else {
                    template = "";
                    pattern = "";
                }
                TextFieldPanel templatePanel = new TextFieldPanel(template, "Template:", null);
                TextFieldPanel patternPanel = new TextFieldPanel(pattern, "Pattern:", null);
                Box box = Box.createVerticalBox();
                box.add(templatePanel);
                box.add(Box.createRigidArea(new Dimension(0, 2 * VSPACE)));
                box.add(patternPanel);
                box.add(Box.createRigidArea(new Dimension(HSPACE, VSPACE)));
                int ok = JOptionPane.showConfirmDialog(parent, box, "Filter definition", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);
                if (ok == JOptionPane.OK_OPTION) {
                    try {
                        step.code = new String[] {templatePanel.get(), patternPanel.get()};
                        step.delegate = null;
                    } catch (NullPointerException x) {
                    }
                }
                break;
            default:
                switch (step.method) {
                    case REGEX:
                        enterCode(step, "Regular expression:", null, parent);
                        break;
                    case WILDCARD:
                        enterCode(step, "Wildcard:", null, parent);
                        break;
                    case CONTAINS:
                    case EQUALS:
                        enterCode(step, "String:", null, parent);
                        break;
                    case CLASS:
                        enterCode(step, "Class name:", "Enter full or short class name", parent);
                        break;
                    case NAME:
                        try {
                            Tracer.Descriptor desc = step.delegate == null ? null : step.delegate.getDescriptor();
                            Tracer t = (Tracer) PersistenceService.getService().make(desc, "Select filter", parent, Tracer.CATEGORY);
                            if (t != null) {
                                step.delegate = t;
                                step.code = FilterStep.getCode(t);
                            }
                        } catch (IllegalArgumentException | CancellationException | NullPointerException x) {
                        }
                        break;
                    default:
                }
        }
    }
    
    private void enterCode(Step step, String title, String description, Component parent) {
        TextFieldPanel p = new TextFieldPanel(step.code[0], title, description);
        int ok = JOptionPane.showConfirmDialog(parent, p, "Filter definition", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);
        if (ok == JOptionPane.OK_OPTION) {
            step.code = new String[] {p.get()};
            step.delegate = null;
        }
    }
    
    static private class TextFieldPanel extends JPanel implements Supplier<String> {

        private JTextField field;

        TextFieldPanel(String seed, String title, String description) {
            setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            JLabel label = new JLabel(title);
            label.setAlignmentX(LEFT_ALIGNMENT);
            add(label);
            add(Box.createRigidArea(new Dimension(0, VSPACE)));
            field = new JTextField(40);
            field.setAlignmentX(LEFT_ALIGNMENT);
            if (seed != null) field.setText(seed);
            field.setToolTipText(description);
            add(field);
        }

        @Override
        public String get() {
            return field.getText();
        }
    }

    
// -- FILTER table model : -----------------------------------------------------
    
    private class Model extends AbstractTableModel {

        ArrayList<Step> rows;
        
        Model() {
            MessageFilter filter = tracer == null ? null : tracer.getFilter();
            if (filter == null) {
                rows = new ArrayList<>(1);
                rows.add(new Step());
            } else if (filter instanceof MultistepFilter) {
                MultistepFilter msFilter = (MultistepFilter) filter;
                int n = msFilter.ors.length;
                rows = new ArrayList<>(n);
                for (int i=0; i<n; i++) {
                    rows.add(new Step(msFilter.ors[i], msFilter.steps[i]));
                }
            } else {
                rows = new ArrayList<>(1);
                Tracer copy = (Tracer) PersistenceService.getService().make(tracer.save());
                if (copy == null) {
                    rows.add(new Step());
                } else {
                    rows.add(new Step(copy));
                }
            }
        }
        
        @Override
        public Class<?> getColumnClass(int columnIndex) {
            return columnClasses[columnIndex];
        }

        @Override
        public String getColumnName(int column) {
            return columnNames[column];
        }

        @Override
        public int getRowCount() {
            return rows.size();
        }

        @Override
        public int getColumnCount() {
            return columnNames.length - 1;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            Step step = rows.get(rowIndex);
            switch (columnIndex) {
                case C_OR:
                    return step.or;
                case C_MODE:
                    return step.mode;
                case C_INVERT:
                    return step.invert;
                case C_FORMAT:
                    return step.format;
                case C_TARGET:
                    return step.target;
                case C_METHOD:
                    return step.method;
                case C_CODE:
                    return step.code;
                case C_COLOR:
                    return step.color;
                case C_FLAG:
                    return step.flag;
                default:
                    return null;
            }
        }

        @Override
        public void setValueAt(Object value, int rowIndex, int columnIndex) {
            Step step = rows.get(rowIndex);
            switch (columnIndex) {
                case C_OR:
                    step.or = (Boolean) value;
                    break;
                case C_MODE:
                    step.mode = (Mode) value;
                    break;
                case C_INVERT:
                    step.invert = (Boolean) value;
                    break;
                case C_FORMAT:
                    step.format = (Boolean) value;
                    break;
                case C_TARGET:
                    step.target = (Target) value;
                    switch (step.method) {
                        case NAME:
                            if (step.target != Target.MESSAGE) {
                                step.method = Method.CONTAINS;
                            }
                            break;
                        case CLASS:
                            if (step.target != Target.MESSAGE && step.target != Target.OBJECT) {
                                step.method = Method.CONTAINS;
                            }
                    }
                    validateRow(step, rowIndex, columnIndex);
                    break;
                case C_METHOD:
                    step.method = (Method) value;
                    switch (step.method) {
                        case NAME:
                            if (step.target != Target.MESSAGE) {
                                step.target = Target.MESSAGE;
                            }
                            break;
                        case CLASS:
                            if (step.target != Target.MESSAGE && step.target != Target.OBJECT) {
                                step.target = Target.MESSAGE;
                            }
                    }
                    validateRow(step, rowIndex, columnIndex);
                    break;
                case C_CODE:
                    step.code = (String[]) value;
                    validateRow(step, rowIndex, columnIndex);
                    break;
                case C_COLOR:
                    step.color = (Color) value;
                    break;
                case C_FLAG:
                    step.flag = (FilteredMessage.Flag) value;
                    break;
            }
        }
        
        private void validateRow(Step step, int rowIndex, int columnIndex) {
            boolean wasValid = step.valid;
            try {
                makeFilter(step);
                step.valid = true;
            } catch (Exception x) {
                step.valid = false;
            }
            if (wasValid != step.valid) {
                validateTable();
            }
            fireTableRowsUpdated(rowIndex, rowIndex);
        }

        @Override
        public boolean isCellEditable(int rowIndex, int columnIndex) {
            return true;
        }
        
        void insertRow(int i) {
            rows.add(i, new Step());
            fireTableRowsInserted(i, i);
        }
        
        void removeRows(int[] ii) {
            if (ii.length == 0) return;
            Arrays.sort(ii);
            for (int i=ii.length; i>0; ) {
                rows.remove(ii[--i]);
            }
            fireTableRowsDeleted(ii[0], ii[ii.length-1]);
            validateTable();
        }
    
        private void move(int b1, int e1, int b2, int e2) {
            int n = rows.size();
            ArrayList<Step> newRows = new ArrayList<>(n);
            for (int i=0; i<b1; i++) {
                newRows.add(rows.get(i));
            }
            for (int i=b2; i<=e2; i++) {
                newRows.add(rows.get(i));
            }
            for (int i=b1; i<=e1; i++) {
                newRows.add(rows.get(i));
            }
            for (int i=e2+1; i<n; i++) {
                newRows.add(rows.get(i));
            }
            rows = newRows;
            fireTableRowsUpdated(b1, e2);
        }
    
        private MultistepFilter makeFilter() {
            int nSteps = rows.size();
            if (nSteps == 0) return null;
            FilterStep[] steps = new FilterStep[nSteps];
            boolean[] ors = new boolean[nSteps];
            for (int i = 0; i < nSteps; i++) {
                Step step = rows.get(i);
                ors[i] = step.or;
                steps[i] = makeFilter(step);
            }
            return new MultistepFilter(steps, ors);
        }
        
        private FilterStep makeFilter(Step step) {
            if (step.method == Method.NAME) {
                return new FilterStep(step.delegate, step.mode, step.invert, step.format, step.color, step.flag);
            } else {
                return new FilterStep(step.mode, step.invert, step.format, step.target, step.method, step.code, step.color, step.flag);
            }
        }

    }
    
    static private class Step {
        
        boolean or;
        Mode mode;
        boolean invert;
        boolean format;
        Target target;
        Method method;
        String[] code;
        Color color;
        FilteredMessage.Flag flag;
        
        Tracer delegate;
        
        boolean valid = true;
        
        Step() {
            or = false;
            mode = Mode.ON;
            invert = false;
            format = true;
            target = Target.MESSAGE;
            method = Method.CONTAINS;
            code = new String[] {""};
            color = null;
            flag = null;
            delegate = null;
        }
        
        Step(boolean or, FilterStep step) {
            this.or = or;
            mode = step.getMode();
            invert = step.isInverted();
            format = step.isFormatting();
            target = step.getTarget();
            method = step.getMethod();
            code = step.getCode();
            color = step.getColor();
            flag = step.getFlag();
            delegate = step.getDelegate();
        }
        
        Step(Tracer tracer) {
            or = false;
            mode = Mode.ON;
            invert = false;
            format = true;
            target = Target.MESSAGE;
            method = Method.NAME;
            code = FilterStep.getCode(tracer);
            color = null;
            flag = null;
            delegate = tracer;
        }
        
    }
    
    
// -- FILTER table class : -----------------------------------------------------
    
    private class Table extends JTable {
        
        Table(TableModel model) {
            super(model);
        }

        @Override
        protected JTableHeader createDefaultTableHeader() {
            JTableHeader th = new JTableHeader(columnModel) {
                @Override
                public String getToolTipText(MouseEvent e) {
                    int index = columnModel.getColumnIndexAtX(e.getPoint().x);
                    int realIndex = columnModel.getColumn(index).getModelIndex();
                    return getToolTip(realIndex);
                }
            };
            th.setReorderingAllowed(false);
            return th;
        }
        
        String getToolTip(int column) {
            StringBuilder sb;
            switch (column) {
                case C_OR:
                    return "Check if this step is a part of an OR group";
                case C_MODE:
                    sb = new StringBuilder("<html><h3>Mode:</h3><dl>");
                    for (Mode mode : Mode.values()) {
                        sb.append("<dt>").append(mode).append("</dt><dd>").append(mode.getToolTip()).append("</dd>");
                    }
                    return sb.append("</dl></html>").toString();
                case C_INVERT:
                    return "Check if this step acceptance criteria should be inverted";
                case C_FORMAT:
                    return "<html>Check if the target string produced by this step should be saved and passed to subsequent filters</html>";
                case C_TARGET:
                    sb = new StringBuilder("<html><b>Target to which this step should be applied:</b><dl>");
                    for (Target target : Target.values()) {
                        sb.append("<dt>").append(target).append("</dt><dd>").append(target.getToolTip()).append("</dd>");
                    }
                    return sb.append("</dl></html>").toString();
                case C_METHOD:
                    sb = new StringBuilder("<html>Operation to be applied to the target.<dl>");
                    for (Method method : Method.values()) {
                        sb.append("<dt>").append(method).append("</dt><dd>").append(method.getToolTip()).append("</dd>");
                    }
                    return sb.append("</dl></html>").toString();                    
                case C_CODE:
                    return "Operation parameters (regex, pattern, template definition, etc.)";
                case C_COLOR:
                    return "Color to be assigned to messages that satisfy this filter.";
                case C_FLAG:
                    return "Flag to be added to messages that satisfy this filter.";
                default:
                    return null;
            }

        }
        
    }
    
    
// -- FILTER table cell editors : ----------------------------------------------
    
    private class ColorRenderer extends JLabel implements TableCellRenderer {
        Border unselectedBorder = null;
        Border selectedBorder = null;
        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            if (isSelected) {
                if (selectedBorder == null) {
                    selectedBorder = BorderFactory.createMatteBorder(2,3,2,3, table.getSelectionBackground());
                }
                setBorder(selectedBorder);
            } else {
                if (unselectedBorder == null) {
                    unselectedBorder = BorderFactory.createMatteBorder(2,3,2,3, table.getBackground());
                }
                setBorder(unselectedBorder);
            }
            if (value != null) {
                setOpaque(true);
                setBackground((Color) value);
                setText("    ");
             } else {
                setOpaque(false);
                setText("None");
            }
            return this;
        }
    }
    
    private class ColorEditor extends AbstractCellEditor implements TableCellEditor {
        
        private final ColorComboBox combo;
        
        ColorEditor() {
            combo = new ColorComboBox(
                    new Color[]{null, Color.RED, Color.GREEN, Color.BLUE, Color.BLACK},
                    new String[]{"None", "Red", "Green", "Blue", "Black"},
                    true);
        }

        @Override
        public Color getCellEditorValue() {
            return combo.getSelectedColor();
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            combo.setSelectedColor((Color)value);
            return combo;
        }
        
    }
    
    private class CodeRenderer extends DefaultTableCellRenderer {
        private Color defSelected, defUnselected;
        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            if (value == null) value = new String[] {""};
            Step step = TracerEditor.this.model.rows.get(row);
            String label;
            if (step.method == Method.NAME) {
                label = step.code[0];
                if (step.code.length > 1) {
                    label += " ...";
                }
//                int i = label.lastIndexOf("/");
//                if (i != -1) {
//                    label = label.substring(i + 1);
//                }
            } else if (step.target == Target.TEMPLATE) {
                if (step.code.length == 2) {
                    label = step.code[0] +" <=> "+ step.code[1];
                    if (label.length() > 30) {
                        label = "... <=> ...";
                    }
                } else {
                    label = "";
                }
            } else {
                label = step.code[0];
            }
            if (defSelected == null) { // memorize default colors to work around color memory issue
                super.getTableCellRendererComponent(table, label, true, hasFocus, row, column);
                defSelected = getBackground();
                super.getTableCellRendererComponent(table, label, false, hasFocus, row, column);
                defUnselected = getBackground();
            }
            super.getTableCellRendererComponent(table, label, isSelected, hasFocus, row, column);
            if (step.valid) {
                setBackground(isSelected ? defSelected : defUnselected);
            } else {
                setBackground(Color.RED);
            }
            return this;
        }
    }

    private class CodeEditor extends AbstractCellEditor implements TableCellEditor {
        
        Step step;
        private final CodeRenderer renderer = new CodeRenderer();
        
        CodeEditor() {
            renderer.addMouseListener(new MouseAdapter() {
                @Override
                public void mouseReleased(MouseEvent e) {
                    if (e.getClickCount() == 1) {
                        editCode(step, renderer);
                        fireEditingStopped();
                    }
                }
            });
        }

        @Override
        public String[] getCellEditorValue() {
            return step.code;
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            step = TracerEditor.this.model.rows.get(row);
            return renderer.getTableCellRendererComponent(table, value, true, true, row, column);
        }
                
    }
    
}
