package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.*;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.MenuElement;
import javax.swing.SwingConstants;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.plugins.trending.TrendingService;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.gconsole.services.lock.Lock;
import org.lsst.ccs.gconsole.util.swing.BoxedTextField;

/**
 * Skeleton implementation of a table that displays monitored values, and its model.
 * <p>
 * Functionality:<ul>
 * <li>Data storage ({@code ArrayList<ArrayList<MonitorCell>>}) and simple implementation of {@code TableModel}.
 * <li>Implements {@code Updatable}. {@code update(...)} uses {@code getCells(displayChannel, monitorField)} to
 *     find affected cells, computes new {@code FormattedValue} for each of them (firing table model event if
 *     necessary), then fires CELLS_EVENT.
 * <li>Allows registration of a {@code Listener} that is notified of CELLS_EVENT or TABLE_EVENT.
 * <li>Provides customized {@code JTable} (rendering, mouse clicks, etc.)
 * <li>Utility for trimming absent fields.
 * <li>DEFAULT_FORMAT, DEFAULT_FIELDS, DEFAULT_COMPACT_FIELDS.
 * </ul>
 *
 * @author onoprien
 */
abstract public class MonitorTable extends AbstractTableModel implements Updatable {

// -- Fields : -----------------------------------------------------------------
    
    /** Stateless default {@code MonitorFormat} instance. */
    static public final MonitorFormat DEFAULT_FORMAT = new MonitorFormat();

    /** Default list of fields to be displayed by monitoring tables in full mode. */
    static public final List<MonitorField> DEFAULT_FIELDS = Collections.unmodifiableList(Arrays.asList(MonitorField.VALUE, MonitorField.UNITS, MonitorField.ALERT_LOW, MonitorField.LOW_ALARM, MonitorField.LOW_WARN, MonitorField.HIGH_WARN, MonitorField.HIGH_ALARM, MonitorField.ALERT_HIGH, MonitorField.DESCR));
    /** Default list of fields to be displayed by monitoring tables in compact mode. */
    static public final List<MonitorField> DEFAULT_COMPACT_FIELDS = Collections.unmodifiableList(Arrays.asList(MonitorField.VALUE, MonitorField.DESCR));

    protected int nRows, nColumns;
    protected ArrayList<ArrayList<MonitorCell>> cells; // [row][column], covers actual table
    protected MonitorFormat format = DEFAULT_FORMAT;
    protected boolean limitMaxWidth = false;
    
    protected Listener listener;
    
    protected final Event CELLS_EVENT = new Event(this, Event.Reason.CELLS);
    protected final Event TABLE_EVENT = new Event(this, Event.Reason.TABLE);

// -- Life cycle : -------------------------------------------------------------
    
    /** Empty default constructor. */
    protected MonitorTable() {
    }
    
    /** Constructs the table with the specified content. */
    protected MonitorTable(MonitorCell[][] cells) {
        nRows = cells.length;
        nColumns = cells[0].length;
        this.cells = new ArrayList<>(nRows);
        for (MonitorCell[] row : cells) {
            this.cells.add(new ArrayList<>(Arrays.asList(row)));
        }
    }
    
    /** Called when the table is discarded. */
    public void destroy() {
    }
    
    
// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the formatter to be used by this table.
     * @param format Formatter.
     */
    public void setFormat(MonitorFormat format) {
        this.format = format;
        if (cells != null) {
            boolean change = false;
            for (ArrayList<MonitorCell> row : cells) {
                for (MonitorCell cell : row) {
                    if (cell != null && cell.getField() != null && cell.getField().isUpdatable() && !cell.getField().equals(MonitorField.NULL)) {
                        change = format.format(cell);
                    }
                }
            }
            if (change) fireChangeEvent(CELLS_EVENT);
        }
    }
    
    /**
     * Sets {@code limitMaxWidth} property.
     * @param limitMaxWidth If true, the table max width will be limited to its preferred width.
     */
    public void setLimitMaxWidth(boolean limitMaxWidth) {
        this.limitMaxWidth = limitMaxWidth;
    }


// -- Implementing TableModel : ------------------------------------------------

    @Override
    public int getRowCount() {
        return nRows;
    }

    @Override
    public int getColumnCount() {
        return nColumns;
    }

    @Override
    public FormattedValue getValueAt(int row, int column) {
        return cells.get(row).get(column).getFormattedValue();
    }
    
    
// -- Updating table model : ---------------------------------------------------

    @Override
    public void update(DisplayChannel item, List<MonitorField> fields) {
        fields.forEach(field -> {
            List<int[]> affectedCells = getCells(item, field);
            for (int[] index : affectedCells) {
                MonitorCell cell = cells.get(index[0]).get(index[1]);
                field = cell.getField();
                FormattedValue data = format.format(field, item);
                if (!data.equals(cell.getFormattedValue())) {
                    cell.setFormattedValue(data);
                    fireTableCellUpdated(index[0], index[1]);
                }
            }
        });
        fireChangeEvent(CELLS_EVENT);
    }

    @Override
    public void update(DisplayChannel channelHandle) {
        update(channelHandle, Collections.singletonList(null));
    }

    /**
     * Returns indices of all cells whose content is affected by the specified item and field.
     * The implementation provided by this class is universal but very inefficient - subclasses
     * handling specific types of tables are expected to override it.
     * 
     * @param item DisplayChannel.
     * @param field Field. If {@code null}, all cells affected by the specified item are included.
     * @return List of cells.
     */
    protected List<int[]> getCells(DisplayChannel item, MonitorField field) {
        List<int[]> out = new ArrayList<>();
        for (int row = 0; row < nRows; row++) {
            for (int col = 0; col < nColumns; col++) {
                MonitorCell cell = cells.get(row).get(col);
                if (cell.getChannels().contains(item) && (field == null || field.equals(cell.getField()))) {
                    out.add(new int[] {row, col});
                }
            }
        }
        return out;
    }
    
    
// -- Handling change listener : -----------------------------------------------
    
    public void setListener(Listener listener) {
        this.listener = listener;
    }
    
    protected void fireChangeEvent(Event.Reason reason) {
        fireChangeEvent(reason == Event.Reason.CELLS ? CELLS_EVENT : TABLE_EVENT);
    }
    
    protected void fireChangeEvent(Event event) {
        if (listener != null) listener.stateChanged(event);
    }
    
    static public class Event extends EventObject {
        
        public enum Reason {CELLS, TABLE} 
        
        private final Reason reason;
        
        public Event(MonitorTable source, Reason reason) {
            super(source);
            this.reason = reason;
        }

        @Override
        public MonitorTable getSource() {
            return (MonitorTable) super.getSource();
        }
        
        public Reason getReason() {
            return reason;
        }
        
    }
    
    public interface Listener {

        public void stateChanged(Event e);
        
    }
    
    
// -- Customized JTable : ------------------------------------------------------
    
    public JTable getTable() {
        
        JTable table = makeTable();
        
        JTableHeader header = table.getTableHeader();
        header.setReorderingAllowed(false);
        header.setResizingAllowed(false); // no resizing to avoid mismatch between renderer and editor tables
        TableCellRenderer headerRenderer = header.getDefaultRenderer();
        if (headerRenderer instanceof DefaultTableCellRenderer) {
            ((DefaultTableCellRenderer)headerRenderer).setHorizontalAlignment(SwingConstants.CENTER);
        }

        MonitorTableCellRenderer renderer = new MonitorTableCellRenderer();
        renderer.setEnsureWidth(true);
        table.setDefaultRenderer(Object.class, renderer);
        table.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
        table.setRowSelectionAllowed(false);
        table.setColumnSelectionAllowed(false);
        table.setRowHeight(table.getRowHeight() + 2);
        table.setShowGrid(true);
        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent evt) {
                tableMouseClicked(evt);
            }
        });

        for (int col = 0; col < nColumns; col++) {
            TableColumn column = table.getColumnModel().getColumn(col);
            Component comp = headerRenderer.getTableCellRendererComponent(null, column.getHeaderValue(), false, false, 0, 0);
            int maxWidth = comp.getPreferredSize().width;
            for (int row=0; row<nRows; row++) {
                comp = renderer.getTableCellRendererComponent(table, getValueAt(row, col), false, false, row, col);
                maxWidth = Math.max(maxWidth, comp.getPreferredSize().width);
            }
            column.setPreferredWidth(maxWidth);
            if (limitMaxWidth) column.setMaxWidth(maxWidth);
        }
        return table;
    }
    
    protected JTable makeTable() {
        return new JTable(this);
    }
    
    protected void tableMouseClicked(MouseEvent evt) {
        int nClick = evt.getClickCount();
        JTable table = (JTable) evt.getSource();
        Point point = evt.getPoint();
        int row = table.rowAtPoint(point);
        int column = table.columnAtPoint(point);
        MonitorCell cc = cells.get(row).get(column);
        FormattedValue fv = cc.getFormattedValue();
        if (nClick == 1) {
            if (fv != null && fv.getClick1() != null) {
                fv.getClick1().accept(cc);
            }
        } else if (nClick == 2) {
            if (fv != null && fv.value instanceof AgentLock) {
                editLock((AgentLock)fv.value, table);
//                Lock lockLabel = new Lock(((AgentLock)fv.value).getAgentName());
//                JPanel panel = new JPanel();
//                panel.add(lockLabel);
//                panel.add(new JLabel("Right-click icon to modify the lock."));
//                JOptionPane.showMessageDialog(table, panel, "Lock / Unlock", JOptionPane.PLAIN_MESSAGE);
                return;
            }
            MonitorField field = cc.getField();
            List<DisplayChannel> items = cc.getChannels();
            if (field != null && items.size() == 1) {
                AgentChannel channel = items.get(0).getChannel();
                if (channel != null) {
                    Object published = field.getValue(channel);
                    if (published instanceof ConfigurationParameterInfo) { // edit configuration parameter
                        ConfigurationParameterInfo conf = (ConfigurationParameterInfo) published;
                        if (!(conf.isFinal() || conf.isReadOnly() || conf.isBuild())) {
                            edit(evt.getComponent(), conf, field, channel);
                        }
                    } else if (MonitorField.VALUE.equals(field)) { // show trending plot
                        String trendingKey = channel.get(AgentChannel.Key.TRENDING);
                        if (trendingKey == null) {
                            trendingKey = channel.getLocalPath();
                        }
                        String[] path = {channel.getAgentName(), trendingKey};
                        TrendingService trending = (TrendingService) Console.getConsole().getConsoleLookup().lookup(TrendingService.class);
                        if (trending == null) {
                            return;
                        }
                        trending.show(path);
                    }
                }
            }
        }
    }
    
    public boolean showHeader() {
        return true;
    }
    
    private void editLock(AgentLock agentLock, JTable table) {
                Lock lock = new Lock(agentLock.getAgentName());
                JPanel panel = new JPanel();
                panel.setLayout(new GridLayout(1, 2));
                Box box = Box.createHorizontalBox();
                box.setBorder(BorderFactory.createCompoundBorder(
//                        BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE),
                        BorderFactory.createEmptyBorder(0, Const.HSPACE, 0, Const.HSPACE),
                        BorderFactory.createEtchedBorder()));
                box.add(Box.createHorizontalGlue());
                box.add(lock);
                box.add(Box.createHorizontalGlue());
                panel.add(box);
                JPanel buttonPanel = new JPanel();
                buttonPanel.setLayout(new GridLayout(3, 1));
                lock.addListener(l -> {
                    buttonPanel.removeAll();
                    JPopupMenu menu = lock.getComponentPopupMenu();
                    ArrayList<Action> actions = new ArrayList<>(3);
                    for (MenuElement e : menu.getSubElements()) {
                        if (e instanceof JMenuItem) {
                            actions.add(((JMenuItem)e).getAction());
                        }
                    }
                    actions.forEach(a -> {
                        buttonPanel.add(new JButton(a));
                    });
                    panel.revalidate();
                    panel.repaint();
                });
                panel.add(buttonPanel);
                JOptionPane.showOptionDialog(table, panel, "Lock / Unlock", JOptionPane.DEFAULT_OPTION, JOptionPane.PLAIN_MESSAGE, null, new Object[] {"Done"}, "Done");
//                JOptionPane.showMessageDialog(table, panel, "Lock / Unlock", JOptionPane.PLAIN_MESSAGE);        
    }
    
    
// -- Static utilities : -------------------------------------------------------
    
    static public List<MonitorField> trimAbsentFields(List<MonitorField> fields, Collection<DisplayChannel> channels, Collection<MonitorField> exclude) {
        if (fields == null) fields = Collections.emptyList();
        LinkedHashSet<MonitorField> requestedFields = new LinkedHashSet<>(fields);
        if (exclude != null) {
            requestedFields.removeAll(exclude);
        }
        HashSet<MonitorField> presentFields = new HashSet<>();
        if (requestedFields.remove(MonitorField.VALUE)) {
            presentFields.add(MonitorField.VALUE);
        }
        for (DisplayChannel ch : channels) {
            for (AgentChannel channel : ch.getChannels()) {
                Iterator<MonitorField> it = requestedFields.iterator();
                while (it.hasNext()) {
                    MonitorField f = it.next();
                    if (f.getValue(channel) != null) {
                        presentFields.add(f);
                        it.remove();
                    }
                }
                if (requestedFields.isEmpty()) break;
            }
        }
        if (fields.size() == presentFields.size()) {
            return fields;
        } else {
            if (presentFields.contains(MonitorField.HIGH_WARN) || presentFields.contains(MonitorField.HIGH_ALARM)) {
                presentFields.add(MonitorField.ALERT_HIGH);
            }
            if (presentFields.contains(MonitorField.LOW_WARN) || presentFields.contains(MonitorField.LOW_ALARM)) {
                presentFields.add(MonitorField.ALERT_LOW);
            }
            ArrayList<MonitorField> out = new ArrayList<>(presentFields.size());
            for (MonitorField f : fields) {
                if (presentFields.contains(f)) {
                    out.add(f);
                }
            }
            return out;
        }
    }
    
    static public void edit(Component parent, ConfigurationParameterInfo conf, MonitorField field, AgentChannel channel) {
        JPanel root = new JPanel(new BorderLayout());
        StringBuilder sb = new StringBuilder("<html>");
        sb.append("Subsystem: <b>").append(channel.getAgentName()).append("</b>.<br>");
        sb.append("Component: <b>").append(conf.getComponentName()).append("</b>.<br>");
        
        sb.append("Parameter: <b>").append(conf.getParameterName()).append("</b><p>");
        sb.append("Category: <b>").append(conf.getCategoryName()).append("</b><br>");
        String s = conf.getDescription();
        if (s != null && !s.trim().isEmpty()) {
            sb.append("Description: ").append(s).append("<p>");
        }
        sb.append("Current value: <b>").append(conf.getCurrentValue()).append("</b><br>");
        sb.append("Configured value: <b>").append(conf.getConfiguredValue()).append("</b><p>");
        
        sb.append("&nbsp; <p></html>");
        JEditorPane info = new JEditorPane("text/html", sb.toString());
        info.setEditable(false);
        info.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEtchedBorder(), BorderFactory.createEmptyBorder(Const.VSPACE, 0, Const.VSPACE, 0)));
        info.setBackground(new Color(255, 255, 255, 0));
        root.add(info, BorderLayout.CENTER);
        Box input = Box.createHorizontalBox();
        root.add(input, BorderLayout.SOUTH);
        input.add(new JLabel("Enter new value: "));
        JTextField textField = new BoxedTextField(30);
        input.add(textField);
        input.add(Box.createHorizontalGlue());
        int out = JOptionPane.showOptionDialog(parent, root, "Change configuration parameter", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE, null, null, null);
        if (out == JOptionPane.OK_OPTION) {
            String value = textField.getText();
            value = value.trim(); // FIXME: this is a workaround to strip padding that might have been added by formatters
            String component = conf.getComponentName();
            CommandService.getService().sendEncoded(channel.getAgentName(), "change", component, conf.getParameterName(), value);
        }
    }
    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * Returns JavaBean that contains data required to restore this table to its current state.
     * The default implementation provided by this class returns {@code null}, subclasses should override
     * if they need to support saving/restoring their state.
     * 
     * @return JavaBean that contains data required to restore this table to its current state.
     */
    public Serializable save() {
        return null;
    }
    
    /**
     * Restores this table to the state described by the provided object, to the extent possible.
     * The default implementation provided by this class does nothing.
     * 
     * @param storageBean JavaBean that contains data describing the state of this table.
     */
    public void restore(Serializable storageBean) {
    }

}
