package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Color;
import java.math.BigInteger;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.swing.SwingConstants;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;

/**
 * Descriptor of a field that can be displayed by monitoring views.
 * Static constants of this class enumerate standard types of monitored values.
 * Additional instances may be created at run time.
 * <p>
 * Instances can be registered as default handlers for data channel attribute keys.
 * <p>
 * Equality and hash code are based on the object identity.
 *
 * @author onoprien
 */
public class MonitorField {
    
// -- Static public constants : ------------------------------------------------
    
    static public final Color COLOR_FG = Color.BLACK; // default foreground
    static public final Color COLOR_BG = Color.WHITE; // default background
    
    static public final Color COLOR_CHANGED = Color.BLUE;
    
    static public final Color COLOR_GOOD = new Color(160, 255, 160);
    static public final Color COLOR_WARN = new Color(255, 255, 100);
    static public final Color COLOR_ERR = new Color(255, 160, 160);
    static public final Color COLOR_DISABLED = new Color(255, 200, 255);
    static public final Color COLOR_OFF = new Color(160, 200, 255);
    static public final Color COLOR_POPUP = new Color(255, 255, 160);
    static public final Color COLOR_MULTI = Color.LIGHT_GRAY;
    static public final Color COLOR_NA = COLOR_BG;
    
    static public final EnumMap<DataProviderState,Color> COLOR_STATE = new EnumMap(DataProviderState.class);
    static {
        COLOR_STATE.put(DataProviderState.OFF_LINE, COLOR_OFF);
        COLOR_STATE.put(DataProviderState.DISABLED, COLOR_DISABLED);
        COLOR_STATE.put(DataProviderState.NOT_VALIDATED, COLOR_BG);
        COLOR_STATE.put(DataProviderState.NOMINAL, COLOR_GOOD);
        COLOR_STATE.put(DataProviderState.WARNING, COLOR_WARN);
        COLOR_STATE.put(DataProviderState.ALARM, COLOR_ERR);
    }
    
    static public final String TEXT_MULTI = "---";
    static public final int HA_MULTI = SwingConstants.CENTER;
    
// -- Static privates : --------------------------------------------------------
    
    static private final HashMap<String,MonitorField> fields = new HashMap<>();
    static private final Pattern CORE_STATE_PATTERN = Pattern.compile("[^/]+/state/([^/]+)");
    static private final Pattern FORMAT_CONVERSION_PATTERN = Pattern.compile("(?:\\d+\\$)?[^ ]*?\\d*(?:\\.\\d*)?([a-zA-Z]{1,2})(?: .*)?");
    
// -- Pre-defined standard MonitorField constants : ----------------------------
    
    static public final MonitorField NULL = new MonitorField(null, "", null);
    
    static public final MonitorField NAME = new MonitorField("name", "Name", null);
    static {registerInstance(NAME);}
    
    static public final MonitorField VALUE = new Value();
    static {registerInstance(VALUE);}
    
    static public final MonitorField UNITS = new MonitorField(AgentChannel.Key.UNITS, "Units", new FormattedValue(null, null, null, SwingConstants.LEFT, null, false));
    static {registerInstance(UNITS);}
    
    static public final MonitorField LOW_ALARM = new Limit(AgentChannel.Key.LOW_ALARM, "Low Limit");
    static {registerInstance(LOW_ALARM);}
    
    static public final MonitorField LOW_WARN = new Limit(AgentChannel.Key.LOW_WARN, "Low Warn");
    static {registerInstance(LOW_WARN);}
    
    static public final MonitorField ALERT_LOW = new Alert(AgentChannel.Key.ALERT_LOW);
    static {registerInstance(ALERT_LOW);}

    static public final MonitorField HIGH_ALARM = new Limit(AgentChannel.Key.HIGH_ALARM, "High Limit");
    static {registerInstance(HIGH_ALARM);}
    
    static public final MonitorField HIGH_WARN = new Limit(AgentChannel.Key.HIGH_WARN, "High Warn");
    static {registerInstance(HIGH_WARN);}
    
    static public final MonitorField ALERT_HIGH = new Alert(AgentChannel.Key.ALERT_HIGH);
    static {registerInstance(ALERT_HIGH);}
    
    static public final MonitorField DESCR = new MonitorField("description", "Description", new FormattedValue(null, null, null, SwingConstants.LEFT, null, false));
    static {registerInstance(DESCR);}
    
    static public final MonitorField AVERAGE_VALUE = new StateColor(AgentChannel.Key.VALUE, 
            "Average", VALUE.template, Collections.unmodifiableSet(new HashSet(Arrays.asList(new String[] {AgentChannel.Key.VALUE, AgentChannel.Key.STATE})))) {
        @Override
        public FormattedValue format(List<AgentChannel> channels) {
            if (channels.isEmpty()) return FormattedValue.EMPTY;
            FormattedValue fv = super.format(channels);
            int n = 0;
            double sum = 0.;
            for (AgentChannel ch : channels) {
                if (ch != null) {
                    try {
                        Double value = ch.get(key);
                        if (!(value == null || value.isInfinite() || value.isNaN())) {
                            n++;
                            sum += value;
                        }
                    } catch (ClassCastException x) {
                    }
                }
            }
            if (n > 0) {
                sum /= n;
                if (fv.format == null) {
                    fv.text = String.format(DEFAULT_FLOAT_FORMAT, sum);
                } else {
                    fv.text = String.format(fv.format, sum, n);
                }
            } else {
                fv.text = "";
            }
            return fv;
        }
    };

    static public final MonitorField MAX_VALUE = new StateColor(AgentChannel.Key.VALUE, 
            "Max", VALUE.template, Collections.unmodifiableSet(new HashSet(Arrays.asList(new String[] {AgentChannel.Key.VALUE, AgentChannel.Key.STATE})))) {
        @Override
        public FormattedValue format(List<AgentChannel> channels) {
            if (channels.isEmpty()) return FormattedValue.EMPTY;
            FormattedValue fv = super.format(channels);
            int n = 0;
            double sum = Double.NEGATIVE_INFINITY;
            for (AgentChannel ch : channels) {
                if (ch != null) {
                    try {
                        Double value = ch.get(key);
                        if (!(value == null || value.isInfinite() || value.isNaN())) {
                            n++;
                            sum = Math.max(sum, value);
                        }
                    } catch (ClassCastException x) {
                    }
                }
            }
            if (n > 0) {
                if (fv.format == null) {
                    fv.text = String.format(DEFAULT_FLOAT_FORMAT, sum);
                } else {
                    fv.text = String.format(fv.format, sum, n);
                }
            } else {
                fv.text = "";
            }
            return fv;
        }
    };

    static public final MonitorField MIN_VALUE = new StateColor(AgentChannel.Key.VALUE, 
            "Min", VALUE.template, Collections.unmodifiableSet(new HashSet(Arrays.asList(new String[] {AgentChannel.Key.VALUE, AgentChannel.Key.STATE})))) {
        @Override
        public FormattedValue format(List<AgentChannel> channels) {
            if (channels.isEmpty()) return FormattedValue.EMPTY;
            FormattedValue fv = super.format(channels);
            int n = 0;
            double sum = Double.POSITIVE_INFINITY;
            for (AgentChannel ch : channels) {
                if (ch != null) {
                    try {
                        Double value = ch.get(key);
                        if (!(value == null || value.isInfinite() || value.isNaN())) {
                            n++;
                            sum = Math.min(sum, value);
                        }
                    } catch (ClassCastException x) {
                    }
                }
            }
            if (n > 0) {
                if (fv.format == null) {
                    fv.text = String.format(DEFAULT_FLOAT_FORMAT, sum);
                } else {
                    fv.text = String.format(fv.format, sum, n);
                }
            } else {
                fv.text = "";
            }
            return fv;
        }
    };

    static public final MonitorField MEDIAN_VALUE = new StateColor(AgentChannel.Key.VALUE, 
            "Median", VALUE.template, Collections.unmodifiableSet(new HashSet(Arrays.asList(new String[] {AgentChannel.Key.VALUE, AgentChannel.Key.STATE})))) {
        @Override
        public FormattedValue format(List<AgentChannel> channels) {
            if (channels.isEmpty()) return FormattedValue.EMPTY;
            FormattedValue fv = super.format(channels);
            int n = 0;
            double[] vv = new double[channels.size()];
            for (AgentChannel ch : channels) {
                if (ch != null) {
                    try {
                        Double value = ch.get(key);
                        if (!(value == null || value.isInfinite() || value.isNaN())) {
                            vv[n++] = value;
                        }
                    } catch (ClassCastException x) {
                    }
                }
            }
            if (n > 0) {
                Arrays.sort(vv);
                double m;
                if (n % 2 == 0) {
                    m = (vv[n/2]+vv[n/2-1])/2.;
                } else {
                    m = vv[n/2];
                }
                if (fv.format == null) {
                    fv.text = String.format(DEFAULT_FLOAT_FORMAT, m);
                } else {
                    fv.text = String.format(fv.format, m, n);
                }
            } else {
                fv.text = "";
            }
            return fv;
        }
    };
    
    static public final MonitorField HOST = new MonitorField("org.lsst.ccs.agent.hostname", "Host", new FormattedValue(null, null, null, SwingConstants.CENTER, null, false));
    static {registerInstance(HOST);}

// -- Defaults to be used by views : -------------------------------------------
    
    /** Default {@code MonitorField} to be used for displaying a value corresponding to a group of display channels. */
    static public final MonitorField DEFAULT_GROUP_FIELD = AVERAGE_VALUE;
    
    /** Available standard {@code MonitorField} constants for displaying a value corresponding to a group of display channels. */
    static public final MonitorField[] GROUP_FIELDS = {
        AVERAGE_VALUE, 
        MAX_VALUE, 
        MIN_VALUE, 
        MEDIAN_VALUE
    };

    
// -- Fields : -----------------------------------------------------------------
    
    protected String DEFAULT_FLOAT_FORMAT = "%14.5f"; // FIXME: "%.2f" ?
    protected String DEFAULT_INT_FORMAT = "%10d";
    protected String DEFAULT_DT_FORMAT = "%1$tF %1$tT";
    protected int LENGTH_LIMIT = 40;
    
    protected final String key;
    protected final String title;
    protected final FormattedValue template;
    protected final Set<String> keys;

// -- Life cycle and registration : --------------------------------------------
    
    /**
     * Constructs an instance.
     * 
     * @param key Attribute key. Unless overridden, {@code getKey()} will return this key. The {@code getValue(channel)}
     *            method will return the value of the data channel attribute identified by this key.
     * @param title Human readable title.
     * @param template Formatted value template.
     * @param keys Keys of attributes that might affect this field. If {@code null}, the field is assumed to be affected
     *             by the attribute specified by the {@code key} argument. If that argument is {@code null} as well,
     *             the field is not affected by any parameters.
     */
    public MonitorField(String key, String title, FormattedValue template, Set<String> keys) {
        this.key = key;
        this.title = title;
        this.template = template == null ? new FormattedValue() : new FormattedValue(template);
        this.keys = keys == null? (key == null ? Collections.emptySet() : Collections.singleton(key)) : keys;
    }
    
    /**
     * Constructs an instance without specifying the set of affecting keys.
     * Equivalent to {@code MonitorField(key, title, template, null)}.
     * 
     * @param key Attribute key. Unless overridden, {@code getKey()} and {@code getKeys()}
     *            methods will return this key and a singleton set containing this key, respectively.
     *            The {@code getValue(channel)} method will return the value of the data channel attribute
     *            identified by this key.
     * @param title Human readable title.
     * @param template Formatted value template.
     */
    public MonitorField(String key, String title, FormattedValue template) {
        this(key, title, template, null);
    }
    
    /**
     * Deep copy constructor.
     * @param other Field to be cloned.
     */
    public MonitorField(MonitorField other) {
        this.key = other.key;
        this.title = other.title;
        this.template = other.template;
        this.keys = other.keys;
    }
    
    /**
     * Registers the specified instance as the default handler for the attribute
     * identified by the key returned by its {@code getKey()} method.
     * 
     * @param field Instance to register.
     * @throws IllegalArgumentException if an instance with this {@code key} already registered.
     */
    static public synchronized void registerInstance(MonitorField field) {
        String key = field.getKey();
        if (fields.containsKey(key)) throw new IllegalArgumentException("Field with key "+ key +" is already registered.");
        fields.put(key, field);
    }
    
    /**
     * Returns an instance identified by the specified string.
     * If a default handler for the specified key is already registered, it is returned;
     * otherwise, if a standard field with the specified name (static field in this class) exists, it is returned;
     * otherwise, a new instance is created with default parameters:<br>
     * {@code new MonitorField(key, key, null)}, and registered as the default handler for the key.
     * 
     * @param key Attribute key.
     * @return Registered instance for the specified key.
     */
    static public synchronized MonitorField getInstance(String key) {
        MonitorField field = getRegisteredInstance(key);
        if (field == null) {
            try {
                if (key.startsWith("MonitorField.")) {
                    key = key.substring("MonitorField.".length());
                }
                field =  (MonitorField) MonitorField.class.getField(key).get(null);
            } catch (NoSuchFieldException | SecurityException | IllegalArgumentException | IllegalAccessException | NullPointerException x) {
            }
        }
        if (field == null) {
            field = new MonitorField(key, key, null);
            registerInstance(field);
        }
        return field;
    }
    
    
// -- Statis utilities : -------------------------------------------------------
    
    /**
     * Returns an instance registered as the default handler for the attribute identified
     * by the specified key, or {@code null} if no such instance exists.
     * 
     * @param key Attribute key.
     * @return Registered instance, or {@code null} if no such instance exists.
     */
    static public synchronized MonitorField getRegisteredInstance(String key) {
        return fields.get(key);
    }
    
    static public synchronized List<MonitorField> getDefaultFields(Collection<String> keys) {
        return keys.stream().map(k -> getRegisteredInstance(k)).filter(f -> f != null).collect(Collectors.toList());
    }
    
    static public List<MonitorField> getAffectedFields(Collection<String> keys, Collection<MonitorField> fields) {
        if (keys == null) return null;
        return fields.stream().filter(field -> {
            Set<String> att = field.getKeys();
            return keys.stream().anyMatch(k -> att.contains(k));
        }).collect(Collectors.toList());
    }
    
    static public List<MonitorField> getAffectedFields(Collection<String> keys) {
        if (keys == null) return null;
        synchronized (MonitorField.class) {
            return getAffectedFields(keys, fields.values());
        }
    }
    
    
// -- Retrieving or modifying current value : ----------------------------------
    
    /**
     * Returns the current value of this field in the specified data channel, or {@code null}
     * if the channel does not contain a value for this field.
     * 
     * @param channel Data channel.
     * @return Current value of this field.
     */
    public Object getValue(AgentChannel channel) {
        return (channel == null || key == null) ? null : channel.get(key);
    }

    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns a unique key of this field.
     * @return Name of this field.
     */
    public String getKey() {
        return key;
    }
    
    /**
     * Returns human readable title associated with this field.
     * @return Title of this field.
     */
    public String getTitle() {
        return title;
    }
    
    /**
     * Returns a set of keys of data channel attributes that might effect the value of this field or its formatting.
     * For fields that are not updated throughout lifetime of a subsystem, this method returns an empty set.
     * 
     * @return Set of attribute keys that might affect this field.
     */
    public Set<String> getKeys() {
        return keys;
    }
    
    public boolean isUpdatable() {
        return !getKeys().isEmpty();
    }
    
    
// -- Formatting : -------------------------------------------------------------

    /**
     * Computes the current formatted value of this field in the specified data channel.
     * Implementation provided by this class clones {@code template} and passes it to {@code defaultFormat(...)}.
     * Subclasses may add processing before or after the call to {@code defaultFormat(...)} (or override this method entirely).
     * 
     * @param channel Data channel.
     * @return Formatted value of this field.
     */
    public FormattedValue format(AgentChannel channel) {
        FormattedValue fv = new FormattedValue(template);
        fv = defaultFormat(channel, fv);
        return fv;
    }

    /**
     * Computes the current formatted value of this field for the specified list of channels.
     * 
     * @param channels List of channels.
     * @return Formatted value ({@code FormattedValue.NA} if this field does not support formatting multiple channels).
     */
    public FormattedValue format(List<AgentChannel> channels) {
        List<FormattedValue> fvList = channels.stream().map(ch -> format(ch)).collect(Collectors.toList());
        FormattedValue fv = merge(fvList);
        return fv;
    }

    /**
     * Applies standard formatting, without overwriting fields already set on the provided {@code FormattedValue}.
     * 
     * @param channel Data channel.
     * @param fv Partially formatted value.
     * @return Formatted value.
     */
    protected FormattedValue defaultFormat(AgentChannel channel, FormattedValue fv) {
        
        // Set value
        
        if (channel == null) return FormattedValue.NA;
        if (fv.value == null) {
            fv.value = getValue(channel);
            if (fv.value == null) return FormattedValue.NA;
        }
        
        // Set background color:
        if (channel.getAgent().hasAgentProperty(AgentInfo.AGENT_OFFLINE_PROP) && keys.contains(AgentChannel.Key.STATE)) {
            fv.bgColor = COLOR_OFF;
        }
        if (fv.value instanceof ConfigurationParameterInfo) {
            colorConfigurationParameter(fv, channel);
        }
        colorByChannelState(fv, channel);
        
        // Format raw value:
        
        if (fv.text == null) {
            
            // if format is known, try to use it
            
            if (fv.format != null) {
                
                char conv = getFormatConversion(fv.format).charAt(0);
                try {
                    switch (conv) {
                        case 'd': case 'o': case 'x': case 'X':  // integer
                            if (!(fv.value instanceof Integer || fv.value instanceof Long || fv.value instanceof Short || fv.value instanceof Byte || fv.value instanceof BigInteger)) {
                                fv.value = Long.parseLong(fv.value.toString());
                            }
                            break;
                        case 'f': case 'e': case 'E': case 'g': case 'G':
                            if (!(fv.value instanceof Double || fv.value instanceof Float || fv.value instanceof BigInteger)) {
                                fv.value = Double.parseDouble(fv.value.toString());
                            }
                            break;
                    }
                    fv.text = String.format(fv.format, fv.value);
                } catch (NumberFormatException | IllegalFormatException x) {
                }
            }
            
            // otherwise, if a number, try to use default format
            
            if (fv.text == null) {
                if (fv.value instanceof Double || fv.value instanceof Float) {
                    try {
                        fv.text = String.format(DEFAULT_FLOAT_FORMAT, fv.value);
                    } catch (IllegalFormatException x) {
                    }
                } else if (fv.value instanceof Integer) {
                    try {
                        fv.text = String.format(DEFAULT_INT_FORMAT, fv.value);
                    } catch (IllegalFormatException x) {
                    }
                } else if (fv.value instanceof Instant) {
                    try {
                        fv.text = String.format(DEFAULT_DT_FORMAT, ZonedDateTime.ofInstant((Instant)fv.value, ZoneId.systemDefault()));
                    } catch (IllegalFormatException | DateTimeException x) {
                    }
                } else if (fv.value instanceof ZonedDateTime) {
                    try {
                        fv.text = String.format(DEFAULT_DT_FORMAT, fv.value);
                    } catch (IllegalFormatException x) {
                    }
                }
            }

            // otherwise, convert to String:
            
            if (fv.text == null) {
                fv.text = fv.value.toString();
            }
            
            // limit text length:
            
            if (fv.text.length() > LENGTH_LIMIT) {
                if (fv.toolTip == null) {
                    fv.toolTip = fv.text;
                } else {
                    fv.toolTip = "<html><b>"+ fv.text +"</b><br>"+ fv.toolTip;
                }
            }
        }
        
        return fv;
    }
    
    /**
     * Applies configuration parameter specific formatting.
     * 
     * @param fv Original formatted value. {@code fv.value} should be an instance of {@code ConfigurationParameterInfo}.
     * @param channel Data channel.
     * @return Modified formatted value.
     */
    public FormattedValue colorConfigurationParameter(FormattedValue fv, AgentChannel channel) {
        ConfigurationParameterInfo conf = (ConfigurationParameterInfo) fv.value;
        fv.value = conf.getCurrentValue();
        if (fv.editable == null) {
            fv.editable = !(conf.isFinal() || conf.isReadOnly() || conf.isBuild());
        }
        if (fv.toolTip == null) {
            if (fv.editable) {
                fv.toolTip = conf.getDescription();
                if (fv.bgColor == null && !channel.getAgent().hasAgentProperty(AgentInfo.AGENT_OFFLINE_PROP)) {
                    fv.bgColor = conf.isDirty() ? COLOR_WARN : COLOR_GOOD;
                }
            } else {
                fv.toolTip = conf.getDescription() + " (" + (conf.isReadOnly() ? "READ-ONLY" : (conf.isBuild() ? "BUILD" : "FINAL")) + ")";
            }
        }
        return fv;
    }
    
    /**
     * For channels that have {@code AgentChannel.Key.STATE} among their keys, sets background
     * color based on current state. If background color of the provided {@code FormattedValue}
     * is already set, this method returns unmodified formatted value.
     * 
     * @param fv Formatted value to modify.
     * @param channel Data channel.
     * @return Modified formatted value.
     */
    public FormattedValue colorByChannelState(FormattedValue fv, AgentChannel channel) {
        if (fv.bgColor == null && keys.contains(AgentChannel.Key.STATE)) {
            if (channel.getAgent().hasAgentProperty(AgentInfo.AGENT_OFFLINE_PROP)) {
                fv.bgColor = COLOR_OFF;
            } else {
                DataProviderState state = channel.get(AgentChannel.Key.STATE);
                if (state != null) {
                    fv.bgColor = COLOR_STATE.get(state);
                }
            }
        }
        return fv;
    }
    
    /**
     * Merges zero or more formatted values.
     * 
     * @param values Formatted values to merge.
     * @return Merged formatted value.
     */
    public FormattedValue merge(List<FormattedValue> values) {
        switch (values.size()) {
            case 0:
                return new FormattedValue(getTitle(), COLOR_FG, COLOR_BG, SwingConstants.CENTER, null, false);
            case 1:
                return values.get(0);
            default:
                Iterator<FormattedValue> it = values.iterator();
                FormattedValue first = it.next();
                String text = first.getText();
                Color fgColor = first.getFgColor();
                Color bgColor = first.getBgColor();
                int horizontalAlignment = first.getHorizontalAlignment();
                String toolTip = first.getToolTip();
                while (it.hasNext()) {
                    FormattedValue other = it.next();
                    if (!Objects.equals(text, other.getText())) {
                        text = TEXT_MULTI;
                    }
                    fgColor = mergeConfigColor(fgColor, other.getFgColor());
                    bgColor = mergeStateColor(bgColor, other.getBgColor());
                    if (horizontalAlignment != other.getHorizontalAlignment()) {
                        horizontalAlignment = HA_MULTI;
                    }
                    if (!Objects.equals(toolTip, other.getToolTip())) {
                        toolTip = null;
                    }
                }
                return new FormattedValue(text, fgColor, bgColor, horizontalAlignment, toolTip, false);
        }
    }
    
    /**
     * Merges colors used to indicate a state of a configurable parameter. <br>
     * Used internally by {@link #merge(MonitorField field, List formattedValues)}.
     * 
     * @param c1 First color.
     * @param c2 Second color.
     * @return Color of the merged value.
     */
    public Color mergeConfigColor(Color c1, Color c2) {
        return COLOR_FG.equals(c1) ? c2 : c1; 
    }
    
    /**
     * Merges colors used to indicate the channel state. <br>
     * Used internally by {@link #merge(MonitorField field, List formattedValues)}.
     * 
     * @param c1 First color.
     * @param c2 Second color.
     * @return Color of the merged value.
     */
    public Color mergeStateColor(Color c1, Color c2) {
        if (c1 == null) return c2;
        if (c2 == null) return c1;
        for (Color c : COLOR_STATE.values()) {
            if (c.equals(c1)) {
                return c2;
            } else if (c.equals(c2)) {
                return c1;
            }
        }
        return c1;
    }


// -- Local methods : ----------------------------------------------------------

    /**
     * Extracts defaultFormat conversion symbol from a defaultFormat string.
     * This horrible code is designed to make best guess for all the misspelled formats in existing groovy files. // FIXME
     * 
     * @param format Format string, ideally as specified in https://docs.oracle.com/javase/8/docs/api/java/util/Formatter.html#syntax .
     * @return Format conversion characters, or a string with a single space character if failed to guess.
     */
    static private String getFormatConversion(String format) {
        String[] ss = format.split("%");
        for (String s : ss) {
            Matcher m = FORMAT_CONVERSION_PATTERN.matcher(s);
            if (m.matches()) {
                return m.group(1);
            }
        }
        return " ";
    }
    

// -- Overriding Object : ------------------------------------------------------

    /**
     * Returns the human readable title of this field.
     * @return Title of this field.
     */
    @Override
    public String toString() {
        return title;
    }
    
    
// -- Specialized subclusses : -------------------------------------------------
    
    /** Central value of a channel. */
    static private class Value extends MonitorField {
        
        Value() {
            super(AgentChannel.Key.VALUE, 
            "Value",
            new FormattedValue(null, null, null, SwingConstants.RIGHT, null, null),
            Collections.unmodifiableSet(new HashSet(Arrays.asList(new String[] {AgentChannel.Key.VALUE, AgentChannel.Key.STATE}))));
        }

        @Override
        public FormattedValue format(AgentChannel channel) {
        
            if (channel == null) return FormattedValue.NA;
            FormattedValue fv = new FormattedValue(template);
            
            // Set defaultFormat specification
            
            if (fv.format == null) {
                fv.format = channel.get(AgentChannel.Key.FORMAT);
            }
            
            // Standard formatting
            
            fv = defaultFormat(channel, fv);
            
            // Color backgrounds for standard states
            
            if (fv.bgColor == null && fv.value instanceof String) {
                String path = channel.getPath();
                if (path.contains("/state/")) {
                    String value = (String) fv.value;
                    Matcher m = CORE_STATE_PATTERN.matcher(path);
                    if (m.matches()) {
                        switch (m.group(1)) {
                            case "AlertState":
                                switch (value) {
                                    case "NOMINAL":
                                        fv.bgColor = COLOR_GOOD;
                                        break;
                                    case "WARNING":
                                        fv.bgColor = COLOR_WARN;
                                        break;
                                    case "ALARM":
                                        fv.bgColor = COLOR_ERR;
                                        break;
                                }
                                break;
                            case "CommandState":
                                switch (value) {
                                    case "READY":
                                        fv.bgColor = COLOR_GOOD;
                                        break;
                                    case "ACTIVE":
                                        fv.bgColor = COLOR_WARN;
                                        break;
                                }
                                break;
                            case "ConfigurationState":
                                switch (value) {
                                    case "CONFIGURED":
                                        fv.bgColor = COLOR_GOOD;
                                        break;
                                    default:
                                        fv.bgColor = COLOR_WARN;
                                        break;
                                }
                                break;
                            case "OperationalState":
                                switch (value) {
                                    case "NORMAL":
                                        fv.bgColor = COLOR_GOOD;
                                        break;
                                    case "ENGINEERING_OK":
                                        fv.bgColor = COLOR_WARN;
                                        break;
                                    case "ENGINEERING_FAULT":
                                        fv.bgColor = COLOR_ERR;
                                        break;
                                }
                                break;
                            case "PhaseState":
                                switch (value) {
                                    case "OPERATIONAL":
                                        fv.bgColor = COLOR_GOOD;
                                        break;
                                    case "INITIALIZING":
                                    case "CLOSING":
                                        fv.bgColor = COLOR_WARN;
                                        break;
                                    case "OFF_LINE":
                                        fv.bgColor = COLOR_ERR;
                                        break;
                                }
                                break;
                        }
                    }
                }
            }
            return fv;
        }
        
    }
    
    /** Alert associated with a channel. */
    static private class Alert extends MonitorField {
        
        Alert(String key) {
            super(key, "Al.", null);
        }
        
        @Override
        public FormattedValue format(AgentChannel channel) {
            String text;
            String alertName;
            Object value = channel.get(key);
            //FIXME: once LSSTCCS-1420 has been resolved and released in the toolkit
            //The line below can be un-commented and the one after removed.
            //This is due to a bug in the core that published String.valueOf(null), so we would sometimes
            //get the String "null"
            if (value instanceof String && !((String)value).trim().isEmpty() ) {
//            if (value instanceof String && !((String)value).trim().isEmpty() && !((String)value).equals("null")) {
                alertName = (String) value;
                text = "  \u2713  ";
            } else {
                alertName = "No alert";
                text = " ";
            }
            return new FormattedValue(text, COLOR_FG, COLOR_BG, SwingConstants.CENTER, alertName, false);
        }
        
    }
    
    /** Warning and alarm limits. */
    static private class Limit extends MonitorField {
        
        Limit(String key, String title) {
            super(key, title, null);
        }

        @Override
        public FormattedValue format(AgentChannel channel) {

            if (channel == null) return FormattedValue.NA;
            
            FormattedValue fv = new FormattedValue();
            fv.horizontalAlignment = SwingConstants.RIGHT;
            
            // Set defaultFormat specification
            
            Object f = channel.get(AgentChannel.Key.FORMAT);
            if (f instanceof String) {
                fv.format = (String) f;
            } else {
                fv.format = DEFAULT_FLOAT_FORMAT;
            }
            
            // Standard formatting
            
            fv = defaultFormat(channel, fv);
            return fv;
        }
        
    }
    
    /** Base class for multi-channel value fields. */
    static private class StateColor extends MonitorField {
        
        private final boolean checkState;
        
        public StateColor(String key, String title, FormattedValue template, Set<String> keys) {
            super(key, title, template, keys);
            checkState = keys.contains(AgentChannel.Key.STATE);
        }

        @Override
        public FormattedValue format(List<AgentChannel> channels) {
            int highestState = -1;
            for (AgentChannel ch : channels) {
                if (ch != null) {
                    if (checkState) {
                        DataProviderState state = ch.get(AgentChannel.Key.STATE);
                        int ordinal = state == null ? -1 : state.ordinal();
                        highestState = Math.max(highestState, ordinal);
                    }
                }
            }
            FormattedValue fv = new FormattedValue(template);
            if (fv.bgColor == null && highestState >= 0) {
                fv.bgColor = COLOR_STATE.get(DataProviderState.values()[highestState]);
            }
            return fv;
        }
    }
    
    static public class Average extends MonitorField {
        
        private final boolean checkState;
        
        public Average(String key, String title, FormattedValue template, Set<String> keys) {
            super(key, title, template, keys);
            checkState = keys.contains(AgentChannel.Key.STATE);
        }

        @Override
        public FormattedValue format(List<AgentChannel> channels) {
            
            if (channels.isEmpty()) return FormattedValue.EMPTY;
            
            int n = 0;
            double sum = 0.;
            int highestState = -1;
            for (AgentChannel ch : channels) {
                if (ch != null) {
                    if (checkState) {
                        DataProviderState state = ch.get(AgentChannel.Key.STATE);
                        int ordinal = state == null ? -1 : state.ordinal();
                        highestState = Math.max(highestState, ordinal);
                        if (ordinal < DataProviderState.NOMINAL.ordinal()) continue;
                    }
                    try {
                        Double value = ch.get(key);
                        if (!(value == null || value.isInfinite() || value.isNaN())) {
                            n++;
                            sum += value;
                        }
                    } catch (ClassCastException x) {
                    }
                }
            }
            if (n > 0) {
                sum /= n;
            } else {
                return FormattedValue.NA;
            }
            
            FormattedValue fv = new FormattedValue(template);
            if (fv.bgColor == null && highestState >= 0) {
                fv.bgColor = COLOR_STATE.get(DataProviderState.values()[highestState]);
            }
            if (fv.format == null) {
                fv.text = String.format(DEFAULT_FLOAT_FORMAT, sum);
            } else {
                fv.text = String.format(fv.format, sum, n);
            }
            return fv;
            
        }
    }

    
// -- Static utilities : -------------------------------------------------------
    
    /**
     * Returns {@code MonitorField} given its string representation.
     * The default implementation selects from {@link #AVAILABLE_GROUP_FIELDS}, and returns {@link #DEFAULT_GROUP_FIELD} if none matches.
     * 
     * @param field String representation of {@code MonitorField}.
     * @return {@code MonitorField} corresponding to the provided string.
     */
    static public MonitorField stringToField(String field) {
        for (MonitorField mf : MonitorField.GROUP_FIELDS) {
            if (mf.toString().equals(field)) {
                return mf;
            }
        }
        return MonitorField.DEFAULT_GROUP_FIELD;
    }
    
    /**
     * Converts {@code MonitorField} to its string representation.
     * The default implementation uses {@code toString()}.
     * 
     * @param field {@code MonitorField} constant to convert.
     * @return String representation.
     */
    static public String fieldToString(MonitorField field) {
        return field.toString();
    }
    
}
