package org.lsst.ccs.gconsole.services.lock;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPopupMenu;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.gconsole.plugins.monitor.ConfigView;

/**
 * Lock widget that can be added to any panel, and used to lock and unlock a remote subsystem.
 * Just create an instance, providing the subsystem name to its constructor, and add to any
 * part of your subsystem-specific GUI. No wiring is necessary, the only method of this class
 * that should ever be used by client code is its constructor.
 * 
 * See {@link ConfigView} for an example of use.
 *
 * @author onoprien
 */
public class Lock extends JLabel {

// -- Fields : -----------------------------------------------------------------
        
    static private final ImageIcon ICON_UNLOCKED = new ImageIcon(Lock.class.getResource("black_unlocked_24.png"), "unlocked black");
    static private final ImageIcon ICON_UNLOCKED_OFFLINE = new ImageIcon(Lock.class.getResource("gray_unlocked_24.png"), "unlocked gray");
    static private final ImageIcon ICON_LOCKED = new ImageIcon(Lock.class.getResource("black_locked_24.png"), "locked black");
    static private final ImageIcon ICON_LOCKED_OFFLINE = new ImageIcon(Lock.class.getResource("gray_locked_24.png"), "locked gray");
    static private final ImageIcon ICON_DETACHED = new ImageIcon(Lock.class.getResource("red_locked_24.png"), "locked red");
    static private final ImageIcon ICON_DETACHED_OFFLINE = new ImageIcon(Lock.class.getResource("light_red_locked_24.png"), "locked light red");
    static private final ImageIcon ICON_ATTACHED = new ImageIcon(Lock.class.getResource("green_locked_24.png"), "locked green");
    static private final ImageIcon ICON_ATTACHED_OFFLINE = new ImageIcon(Lock.class.getResource("light_green_locked_24.png"), "locked green");
    
    private final String agent;
    
    private final ChangeListener lockerListener = e -> update();
    private final LockService.Listener lockServiceListener = new LockService.Listener() {
        @Override
        public void agentsAdded(Locker... agents) {
            for (Locker locker : agents) {
                if (locker.getName().equals(agent)) {
                    locker.addListener(lockerListener);
                    break;
                }
            }
            update();
        }
        @Override
        public void agentsRemoved(Locker... agents) {
            for (Locker locker : agents) {
                if (locker.getName().equals(agent)) {
                    locker.removeListener(lockerListener);
                    break;
                }
            }
            update();
        }
        @Override
        public void userChanged(String oldUserID, String newUserID) {
            update();
        }
    };


// -- Life cycle : -------------------------------------------------------------

    /**
     * Constructs the widget.
     * @param agentName Subsystem name.
     */
    public Lock(String agentName) {
        agent = agentName;
    }

    @Override
    public void removeNotify() {
        LockService serv = LockService.getService();
        serv.removeListener(lockServiceListener);
        Locker locker = serv.getAgent(agent);
        if (locker != null) {
            locker.removeListener(lockerListener);
        }
        super.removeNotify();
    }

    @Override
    public void addNotify() {
        super.addNotify();
        LockService serv = LockService.getService();
        serv.addListener(lockServiceListener);
        Locker locker = serv.getAgent(agent);
        if (locker != null) {
            locker.addListener(lockerListener);
        }
        update();
    }
    
    
// -- Populate label : ---------------------------------------------------------
    
    private void update() {
        Locker locker = LockService.getService().getAgent(agent);
        if (locker == null) {
            setIcon(ICON_UNLOCKED_OFFLINE);
            setToolTipText("Subsystem offline, unlocked");
            setEnabled(true);
        } else {
            if (locker.isAdjusting()) {
                setEnabled(false);
                setToolTipText("Waiting for response from Lock Manager");
            } else {
                setEnabled(true);
            }
            switch (locker.getState()) {
                case UNLOCKED:
                    if (locker.isOnline()) {
                        setIcon(ICON_UNLOCKED);
                        setToolTipText("Unlocked");
                    } else {
                        setIcon(ICON_UNLOCKED_OFFLINE);
                        setToolTipText("Unlocked, Offline");
                    }
                    break;
                case LOCKED:
                    if (locker.isOnline()) {
                        setIcon(ICON_LOCKED);
                        setToolTipText("Locked by " + locker.getLock().getOwner());
                    } else {
                        setIcon(ICON_LOCKED_OFFLINE);
                        setToolTipText("Locked by " + locker.getLock().getOwner() +", Offline");
                    }
                    break;
                case DETACHED:
                    if (locker.isOnline()) {
                        setIcon(ICON_DETACHED);
                        setToolTipText("Locked, Detached");
                    } else {
                        setIcon(ICON_DETACHED_OFFLINE);
                        setToolTipText("Locked, Detached, Offline");
                    }
                    break;
                case ATTACHED:
                    if (locker.isOnline()) {
                        setIcon(ICON_ATTACHED);
                        setToolTipText("Locked, level "+ locker.getLevel());
                    } else {
                        setIcon(ICON_ATTACHED_OFFLINE);
                        setToolTipText("Locked, Offline");
                    }
                    break;
            }
        }        
    }

    @Override
    public JPopupMenu getComponentPopupMenu() {
        LockService lockService = LockService.getService();
        Locker locker = lockService.getAgent(agent);
        if (locker == null || locker.isAdjusting()) {
            return null;
        } else {
            AgentLock lock = locker.getLock();
            boolean canSetLevel = lock == null || lock.getMaxLevel() > 0;
            JPopupMenu menu = new JPopupMenu();
            switch (locker.getState()) {
                case UNLOCKED:
                    menu.add(LockService.Operation.LEVEL.getAction(agent));
                    menu.add(LockService.Operation.LOCK.getAction(agent));
                    break;
                case LOCKED:
                    menu.add(LockService.Operation.DESTROY.getAction(agent));
                    break;
                case DETACHED:
                    if (canSetLevel) menu.add(LockService.Operation.LEVEL.getAction(agent));
                    menu.add(LockService.Operation.ATTACH.getAction(agent));
                    menu.add(LockService.Operation.UNLOCK.getAction(agent));
                    break;
                case ATTACHED:
                    if (canSetLevel) menu.add(LockService.Operation.LEVEL.getAction(agent));
                    menu.add(LockService.Operation.DETACH.getAction(agent));
                    menu.add(LockService.Operation.UNLOCK.getAction(agent));
                    break;
            }
            return menu;
        }        
    }
    
    


}
