package org.lsst.ccs.gconsole.plugins.trending;

import java.awt.Component;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.*;
import java.util.concurrent.CancellationException;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import org.freehep.application.Application;
import org.freehep.application.studio.Studio;
import org.freehep.jas.plugin.plotter.DefaultPage;
import org.freehep.jas.plugin.plotter.DefaultRegion;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.panel.DataPage;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelType;
import org.lsst.ccs.gconsole.jas3.JasPanelManager;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.gconsole.services.persist.Persistable;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;

/**
 * Graphical console data panel that contains trending plots.
 * {@code TrendPage} displays a set of regions, each containing {@code TrendPlotter}.
 * All access to this class should happen on EDT.
 *
 * @author onoprien
 */
public class TrendPage extends DefaultPage implements Persistable, DataPage {

// -- Fields : -----------------------------------------------------------------
    
    static public final String CATEGORY = "TrendPage";
    private final Descriptor descriptor;
    
    private final LsstTrendingPlugin plugin = Console.getConsole().getSingleton(LsstTrendingPlugin.class);

// -- Life cycle : -------------------------------------------------------------
    
    public TrendPage(String title) {
        super(((Studio)Application.getApplication()), Console.getConsole().getSingleton(LsstTrendingPlugin.class).getPlotFactory(), title);
        descriptor = new Descriptor();
        createRegions(1, 1);
    }
    
    public TrendPage(Descriptor desc) {
        super(((Studio)Application.getApplication()), Console.getConsole().getSingleton(LsstTrendingPlugin.class).getPlotFactory(), makeName(desc));
        descriptor = desc.clone();
    }
    
    
// -- Showing pages : ----------------------------------------------------------
    
    @Override
    public void showPage() {
        if (context == null) {
            HashMap<Object, Object> prop = new HashMap<>();
            DataPanelDescriptor pd = getDescriptor().getPanel();
            if (pd != null) {
                HashMap<String, Serializable> data = pd.getData();
                if (data != null) {
                    prop.putAll(data);
                }
            }
            prop.put(Panel.TITLE, title());
            prop.put(Panel.GROUP, plugin.PANEL_GROUP);
            prop.put(Panel.ICON, histogramIcon);
            prop.put(Panel.TYPE, PanelType.DATA);
            Consumer<JComponent> onSaveAs = c -> {
                PersistenceService service = plugin.getConsole().getSingleton(PersistenceService.class);
                service.saveAs(save(), "Save trending page", null);
            };
            prop.put(Panel.ON_SAVE_AS, onSaveAs);
            JasPanelManager panMan = (JasPanelManager) plugin.getConsole().getPanelManager();
            panMan.open(this, prop);
            context = panMan.getContext(this);
        } else {
            plugin.getConsole().getPanelManager().set(this, Panel.SELECTED, true);
        }
    }

    @Override
    public final void createRegions(int columns, int rows) {
        super.createRegions(columns, rows);
        int n = numberOfRegions();
        for (int i=0; i<n; i++) {
            PlotRegion reg = region(i);
            if (reg instanceof DefaultRegion) {
                DefaultRegion region = (DefaultRegion) reg;
                region.addPopupItems((JPopupMenu menu, Component component, Point point) -> {
                    Action act = new AbstractAction("Load Plot...") {
                        @Override
                        public void actionPerformed(ActionEvent e) {
                            PersistenceService service = plugin.getConsole().getSingleton(PersistenceService.class);
                            try {
                                TrendPlotter.Descriptor desc = (TrendPlotter.Descriptor) service.load(TrendPlotter.CATEGORY, "Load trending plot", null);
                                TrendPlotter plotter = (TrendPlotter) service.make(desc);
                                LsstTrendingPlugin.showPlotter(plotter, region);
                            } catch (CancellationException | ClassCastException x) {
                                System.out.println(x);
                            }
                        }
                    };
                    Component[] components = menu.getComponents();
                    int nc = components.length;
                    for (int ic = 0; ic < nc; ic++) {
                        Component c = components[ic];
                        if (c instanceof JMenuItem && "Clear Region".equals(((JMenuItem) c).getText())) {
                            menu.insert(act, ic);
                        }
                    }
                    return menu;
                });
            }
        }
    }

    
// -- Responding to page events : ----------------------------------------------
    
    @Override
    public boolean close() {
        boolean out = super.close();
        if (out) {
            plugin.closePage(this);
        }
        return out;
    }
    
    
// -- Saving/Restoring : -------------------------------------------------------
    
    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }

    @Override
    public Descriptor save() {
        Descriptor desc = getDescriptor().clone();
        desc.setName(title());
        desc.setRows(rows());
        desc.setColumns(columns());
        int n = numberOfRegions();
        ArrayList<TrendPlotter.Descriptor> plotterDescriptors = new ArrayList<>(n);
        for (int i=0; i<n; i++) {
            PlotRegion region = region(i);
            if (region != null) {
                Plotter plotter = region.currentPlot();
                if (plotter != null && plotter instanceof TrendPlotter) {
                    TrendPlotter.Descriptor plotterDescriptor = ((TrendPlotter)plotter).save();
                    if (plotterDescriptor != null) {
                        plotterDescriptors.add(plotterDescriptor);
                    }
                }
            }
        }
        desc.setPlotters(plotterDescriptors.toArray(new TrendPlotter.Descriptor[0]));
        DataPanelDescriptor pd = DataPanelDescriptor.get(this);
        desc.setPanel(pd);
        return desc;
    }
    
    @Override
    public void restore(Persistable.Descriptor d) {
        if (d instanceof Descriptor) {
            Descriptor desc = (Descriptor) d;
            int rows = desc.getRows();
            int columns = desc.getColumns();
            createRegions(columns, rows);
            int i = 0;
            for (TrendPlotter.Descriptor plotterDescriptor : desc.getPlotters()) {
                TrendPlotter plotter = new TrendPlotter();
                LsstTrendingPlugin.showPlotter(plotter, region(i++));
                plotter.restore(plotterDescriptor);
            }
        }
    }
    
    static public class Descriptor extends Persistable.Descriptor {

        private int rows;
        private int columns;
        private TrendPlotter.Descriptor[] plotters;
        private DataPanelDescriptor panel;

        // Getters and setters:
        
        @Override
        public String getCategory() {
            return TrendPage.CATEGORY;
        }
        
        @Override
        public void setCategory(String category) {
        }

        public TrendPlotter.Descriptor[] getPlotters() {
            return plotters;
        }

        public void setPlotters(TrendPlotter.Descriptor[] plotters) {
            this.plotters = plotters;
        }

        public TrendPlotter.Descriptor getPlotters(int index) {
            return this.plotters[index];
        }

        public void setPlotters(int index, TrendPlotter.Descriptor plotters) {
            this.plotters[index] = plotters;
        }

        public int getColumns() {
            return columns;
        }

        public void setColumns(int columns) {
            this.columns = columns;
        }

        public int getRows() {
            return rows;
        }

        public void setRows(int rows) {
            this.rows = rows;
        }

        public DataPanelDescriptor getPanel() {
            return panel;
        }

        public void setPanel(DataPanelDescriptor panel) {
            this.panel = panel;
        }
        
        // Cloning:

        @Override
        public Descriptor clone() {
            Descriptor clone = (Descriptor) super.clone();
            if (plotters != null) {
                int n = plotters.length;
                clone.plotters = new TrendPlotter.Descriptor[n];
                for (int i=0; i<n; i++) {
                    TrendPlotter.Descriptor tpDesc = plotters[i];
                    if (tpDesc != null) {
                        clone.plotters[i] = tpDesc.clone();
                    }
                }
            }
            if (panel != null) {
                clone.panel = panel.clone();
            }
            return clone;
        }
        
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    static private String makeName(Descriptor desc) {
        String out = desc.getName();
        if (out == null || out.isEmpty()) {
            TrendPlotter.Descriptor[] plotDesc = desc.getPlotters();
            if (plotDesc != null && plotDesc.length > 0) {
                ArrayList<String> plotNames = new ArrayList<>(plotDesc.length);
                for (TrendPlotter.Descriptor d : plotDesc) {
                    plotNames.add(makeName(d));
                }
                out = combineNames(plotNames);
            }
        }
        return out;
    }
    
    static private String makeName(TrendPlotter.Descriptor desc) {
        String out = desc.getName();
        if (out == null || out.isEmpty()) {
            Trend.Descriptor[] trendDesc = desc.getTrends();
            if (trendDesc != null && trendDesc.length > 0) {
                ArrayList<String> trendNames = new ArrayList<>(trendDesc.length);
                for (Trend.Descriptor d : trendDesc) {
                    String n = d.getTitle();
                    trendNames.add(n == null ? d.getPath() : n);
                }
                out = combineNames(trendNames);
            }
        }
        return out;
    }
    
    static private String combineNames(final List<String> names) {
        if (names == null || names.isEmpty()) return null;
        StringBuilder sb = new StringBuilder();
        int minLength = Integer.MAX_VALUE;
        for (String name : names) {
            if (name == null) return null;
            minLength = Math.min(minLength, name.length());
        }
        if (minLength == 0) return null;
        for (int i=0; i<minLength; i++) {
            char c = names.get(0).charAt(i);
            boolean same = true;
            for (int k=1; k < names.size(); k++) {
                if (c != names.get(k).charAt(i)) {
                    same = false;
                    break;
                }
            }
            if (same) {
                sb.append(c);
            } else {
                break;
            }
        }
        while (sb.charAt(0) == '/') {
            sb.deleteCharAt(0);
        }
        while (sb.length() > 0 && sb.charAt(sb.length()-1) == '/') {
            sb.deleteCharAt(sb.length()-1);
        }
        return sb.length() < 3 ? null : sb.toString();
    }
    
}
