package org.lsst.ccs.gconsole.plugins.alert;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.time.Instant;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextPane;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.ToolTipManager;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultCaret;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyleContext;
import javax.swing.text.StyledDocument;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertInstance;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import static org.lsst.ccs.gconsole.base.Const.HSPACE;
import static org.lsst.ccs.gconsole.base.Const.VSPACE;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.gconsole.util.tree.TreeUtil;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;

/**
 * Display of CCS alert summary.
 * All methods except {@link #onAlert} should be called on EDT.
 *
 * @author onoprien
 */
public class AlertViewer implements AlertListener {

// -- Fields : -----------------------------------------------------------------
    
    private final LsstAlertPlugin plugin;
    private Descriptor config;
    
    private AlertTree tree;
    private JTextPane infoPanel;
    private JPanel rightPanel;
    private JScrollPane leftPanel;
    private JSplitPane rootPanel;
    
    private Action clearAction;
    
    private JCheckBox freezeBox;
    private JCheckBox historyBox;
    private JCheckBox muteBox;
    private JCheckBox toFrontBox;
    private JCheckBox selectNewBox;
    
    private StyledDocument document;
    private Style attBase, attBlack, attBlackBold;
    private final EnumMap<AlertState,Style> attPlane = new EnumMap<>(AlertState.class);
    private final EnumMap<AlertState,Style> attBold = new EnumMap<>(AlertState.class);
    
    static private final EnumMap<AlertState,Color> COLOR = new EnumMap<>(AlertState.class);
    static {
        COLOR.put(AlertState.NOMINAL, new Color(0, 0xAA, 0));
        COLOR.put(AlertState.WARNING, new Color(0, 0, 0xAA));
        COLOR.put(AlertState.ALARM, new Color(0xAA, 0, 0));
    }

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Creates an instance of {@code AlertViewer}.
     * No GUI panels are constructed and no resources are allocated at this point.
     * @param plugin Alert plugin instance that created this viewer.
     * @param configuration
     */
    public AlertViewer(LsstAlertPlugin plugin, Descriptor configuration) {
        this.plugin = plugin;
        config = configuration == null ? new Descriptor() : configuration;
    }
    
    /**
     * Wake up and get ready to display alerts.
     * This method is called if an alert event is received while this viewer is asleep,
     * before the event is handled.
     */
    private void start() {
                
        clearAction = new AbstractAction("Clear") {
            @Override public void actionPerformed(ActionEvent e) {
                AlertNode node = tree.getSelectedNode();
                if (node == null) return;
                HashMap<String, String[]> alerts = getAlertMap(node);
                new SwingWorker<Void, Void>() {
                    @Override
                    protected Void doInBackground() throws Exception {
                        plugin.clearAlerts(alerts);
                        return null;
                    }
                }.execute();
            }
        };
        clearAction.putValue(Action.SHORT_DESCRIPTION, "Clear selected alerts.");
        
        Box buttonPanel = Box.createHorizontalBox();
        buttonPanel.add(new JButton(clearAction));
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));

        freezeBox = new JCheckBox("Freeze");
        freezeBox.setSelected(false);
        freezeBox.addActionListener(e -> {
            if (!freezeBox.isSelected()) {
                tree.getSelectedNode().fillInfoPanel(this);
            }
        });
        freezeBox.setToolTipText("Freeze information panel content");
        buttonPanel.add(freezeBox);
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
        
        historyBox = new JCheckBox("History");
        historyBox.setSelected((boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_HISTORY));
        historyBox.addActionListener(e -> {
            plugin.getServices().setProperty(LsstAlertPlugin.OPT_HISTORY, historyBox.isSelected());
            tree.getSelectedNode().fillInfoPanel(this);
        });
        historyBox.setToolTipText("Show history of alerts");
        buttonPanel.add(historyBox);
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
        
        muteBox = new JCheckBox("Mute");
        muteBox.setSelected((boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_MUTE));
        muteBox.addActionListener(e -> plugin.getServices().setProperty(LsstAlertPlugin.OPT_MUTE, muteBox.isSelected()));
        muteBox.setToolTipText("Disable audio alerts");
        buttonPanel.add(muteBox);
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
        
        selectNewBox = new JCheckBox("Select new alerts");
        selectNewBox.setSelected((boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_SELECT));
        selectNewBox.addActionListener(e -> plugin.getServices().setProperty(LsstAlertPlugin.OPT_SELECT, selectNewBox.isSelected()));
        selectNewBox.setToolTipText("Automatically select new alerts");
        buttonPanel.add(selectNewBox);
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
        
        toFrontBox = new JCheckBox("Show viewer on alert");
        toFrontBox.setSelected((boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_TOFRONT));
        toFrontBox.addActionListener(e -> plugin.getServices().setProperty(LsstAlertPlugin.OPT_TOFRONT, toFrontBox.isSelected()));
        toFrontBox.setToolTipText("Display alert viewer when new alerts are received");
        buttonPanel.add(toFrontBox);
        buttonPanel.add(Box.createHorizontalGlue());
        buttonPanel.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
    
        tree = AlertTree.getInstance();
        tree.addTreeSelectionListener(e -> {
            AlertNode node = tree.getSelectedNode();
            if (node == null) { // nothing selected
                clearAction.setEnabled(false);
                historyBox.setEnabled(false);
                if (!freezeBox.isSelected()) {
                    infoPanel.setText("");
                }
            } else {
                clearAction.setEnabled(true);
                historyBox.setEnabled(node.isLeaf());
                if (!freezeBox.isSelected()) {
                    node.fillInfoPanel(this);
                }
            }
        });
        tree.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
        tree.addMouseListener(new MouseAdapter() {
            @Override public void mouseClicked(MouseEvent e) {
                if (freezeBox.isSelected()) freezeBox.doClick();
            }
        });
    
        infoPanel = new JTextPane();
        infoPanel.setEditable(false);
        infoPanel.setContentType("text/plain");
        infoPanel.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
        try {
            DefaultCaret caret = (DefaultCaret) infoPanel.getCaret();
            caret.setUpdatePolicy(DefaultCaret.NEVER_UPDATE);
        } catch (ClassCastException x) {
        }
        infoPanel.setText("");
        document = infoPanel.getStyledDocument();
        Style style = StyleContext.getDefaultStyleContext().getStyle(StyleContext.DEFAULT_STYLE);
        attBase = document.addStyle("attBase", style);
        attBlack = document.addStyle("attBlack", attBase);
        attBlackBold = document.addStyle("attBlackBold", attBase);
        StyleConstants.setBold(attBlackBold, true);
        for (AlertState state : AlertState.values()) {
            style = document.addStyle(state.name(), attBase);
            StyleConstants.setForeground(style, COLOR.get(state));
            attPlane.put(state, style);
            style = document.addStyle(state.name(), style);
            StyleConstants.setBold(style, true);
            attBold.put(state, style);
        }
        
        rightPanel = new JPanel(new BorderLayout());
        JScrollPane scroll = new JScrollPane(infoPanel);
        rightPanel.add(scroll, BorderLayout.CENTER);
        rightPanel.add(buttonPanel, BorderLayout.SOUTH);
        
        leftPanel = new JScrollPane(tree);
        
        rootPanel = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, leftPanel, rightPanel);
        Dimension minimumSize = new Dimension(100, 50);
        leftPanel.setMinimumSize(minimumSize);
        rightPanel.setMinimumSize(minimumSize);
        rootPanel.setOneTouchExpandable(false);
        rootPanel.setResizeWeight(.3);
        
        Map<Object, Object> par = new HashMap<>();
        DataPanelDescriptor panDesc = config.getPage();
        if (panDesc != null && panDesc.isOpen()) {
            Map<String, Serializable> data = panDesc.getData();
            if (data != null) {
                par.putAll(data);
            }
        }
        par.put(Panel.TITLE, "Alerts");
        Consumer<JComponent> onClose = c -> stop();
        par.put(Panel.ON_CLOSE, onClose);
        Console.getConsole().getPanelManager().open(rootPanel, par);
    }

    /** Release resources and go to sleep until the next event is received. */
    void stop() {
        if (rootPanel == null) return;
        config = save();
        clearAction = null;
        historyBox = null;
        muteBox = null;
        toFrontBox = null;
        tree = null;
        infoPanel = null;
        rightPanel = null;
        leftPanel = null;
        rootPanel = null;
    }
    
    /**
     * 
     */
    void destroy() {
        if (rootPanel != null) {
            Console.getConsole().getPanelManager().close(rootPanel);
        }
    }


// -- Processing alert events --------------------------------------------------
    
    @Override
    public void onAlert(AlertEvent event) {
        if (SwingUtilities.isEventDispatchThread()) {
            update(event);
        } else {
            SwingUtilities.invokeLater(() -> update(event));
        }
    }

    private void update(AlertEvent event) {
        
        String source = event.getSource();
        RaisedAlertSummary summary = event.getSummary();
        
        // Process agent disconnects
        
        if (summary == null) {
            if (tree != null) {
                 AlertNode sourceNode = getSourceNode(source);
                if (sourceNode != null) {
                    sourceNode.removeFromParent();
                    tree.getModel().reload();
                    tree.clearSelection();
                }
            }
            return;
        }
        
        // Start viewer if necessary
        
        Set<RaisedAlertHistory> currentAlerts = summary.getAllRaisedAlertHistories();
        if (tree == null) {
            if (currentAlerts.isEmpty()) {
                return;
            } else {
                start();
            }
        }
        
        // Update tree model

        TreePath selPath = tree.getSelectionPath();
        
        AlertNode sourceNode = getSourceNode(source);        
        if (sourceNode == null) { // tree has no alerts from this source
            if (currentAlerts.isEmpty()) {
                return; // and the event does not have any alerts either
            }
            sourceNode = new AlertNode(source);
            for (RaisedAlertHistory h : currentAlerts) {
                updateAlert(sourceNode, h);
            }
            tree.getRoot().add(sourceNode);
        } else {
            HashMap<String,RaisedAlertHistory> id2history = new HashMap<>(currentAlerts.size()*2);
            currentAlerts.forEach(h -> id2history.put(h.getLatestAlert().getAlertId(), h));
            for (IDNode idNode : sourceNode.getLeaves()) {
                    String id = idNode.getID();
                    RaisedAlertHistory history = id2history.remove(id);
                    if (history == null) {
                        removeAlert(idNode);
                    } else {
                            updateAlert(sourceNode, history);
                    }
            }
            for (RaisedAlertHistory h : id2history.values()) {
                updateAlert(sourceNode, h);
            }
        }
        
        // Update tree
        
        ArrayList<TreePath> state = TreeUtil.saveExpansionState(tree);
        tree.getModel().reload();
        TreeUtil.restoreExpansionState(tree, state);

        TreePath path;     
        if (selectNewBox.isSelected()) {
            String id = "";
            Alert alert = event.getAlert();
            if (alert == null) {
                for (String clearedID : event.getClearedIds()) {
                    try {
                        if (event.isPartialClear(clearedID)) {
                            id = clearedID;
                            break;
                        }
                    } catch (RuntimeException x) {
                    }
                }
            } else {
                id = alert.getAlertId();
            }
            IDNode idNode = id.isEmpty() ? null : tree.findID(sourceNode, id);
            path = new TreePath(idNode == null ? new Object[]{tree.getRoot(), sourceNode} : idNode.getPath());
        } else {
            path = selPath;
        }
        
        if (isValidPath(path)) {
            tree.setSelectionPath(path);
            tree.expandPath(path);
        } else {
            tree.clearSelection();
        }
        
        // Get user's attention
        
        if (!muteBox.isSelected()) {
            Toolkit.getDefaultToolkit().beep();
        }
        if (toFrontBox.isSelected()) {
            toFront();
        }
        
    }
    
    void toFront() {
        PanelManager panMan = plugin.getConsole().getPanelManager();
        try {
            panMan.set(rootPanel, Panel.SELECTED, true);
        } catch (RuntimeException x) {
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private AlertNode getSourceNode(String source) {
        AlertNode root = tree.getRoot();
        for (int i=0; i<root.getChildCount(); i++) {
            AlertNode node = root.getChildAt(i);
            if (node.getUserObject().equals(source)) return node;
        }
        return null;
    }
    
    private HashMap<String,String[]> getAlertMap(AlertNode node) {
        HashMap<String, String[]> alerts = new HashMap<>();
        if (node instanceof IDNode) {
            IDNode idNode = (IDNode) node;
            alerts.put(node.getSource(), new String[]{idNode.getID()});
        } else if (node.isRoot()) {
            Enumeration e = node.children();
            while (e.hasMoreElements()) {
                AlertNode sourceNode = (AlertNode) e.nextElement();
                alerts.put(sourceNode.getSource(), null);
            }
        } else if (node.getLevel() == 1) {
            alerts.put(node.getSource(), null);
        } else {
            String source = node.getSource();
            ArrayList<String> ids = new ArrayList<>();
            Enumeration e = node.depthFirstEnumeration();
            while (e.hasMoreElements()) {
                Object o = e.nextElement();
                if (o instanceof IDNode) {
                    IDNode idNode = (IDNode) o;
                    ids.add(idNode.getID());
                }
            }
            alerts.put(source, ids.toArray(new String[0]));
        }
        return alerts;
    }
    
    static public String formatTimeStamp(Instant instant) {
        return Const.DEFAULT_DT_FORMAT.format(instant);
    }
    
    private void updateAlert(AlertNode sourceNode, RaisedAlertHistory history) {
        
        String id = history.getLatestAlert().getAlertId();
        String[] ss = id.split("/");
        ArrayList<String> tokens = new ArrayList<>(ss.length);
        for (String s : ss) {
            s = s.trim();
            if (!s.isEmpty()) tokens.add(s);
        }
        if (tokens.isEmpty()) return;
        
        for (int i=0; i<tokens.size()-1; i++) {
            String s = tokens.get(i);
            AlertNode tokenNode = null;
            Enumeration e = sourceNode.children();
            while (e.hasMoreElements()) {
                AlertNode node = (AlertNode) e.nextElement();
                if (!node.isLeaf()) {
                    if (node.getUserObject().equals(s)) {
                        tokenNode = node;
                        break;
                    }
                }
            }
            if (tokenNode == null) {
                tokenNode = new AlertNode(s);
                sourceNode.add(tokenNode);
            }
            sourceNode = tokenNode;
        }
        
        IDNode idNode = null;
        Enumeration e = sourceNode.children();
        while (e.hasMoreElements()) {
            Object o = e.nextElement();
            if (o instanceof IDNode) {
                IDNode node = (IDNode) o;
                if (node.getID().equals(id)) {
                    idNode = node;
                    break;
                }
            }
        }
        
        if (idNode == null) {
            idNode = new IDNode(history);
            sourceNode.add(idNode);
        } else {
            idNode.setHistory(history);
        }
        
    }
    
    private void removeAlert(IDNode idNode) {
        AlertNode node = idNode;
        while (true) {
            AlertNode parent = node.getParent();
            if (parent == null) return;
            if (parent.isRoot() || parent.getChildCount() > 1) {
                node.removeFromParent();
                return;
            } else {
                node = parent;
            }
        }
    }
    
    private boolean isValidPath(TreePath path) {
        if (path == null) return false;
        int n = path.getPathCount();
        if (n == 0) {
            return false;
        } else {
            AlertNode child = (AlertNode) path.getLastPathComponent();
            for (int i=n-2; i>=0; i--) {
                AlertNode parentFromPath = (AlertNode) path.getPathComponent(i);
                AlertNode parentFromChild = child.getParent();
                if (parentFromPath != null && parentFromChild != null && parentFromPath.equals(parentFromChild)) {
                    child = parentFromPath;
                } else {
                    return false;
                }
            }
            return child.equals(tree.getRoot());
        }
    }
    

// -- Alert tree utilities : ---------------------------------------------------
    
    static private enum WatchState {ACTIVE, MUTE, IGNORE}
    
    static private class AlertTree extends JTree {
        
        static AlertTree getInstance() {
            AlertTree t = new AlertTree();
            t.setModel(new AlertTreeModel());
            t.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
            t.setCellRenderer(new AlertTreeRenderer());
            ToolTipManager.sharedInstance().registerComponent(t);
            return t;
        }
        
        public AlertNode getSelectedNode() {
            return (AlertNode) getLastSelectedPathComponent();
        }
        
        @Override
        public AlertTreeModel getModel() {
            return (AlertTreeModel) super.getModel();
        }
        
        AlertNode getRoot() {
            return getModel().getRoot();
        }
                
        IDNode findID(AlertNode sourceNode, String id) {
            if (sourceNode == null) return null;
            Enumeration e = sourceNode.depthFirstEnumeration();
            while (e.hasMoreElements()) {
                Object node = e.nextElement();
                if (node instanceof IDNode && ((IDNode)node).getID().equals(id))  {
                    return (IDNode)node;
                }
            }
            return null;
        }

    }
    
    static private class AlertTreeModel extends DefaultTreeModel {
        
        AlertTreeModel() {
            super(new AlertNode("CCS Alerts") {
                
                @Override
                WatchState getWatchState() {
                    return isIgnore() ? WatchState.IGNORE : (isMute() ? WatchState.MUTE : WatchState.ACTIVE);
                }

                @Override
                void fillInfoPanel(AlertViewer v) {
                    int activeAlerts = 0;
                    if (children != null) {
                        for (Object child : children) {
                            if (((AlertNode)child).getAlertLevel() != AlertState.NOMINAL) {
                                activeAlerts++;
                            }
                        }
                    }
                    StyledDocument d = v.document;
                    try {
                        d.remove(0, d.getLength());
                        d.insertString(d.getLength(), "CCS Control System\n\n", v.attBlackBold);
                        d.insertString(d.getLength(), "Subsystems with active alerts: ", v.attBlack);
                        d.insertString(d.getLength(), activeAlerts + "\n", v.attBlackBold);
                        d.insertString(d.getLength(), "Overall alert level: ", v.attBlack);
                        d.insertString(d.getLength(), getAlertLevel() +"\n", v.attBold.get(getAlertLevel()));
                        d.insertString(d.getLength(), "Last changed at ", v.attBlack);
                        d.insertString(d.getLength(), Const.DEFAULT_DT_FORMAT.format(getTimeStamp()) + "\n", v.attBlackBold);
                        v.infoPanel.setCaretPosition(0);
                    } catch (BadLocationException x) {
                    }
                }

                @Override
                String getSource() {
                    throw new UnsupportedOperationException("Trying to look up source subsystem for the root node.");
                }
                
            });
        }
        
        @Override
        public AlertNode getRoot() {
            return (AlertNode) super.getRoot();
        }
        
    }
    
    /** Mutable tree node that contains an instance of AlertNodeContent as its user object. */
    static class AlertNode extends DefaultMutableTreeNode implements Comparable<AlertNode> {
        
        private boolean mute;
        private boolean ignore;
        
        AlertNode(String userObject) {
            super(userObject);
        }

        @Override
        public String getUserObject() {
            return (String) super.getUserObject();
        }

        @Override
        public int compareTo(AlertNode other) {
            return getUserObject().compareTo(other.getUserObject());
        }
        
        @Override
        public void add(MutableTreeNode newChildNode) {
            AlertNode newChild = (AlertNode) newChildNode;
            Enumeration<TreeNode> e = children();
            int i = 0;
            boolean replace = false;
            while (e.hasMoreElements()) {
                AlertNode child = (AlertNode) e.nextElement();
                int c = newChild.compareTo(child);
                if (c > 0) {
                    i++;
                } else {
                    if (c == 0) replace = true;
                    break;
                }
            }
            if (replace) remove(i);
            insert(newChild, i);
        }

        @Override
        public AlertNode getParent() {
            return (AlertNode) super.getParent();
        }

        @Override
        public AlertNode getChildAt(int index) {
            return (AlertNode) super.getChildAt(index);
        }
        
        public ArrayList<IDNode> getLeaves() {
            ArrayList<IDNode> out = new ArrayList<>(this.getLeafCount());
            Enumeration e = this.depthFirstEnumeration();
            while (e.hasMoreElements()) {
                Object node = e.nextElement();
                if (node instanceof IDNode) {
                    out.add((IDNode)node);
                }
            }
            return out;
        }
        
        void fillInfoPanel(AlertViewer v) {
            StyledDocument d = v.document;
            try {
                d.remove(0, d.getLength());
                d.insertString(d.getLength(), "Source: ", v.attBlack);
                d.insertString(d.getLength(), getSource() +"\n", v.attBlackBold);
                d.insertString(d.getLength(), "Alerts selected: ", v.attBlack);
                d.insertString(d.getLength(), getLeafCount() +"\n", v.attBlackBold);
                d.insertString(d.getLength(), "Highest alert level: ", v.attBlack);
                AlertState highestState = getAlertLevel();
                d.insertString(d.getLength(), highestState +"\n", v.attBold.get(highestState));
                d.insertString(d.getLength(), "Last changed at ", v.attBlack);
                d.insertString(d.getLength(), Const.DEFAULT_DT_FORMAT.format(getTimeStamp()) +"\n", v.attBlackBold);
                v.infoPanel.setCaretPosition(0);
            } catch (BadLocationException x) {
            }
        }

        String getSource() {
            int level = getLevel();
            if (level > 1) {
                return getParent().getSource();
            } else {
                return getUserObject();
            }
        }
        
        Instant getTimeStamp() {
            Instant out = Instant.EPOCH;
            if (children != null) {
                for (Object child : children) {
                    Instant time = ((AlertNode) child).getTimeStamp();
                    if (time.compareTo(out) > 0) out = time;
                }
            }
            return out;
        }
        
        AlertState getAlertLevel() {
            AlertState alertState = AlertState.NOMINAL;
            Enumeration e = this.children();
            while (e.hasMoreElements()) {
                AlertState childState = ((AlertNode) e.nextElement()).getAlertLevel();
                if (childState.compareTo(alertState) > 0) {
                    alertState = childState;
                    if (alertState.equals(AlertState.ALARM)) break;
                }
            }
            return alertState;
        }
        
        WatchState getWatchState() {
            WatchState thisState = isIgnore() ? WatchState.IGNORE : (isMute() ? WatchState.MUTE : WatchState.ACTIVE);
            if (thisState == WatchState.IGNORE) return WatchState.IGNORE;
            WatchState parentState = this.getParent().getWatchState();
            return thisState.compareTo(parentState) == -1 ? parentState : thisState;
        }
        
        boolean isMute() {
            return mute;
        }
        
        void setMute(boolean isMute) {
            mute = isMute;
        }
        
        boolean isIgnore() {
            return ignore;
        }
        
        void setIgnore(boolean isIgnored) {
            ignore = isIgnored;
        }
    }
    
    final class IDNode extends AlertNode {
        
        private RaisedAlertHistory history;
        
        IDNode(RaisedAlertHistory history) {
            super(history.getLatestAlert().getAlertId());
            setHistory(history);
        }

        void setHistory(RaisedAlertHistory history) {
            this.history = history;
        }
        
        RaisedAlertHistory getHistory() {
            return history;
        }
        
        String getID() {
            return getUserObject();
        }

        @Override
        public String toString() {
            return getLastIdComponent();
        }
        
        String getLastIdComponent() {
            String[] ss = getID().split("/");
            int i = ss.length;
            while (i > 0) {
                String s = ss[--i].trim();
                if (!s.isEmpty()) return s;
            }
            return getID();
        }
        
        @Override
        void fillInfoPanel(AlertViewer v) {
            Alert alert = history.getLatestAlert();
            StyledDocument d = v.document;
            try {
                d.remove(0, d.getLength());
                d.insertString(d.getLength(), "Source: ", v.attBlack);
                d.insertString(d.getLength(), getSource() +"\n", v.attBlackBold);
                d.insertString(d.getLength(), "ID: ", v.attBlack);
                d.insertString(d.getLength(), alert.getAlertId() +"\n", v.attBlackBold);
                d.insertString(d.getLength(), "Description: ", v.attBlack);
                d.insertString(d.getLength(), alert.getDescription() +"\n", v.attBlackBold);
                d.insertString(d.getLength(), "Highest level: ", v.attBlack);
                d.insertString(d.getLength(), history.getHighestAlertState() +"\n\n", v.attBold.get(history.getHighestAlertState()));
                ArrayList<RaisedAlertInstance> alerts = history.getRaisedAlertInstancesList();
                if (historyBox.isSelected()) {
                    d.insertString(d.getLength(), "History:\n\n", v.attBlackBold);
                    int n = alerts.size();
                    int nShow = Math.min(n, 50);
                    for (int i = 0; i < nShow; i++) {
                        RaisedAlertInstance ra = alerts.get(n - 1 - i);
                        d.insertString(d.getLength(), formatTimeStamp(ra.getCCSTimeStamp().getUTCInstant()) + " " + ra.getAlertState() + "\n", v.attBold.get(ra.getAlertState()));
                        d.insertString(d.getLength(), ra.getCause() + "\n\n", v.attBlack);
                    }
                    if (nShow < n) {
                        if (nShow < n - 1) {
                            d.insertString(d.getLength(), "...\n\n" + (n - nShow - 1) + " more.\n\n...\n\n", v.attBlack);
                        }
                        RaisedAlertInstance ra = alerts.get(0);
                        d.insertString(d.getLength(), formatTimeStamp(ra.getCCSTimeStamp().getUTCInstant()) + " " + ra.getAlertState() + "\n", v.attPlane.get(ra.getAlertState()));
                        d.insertString(d.getLength(), ra.getCause() + "\n\n", v.attBlack);
                    }
                } else {
                    int n = alerts.size();
                    if (n > 1) {
                        d.insertString(d.getLength(), "Last raised: " + formatTimeStamp(history.getLatestAlertCCSTimeStamp().getUTCInstant()) + " : ", v.attBlackBold);
                        d.insertString(d.getLength(), history.getLatestAlertState() + "\n", v.attBold.get(history.getLatestAlertState()));
                        d.insertString(d.getLength(), alerts.get(n - 1).getCause() + "\n\n", v.attBlack);
                        RaisedAlertInstance first = alerts.get(0);
                        d.insertString(d.getLength(), "First raised: " + formatTimeStamp(first.getCCSTimeStamp().getUTCInstant()) + " : ", v.attBlackBold);
                        d.insertString(d.getLength(), first.getAlertState() + "\n", v.attBold.get(first.getAlertState()));
                        d.insertString(d.getLength(), first.getCause() + "\n", v.attBlack);
                    } else if (n > 0) {
                        d.insertString(d.getLength(), "Raised: " + formatTimeStamp(history.getLatestAlertCCSTimeStamp().getUTCInstant()) + " : ", v.attBlackBold);
                        d.insertString(d.getLength(), history.getLatestAlertState() + "\n", v.attBold.get(history.getLatestAlertState()));
                        d.insertString(d.getLength(), alerts.get(0).getCause() + "\n", v.attBlack);
                    }
                }
                v.infoPanel.setCaretPosition(0);
            } catch (BadLocationException x) {
            }
        }
        
        @Override
        Instant getTimeStamp() {
            return history.getLatestAlertCCSTimeStamp().getUTCInstant();
        }
        
        @Override
        AlertState getAlertLevel() {
            return history.getHighestAlertState();
        }
        
    }
    
    static private class AlertTreeRenderer extends DefaultTreeCellRenderer {

        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel, boolean expanded, boolean leaf, int row, boolean hasFocus) {
            super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
            try {
                AlertNode node = (AlertNode) value;
                AlertState state = node.getAlertLevel();
                if (!sel) {
                    setForeground(COLOR.get(state));
                }
                if (leaf) {
                    setToolTipText(((IDNode)node).getHistory().getLatestAlert().getDescription());
                } else {
                    setToolTipText(null);
                }
            } catch (ClassCastException | NullPointerException x) {
            }
            return this;
        }
        
    }
    
    
// -- Saving session : ---------------------------------------------------------
    
    static public class Descriptor extends ComponentDescriptor {
        
        private DataPanelDescriptor page;

        public DataPanelDescriptor getPage() {
            return page;
        }

        public void setPage(DataPanelDescriptor page) {
            this.page = page;
        }
      
    }
    
    Descriptor save() {
        if (rootPanel != null) {
            config.setPage(DataPanelDescriptor.get(rootPanel));
        }
        return config;
    }
    
}
