package org.lsst.ccs.gconsole.util.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.GridLayout;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Month;
import java.time.format.TextStyle;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFormattedTextField;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.MatteBorder;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumn;
import static org.lsst.ccs.gconsole.base.Const.HDIM;
import static org.lsst.ccs.gconsole.base.Const.HSPACE;
import static org.lsst.ccs.gconsole.base.Const.VSPACE;

/**
 * A dialog for choosing a date and time.
 *
 * @author onoprien
 */
public final class DateTimePicker extends JDialog {

// -- Fields : -----------------------------------------------------------------
    
    private LocalDate date;
    private LocalTime time;
    
    private boolean armed;
    private JComboBox<String> monthCombo;
    JSpinner yearSpinner;
    private CalendarModel calModel;
    private JTable calTable;

// -- Life cycle : -------------------------------------------------------------
    
    private DateTimePicker(LocalDate seedDate, LocalTime seedTime, String title, Component parent) {
        super(parent == null ? null : SwingUtilities.getWindowAncestor(parent), title, Dialog.ModalityType.APPLICATION_MODAL);
        this.setResizable(false);
        date = seedDate;
        time = seedTime;
        Box root = Box.createVerticalBox();
        add(root);
        root.setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
        
        // Date:
        
        if (time != null) {
            
            JPanel datePanel = new JPanel();
            root.add(datePanel);
            datePanel.setLayout(new BorderLayout());
            datePanel.setBorder(BorderFactory.createTitledBorder("Date"));
            
            Box monthYearPanel = Box.createHorizontalBox();
            datePanel.add(monthYearPanel, BorderLayout.NORTH);
            JButton prevButton = new JButton("<");
            monthYearPanel.add(prevButton);
            prevButton.addActionListener(e -> {
                date = date.minusMonths(1);
                resetFromDate();
            });
            monthYearPanel.add(Box.createRigidArea(HDIM));
            monthCombo = new JComboBox(Arrays.stream(Month.values()).map(m -> m.getDisplayName(TextStyle.FULL, Locale.US)).collect(Collectors.toList()).toArray());
            monthYearPanel.add(monthCombo);
            monthCombo.setEditable(false);
            monthCombo.addActionListener(e -> {
                if (armed) {
                    date = date.withMonth(monthCombo.getSelectedIndex() + 1);
                    calModel.update();
                }
            });
            monthYearPanel.add(Box.createRigidArea(HDIM));
            yearSpinner = new JSpinner(new SpinnerNumberModel(date.getYear(), 0, 9999, 1));
            monthYearPanel.add(yearSpinner);
            yearSpinner.setEditor(new JSpinner.NumberEditor(yearSpinner, "####"));
            yearSpinner.addChangeListener(e -> {
                if (armed) {
                    date = date.withYear((Integer) yearSpinner.getValue());
                    calModel.update();
                }
            });
            monthYearPanel.add(Box.createRigidArea(HDIM));
            JButton nextButton = new JButton(">");
            monthYearPanel.add(nextButton);
            nextButton.addActionListener(e -> {
                date = date.plusMonths(1);
                resetFromDate();
            });
            
            JPanel calPanel = new JPanel();
            datePanel.add(calPanel, BorderLayout.CENTER);
            calPanel.setLayout(new BorderLayout());
            calPanel.setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createEmptyBorder(VSPACE, 0, 0, 0), 
                    BorderFactory.createLineBorder(Color.GRAY)));
            calModel = new CalendarModel();
            calTable = new JTable(calModel);
            JTableHeader head = calTable.getTableHeader();
            calPanel.add(head, BorderLayout.NORTH);
            calPanel.add(calTable, BorderLayout.CENTER);
            calTable.setDefaultRenderer(Integer.class, new CalendarCellRenderer());
            calTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
            calTable.setCellSelectionEnabled(true);
            Enumeration<TableColumn> en = calTable.getColumnModel().getColumns();
            while (en.hasMoreElements()) {
                en.nextElement().setPreferredWidth(calTable.getRowHeight()*2);
            }
            calTable.setRowHeight(Math.round(calTable.getRowHeight()*1.1F));
            calTable.setShowGrid(false);
            head.setReorderingAllowed(false);
            head.setResizingAllowed(false);
            head.setDefaultRenderer(new DefaultTableCellRenderer() {
                @Override
                public boolean isOpaque() {
                    return true;
                }
                @Override
                public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
                    setHorizontalAlignment(SwingConstants.CENTER);
                    return super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
                }
            });
            head.setBorder(new MatteBorder(0,0,1,0, Color.LIGHT_GRAY));
            calTable.getSelectionModel().addListSelectionListener(e -> {
                if (armed && !e.getValueIsAdjusting()) {
                    onCalSelect();
                }
            });
            calTable.getColumnModel().getSelectionModel().addListSelectionListener(e -> {
                if (armed && !e.getValueIsAdjusting()) {
                    onCalSelect();
                }
            });
            resetFromDate();
        }
        
        // Time:
        
        if (time != null) {
            
            JPanel timePanel = new JPanel();
            root.add(timePanel);
            timePanel.setLayout(new GridLayout(1, 3, HSPACE, VSPACE));
            timePanel.setBorder(BorderFactory.createTitledBorder("Time"));
            
            Box box = Box.createHorizontalBox();
            timePanel.add(box);
            box.setBorder(BorderFactory.createTitledBorder("Hours"));
            JSpinner hSpinner = new Spinner(new CycleSpinnerModel(time.getHour(), 0, 23, 1));
            box.add(hSpinner);
            hSpinner.addChangeListener(e -> time = time.withHour((Integer) hSpinner.getValue()));
            
            box = Box.createHorizontalBox();
            timePanel.add(box);
            box.setBorder(BorderFactory.createTitledBorder("Minutes"));
            JSpinner mSpinner = new Spinner(new CycleSpinnerModel(time.getMinute(), 0, 59, 1));
            box.add(mSpinner);
            mSpinner.addChangeListener(e -> time = time.withMinute((Integer) mSpinner.getValue()));
            
            box = Box.createHorizontalBox();
            timePanel.add(box);
            box.setBorder(BorderFactory.createTitledBorder("Seconds"));
            JSpinner sSpinner = new Spinner(new CycleSpinnerModel(time.getSecond(), 0, 59, 1));
            box.add(sSpinner);
            sSpinner.addChangeListener(e -> time = time.withSecond((Integer) sSpinner.getValue()));
        }
        
        // Buttons:
        
        Box buttonsPanel = Box.createHorizontalBox();
        root.add(buttonsPanel);
        buttonsPanel.setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
        buttonsPanel.add(Box.createHorizontalGlue());
        JButton button = new JButton("Cancel");
        buttonsPanel.add(button);
        button.addActionListener(e -> {
            date = null;
            time = null;
            dispose();
        });
        buttonsPanel.add(Box.createHorizontalGlue());
        button = new JButton("  OK  ");
        buttonsPanel.add(button);
        button.addActionListener(e -> {
            dispose();
        });
        buttonsPanel.add(Box.createHorizontalGlue());        
    }
    
    /**
     * Displays a dialog for selecting date and time.
     * 
     * @param seed Initial value. If {@code null}, current date and time.
     * @param title Dialog title. Automatically generated if {@code null}.
     * @param parent Parent component.
     * @return Chosen date and time, or {@code null} if the dialog has been canceled by the user.
     */
    static public LocalDateTime selectDateTime(LocalDateTime seed, String title, Component parent) {
        if (seed == null) seed = LocalDateTime.now().withMinute(0).withSecond(0).withNano(0);
        if (title == null) title = "Select date and time";
        DateTimePicker dialog = new DateTimePicker(seed.toLocalDate(), seed.toLocalTime(), title, parent);
        dialog.setSize(dialog.getPreferredSize());
        dialog.pack();
        dialog.setLocationRelativeTo(parent);
        dialog.setVisible(true);
        if (dialog.date == null || dialog.time == null) return null;
        return LocalDateTime.of(dialog.date, dialog.time);
    }
    
    /**
     * Displays a dialog for selecting date.
     * 
     * @param seed Initial value. If {@code null}, today.
     * @param title Dialog title. Automatically generated if {@code null}.
     * @param parent Parent component.
     * @return Chosen date, or {@code null} if the dialog has been canceled by the user.
     */
    static public LocalDate selectDate(LocalDate seed, String title, Component parent) {
        if (seed == null) seed = LocalDate.now();
        if (title == null) title = "Select date";
        DateTimePicker dialog = new DateTimePicker(seed, null, title, parent);
        dialog.setSize(dialog.getPreferredSize());
        dialog.pack();
        dialog.setLocationRelativeTo(parent);
        dialog.setVisible(true);
        return dialog.date;
    }
    
    /**
     * Displays a dialog for selecting time.
     * 
     * @param seed Initial value. If {@code null}, current time.
     * @param title Dialog title. Automatically generated if {@code null}.
     * @param parent Parent component.
     * @return Chosen time, or {@code null} if the dialog has been canceled by the user.
     */
    static public LocalTime selectTime(LocalTime seed, String title, Component parent) {
        if (seed == null) seed = LocalTime.now().withMinute(0).withSecond(0).withNano(0);
        if (title == null) title = "Select time";
        DateTimePicker dialog = new DateTimePicker(null, seed, title, parent);
        dialog.setSize(dialog.getPreferredSize());
        dialog.pack();
        dialog.setLocationRelativeTo(parent);
        dialog.setVisible(true);
        return dialog.time;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void onCalSelect() {
        int row = calTable.getSelectedRow();
        int column = calTable.getSelectedColumn();
        if (row == -1 || column == -1) {
            resetFromDate();
        } else {
            int day = calModel.getValueAt(row, column);
            if (Math.abs(day) > 100) day /= 100;
            if (day > 0) {
                date = date.withDayOfMonth(day);
            } else {
                day = -day;
                date = date.withDayOfMonth(day);
                if (day < 15) {
                    date = date.plusMonths(1);
                } else {
                    date = date.minusMonths(1);
                }
                resetFromDate();
            }
        }
    }
    
    private void resetFromDate() {
        armed = false;
        monthCombo.setSelectedIndex(date.getMonthValue()-1);
        yearSpinner.setValue(date.getYear());
        calModel.update();
        armed = true;
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private class CalendarModel extends AbstractTableModel {
        
        private int prevLast; // index (row*7 + column) of the last day of the previous month
        private int last; // index of the last day of this month
        private int v00; // value for index 0
        private int today;

        @Override
        public int getRowCount() {
            return 6;
        }

        @Override
        public int getColumnCount() {
            return 7;
        }

        @Override
        public Integer getValueAt(int row, int column) {
            int out;
            int i = row * 7 + column;
            if (i <= prevLast) {
                out = - (v00 + i);
            } else if (i > last) {
                out = last - i;
            } else {
                int date = i - prevLast;
                out = date; 
            }
            if (i == today) out *= 100;
            return out;
        }

        @Override
        public String getColumnName(int column) {
            return DayOfWeek.of(column+1).getDisplayName(TextStyle.NARROW, Locale.US);
        }

        @Override
        public Class<?> getColumnClass(int columnIndex) {
            return Integer.class;
        }
        
        void update() {
            int dayOfWeek = date.withDayOfMonth(1).getDayOfWeek().getValue();
            if (dayOfWeek == 1) { // Monday
                prevLast = 6;
            } else {
                prevLast = dayOfWeek - 2;
            }
            v00 = date.withDayOfMonth(1).minusDays(1).getDayOfMonth() - prevLast;
            last = prevLast + date.getMonth().length(date.isLeapYear());
            
            LocalDate dNow = LocalDate.now();
            LocalDate d00 = date.minusMonths(1).withDayOfMonth(v00);
            long todayIndex = ChronoUnit.DAYS.between(d00, dNow);
            if (todayIndex > -1 && todayIndex < 42) {
                today = (int) todayIndex;
            } else {
                today = -1;
            }

            fireTableDataChanged();
            
            int selected = date.getDayOfMonth() + prevLast;
            int row = selected/7;
            int column = selected%7;
            calTable.getSelectionModel().setValueIsAdjusting(true);
            calTable.getColumnModel().getSelectionModel().setValueIsAdjusting(true);
            calTable.getSelectionModel().setSelectionInterval(row, row);
            calTable.getColumnModel().getSelectionModel().setSelectionInterval(column, column);
            calTable.getSelectionModel().setValueIsAdjusting(true);
            calTable.getColumnModel().getSelectionModel().setValueIsAdjusting(true);
        }
    }
    
    static private final class CalendarCellRenderer extends DefaultTableCellRenderer {
        
        CalendarCellRenderer() {
            setHorizontalAlignment(SwingConstants.CENTER);
        }

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            int v = (Integer) value;
            boolean thisMonth = v > 0;
            boolean today = false;
            if (Math.abs(v) >= 100) {
                today = true;
                v /= 100;
            }
            if (!thisMonth) v = -v;
            super.getTableCellRendererComponent(table, v, isSelected, hasFocus, row, column);
            this.setEnabled(thisMonth);
            if (today) {
                setBorder(BorderFactory.createLineBorder(Color.BLACK));
            } else {
                setBorder(null);
            }
            return this;
        }

    }
    
    static private final class Spinner extends JSpinner {
        
        Spinner(SpinnerNumberModel model) {
            super(model);
            JSpinner.NumberEditor editor = new JSpinner.NumberEditor(this, "00");
            JFormattedTextField field = editor.getTextField();
            FontMetrics fm = field.getFontMetrics(field.getFont());
            int pad = fm.charWidth('0')/2;
//            field.setEditable(false); // way to make spinner not editable
            editor.getTextField().setBorder(BorderFactory.createEmptyBorder(pad, pad, pad, pad));
            setEditor(editor);
        }

        @Override
        public Dimension getMaximumSize() {
            return new Dimension(super.getMaximumSize().width, super.getPreferredSize().height);
        }
        
    }
    
    static private final class CycleSpinnerModel extends SpinnerNumberModel {
        
        CycleSpinnerModel(Number value, Comparable minimum, Comparable maximum, Number stepSize) {
            super(value, minimum, maximum, stepSize);
        }

        @Override
        public Object getPreviousValue() {
            Object out = super.getPreviousValue();
            return out == null ? getMaximum() : out;
        }

        @Override
        public Object getNextValue() {
            Object out = super.getNextValue();
            return out == null ? getMinimum() : out;
        }
        
    }
    
    
// -- Testing : ----------------------------------------------------------------
    
    static public void main(String... args) {
        SwingUtilities.invokeLater(() -> {
            LocalDateTime time = LocalDateTime.now();
            while (time != null) {
                time = selectDateTime(time, null, null);
                System.out.println(time);
            }
        });
    }
    
}
