package org.lsst.ccs.gconsole.plugins.trending.timeselection;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Comparator;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Time window used by trending plots.
 * 
 * The standard format for string representation of time window edges is
 * "MM/dd/yyyy HH:mm:ss" for fixed edges and "now - XXX" (where XXX is time in milliseconds)
 * for sliding edges.
 *
 * @author turri
 * @author onoprien
 */
final public class TimeWindow {
    
// -- Public fields : ----------------------------------------------------------
    
    static public final String DATE_PATTERN = "MM/dd/yyyy HH:mm:ss";
    static public final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat(DATE_PATTERN);
    static public final Pattern EDGE_PATTERN = Pattern.compile("\\s*now\\s*(?:-\\s*(\\d+)\\s*)?");
    static private final String DELIMITER_RE = "\\$";
    static private final String DELIMITER = "$";
    
    private String name;    
    private long lower; // start, millis. If negative, absolute value is duration
    private long upper; // end, millis. Zero = now. If negative, absolute value is duration.
    
// -- Private parts : ----------------------------------------------------------
    
    private boolean persist; // true if this time window should be persisted between sessions
    private long lastUsed; // time when this window was last selected, millis
    
// -- Construction : -----------------------------------------------------------
    
    /**
     * Constructs time window instance.
     * 
     * @param name Time window name.
     * @param start Time window start, in seconds since Epoch.
     * @param end Time window end, in seconds since Epoch.
     * @param persistent True is this time window will be persisted between console sessions.
     * @throws IllegalArgumentException If the combination of arguments is illegal.
     */
    public TimeWindow(String name, long start, long end, boolean persistent) {
        set(name, start, end, persistent);
        lastUsed = System.currentTimeMillis();
    }
    
    /**
     * Constructs time window instance.
     * 
     * @param name Time window name.
     * @param start Time window start, in standard format.
     * @param end Time window end, in standard format.
     * @param persistent True is this time window will be persisted between console sessions.
     * @throws IllegalArgumentException If the combination of arguments is illegal, or the strings cannot be parsed.
     */
    public TimeWindow(String name, String start, String end, boolean persistent) {
        this(name, parseEdge(start)/1000L, parseEdge(end)/1000L, persistent);
    }
    
    /**
     * Constructs a named time window with no start and end values.
     * Used as marker instances.
     * 
     * @param name Time window name.
     */
    public TimeWindow(String name) {
        this.name = name;
    }
    
    /**
     * Constructs anonymous, non-persistent time window with the same start and end values as the seed.
     * Used to provide initial values for dialogs.
     * 
     * @param seed Time window that provides start and end values.
     */
    public TimeWindow(TimeWindow seed) {
        lower = seed.lower;
        upper = seed.upper;
        lastUsed = System.currentTimeMillis();
    }
    
    void set(String name, long start, long end, boolean persistent) {
        this.name = name;
        lower = start * 1000L;
        upper = end * 1000L;
        if ((upper < 0L && lower < 0L) || lower == 0L || getLength() <= 0L) {
            throw new IllegalArgumentException("Invalid time range bounds: "+ start +", "+ end);
        }
        persist = persistent;
    }
    
// -- Setters and getters : ----------------------------------------------------
    
    /** Returns the name of this time window. */
    public String getName() {
        return name;
    }
    
    /** Sets the name of this time window. */
    public void setName(String name) {
        this.name = name;
    }
    
    /** Returns <tt>true</tt> if both edges do not depend on current time. */
    public boolean isFixed() {
        return upper != 0L;
    }
    
    /** Returns <tt>true</tt> if lower edge does not depend on current time. */
    public boolean isLowerEdgeFixed() {
        return lower > 0L || upper != 0L;
    }
    
    /** Returns <tt>true</tt> if upper edge does not depend on current time. */
    public boolean isUpperEdgeFixed() {
        return upper != 0L;
    }
    
    /** Returns the difference between the lower edge of this time window and the current time, in milliseconds. */
    public long getLowerEdgeDelay() {
        if (lower > 0L) {
            return System.currentTimeMillis() - lower;
        } else if (upper == 0L) {
            return -lower;
        } else {
            return (System.currentTimeMillis() - upper) - lower;
        }
    }
    
    /** Returns the difference between the upper edge of this time window and the current time, in milliseconds. */
    public long getUpperEdgeDelay() {
        if (upper == 0L) {
            return 0L;
        } else if (upper > 0L) {
            return System.currentTimeMillis() - upper;
        } else {
            return (System.currentTimeMillis() - lower) + upper;
        }
    }
    
    /** Returns the current value of the lower edge of this window, in milliseconds since Epoch. */
    public long getLowerEdge() {
        return getLowerEdge(System.currentTimeMillis());
    }
    
    /** Returns the current value of the upper edge of this window, in milliseconds since Epoch. */
    public long getUpperEdge() {
        return getUpperEdge(System.currentTimeMillis());
    }
    
    /** Returns the lower edge of this time window at the specified time, in milliseconds since Epoch. */
    public long getLowerEdge(long currentTime) {
        if (lower > 0L) {
            return lower;
        } else if (upper == 0L) {
            return currentTime + lower;
        } else {
            return upper + lower;
        }
    }
    
    /** Returns the upper edge of this time window at the specified time, in milliseconds since Epoch. */
    public long getUpperEdge(long currentTime) {
        if (upper > 0L) {
            return upper;
        } else if (upper == 0L) {
            return currentTime;
        } else {
            return lower - upper;
        }
    }
    
    /** Returns the lower edge of this time window in standard format. */
    public String getLowerEdgeString() {
        if (lower > 0L) {
            return DATE_FORMAT.format(new Date(lower));
        } else if (upper > 0L) {
            return DATE_FORMAT.format(new Date(upper + lower));
        } else {
            return "now - "+ (-lower/1000L);
        }
     }
    
    /** Returns the upper edge of this time window in standard format. */
    public String getUpperEdgeString() {
        if (upper == 0L) {
            return "now";
        } else if (upper > 0L) {
            return DATE_FORMAT.format(new Date(upper));
        } else {
            return DATE_FORMAT.format(new Date(lower - upper));
        }
    }
    
    /**
     * Sets lower edge of this time window.
     * @param edge Edge value in standard format.
     * @throws IllegalArgumentException if the specified string is not in standard format.
     */
    public void setLowerEdge(String edge) {
        lower = parseEdge(edge);
    }
    
    /**
     * Sets upper edge of this time window.
     * @param edge Edge value in standard format.
     * @throws IllegalArgumentException if the specified string is not in standard format.
     */
    public void setUpperEdge(String edge) {
        upper = parseEdge(edge);
    }
    
    /**
     * Returns time window length.
     * If the length is variable, current length is returned.
     * @return Window duration in milliseconds.
     */
    public long getLength() {
        if (upper == 0L) {
            return lower > 0L ? System.currentTimeMillis() - lower : -lower;
        } else if (upper > 0L) {
            return lower > 0L ? upper - lower : -lower;
        } else {
            return -upper;
        }
    }
    
    /** Returns <tt>true</tt> if this time window should be persisted between sessions. */
    public boolean isPersistent() {
        return persist;
    }
    
    /** Sets the flag that indicate whether or not this time window should be persisted between sessions. */
    public void setPersistent(boolean persistent) {
        persist = persistent;
    }
    
    /** Returns the time in milliseconds when this time window was last selected. */
    public long getLastUseTime() {
        return lastUsed;
    }
    
    /** Sets the time when this time window was last selected to the current system time. */
    public void touch() {
        lastUsed = System.currentTimeMillis();
    }

    /** Returns user-readable string representation of this time window. */
    @Override
    public String toString() {
        return name +" : "+ getLowerEdgeString() +" through "+ getUpperEdgeString();
    }
    
    /** Returns string representation of this time window used for saving it in application properties. */
    public String toCompressedString() {
        return name + DELIMITER + getLowerEdgeString() + DELIMITER + getUpperEdgeString();
    }
    
    long[] getBounds() {
        return new long[] {lower, upper};
    }

    
// -- Utility methods : --------------------------------------------------------
    
    /**
     * Parses string representation of a time window edge in standard format.
     * @return Time window edge represented as a long.
     */
    static public long parseEdge(String edgeString) {
        try {
            Date d = DATE_FORMAT.parse(edgeString);
            return d.getTime();
        } catch (ParseException x) {
        }
        try {
            Matcher m = EDGE_PATTERN.matcher(edgeString);
            m.matches();
            String delay = m.group(1);
            return delay == null ? 0L : - Long.parseLong(delay)*1000;
        } catch (NumberFormatException|IllegalStateException x) {
            throw new IllegalArgumentException(x);
        }
    }
    
    /** Parses string representation of this time window used for saving it in application properties. */
    static public TimeWindow parseCompressedString(String s) {
        String[] tokens = s.split(DELIMITER_RE);
        try {
            return new TimeWindow(tokens[0], tokens[1], tokens[2], true);
        } catch (RuntimeException x) {
            throw new IllegalArgumentException(x);
        }
    }

    
// -- Comparators : ------------------------------------------------------------
    
    /** Returns comparator that compares time windows by names. */
    static public Comparator<TimeWindow> compareByName() {
        return new Comparator<TimeWindow>() {
            public int compare(TimeWindow o1, TimeWindow o2) {
                return o1.name.compareTo(o2.name);
            }
        };
    }
    
    /** Returns comparator that compares time windows by last use times, most recent first. */
    static public Comparator compareByTime() {
        return new Comparator() {
            public int compare(Object o1, Object o2) {
                return (int) Math.signum(((TimeWindow)o2).getLastUseTime() - ((TimeWindow)o1).getLastUseTime());
            }
        };
    }
    
    
}
