package org.lsst.ccs.gconsole.base.panel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.gconsole.base.Console;

/**
 * Interface to be implemented by graphical components that will be passed to {@code PanelManager} for displaying as data pages.
 * This interface adds standard items to the popup menu. If that's all that's needed, no methods need to be implemented. 
 * The menu can be further customized by implementing {@code modifyPopup(...)} method.
 * <p>
 * <i>This is a kludgy workaround made necessary by limitations of freehep application framework.</i>
 *
 * @author onoprien
 */
public interface DataPage extends HasPopupItems {

    /**
     * This method should not be overridden.
     * Adds standard items to the popup menu.
     * 
     * @param menu Menu to be modified.
     * @param component Component that emitted the mouse event.
     * @param point Point of the event,
     * @return Modified menu.
     */
    @Override
    default JPopupMenu modifyPopupMenu(JPopupMenu menu, Component component, Point point) {
        menu = modifyPopup(menu, component, point);
        if (this instanceof JComponent) {
            JComponent panel = (JComponent) this;
            PanelManager panMan = Console.getConsole().getPanelManager();
            boolean firstItem = true;
            Panel[] keys = {Panel.ON_SAVE_AS, Panel.ON_EDIT};
            String[] names = {"Save Page As...", "Edit Page..."};
            for (int i=0; i<keys.length; i++) {
                Object o = panMan.get(panel, keys[i]);
                if (o != null) {
                    try {
                        Consumer<JComponent> task = (Consumer<JComponent>) o;
                        if (firstItem) {
                            firstItem = false;
                            menu.addSeparator();
                        }
                        menu.add(new AbstractAction(names[i]) {
                            @Override
                            public void actionPerformed(ActionEvent e) {
                                task.accept(panel);
                            }
                        });
                    } catch (ClassCastException x) {
                    }
                }
                
            }
        }
        return menu;
    }
    
    /**
     * Override to add extra items to the popup menu.
     * This method is called after all modifications by components below, but before adding the standard items.
     * 
     * @param menu Menu to be modified.
     * @param component Component that emitted the mouse event.
     * @param point Point of the event,
     * @return Modified menu.
     */
    default JPopupMenu modifyPopup(JPopupMenu menu, Component component, Point point) {
        return menu;
    }
    
    /**
     * Wraps the provided component to make it implement {@code DataPage}.
     * 
     * @param page Component to wrap.
     * @return Component to display as the graphical console data page.
     */
    static public JComponent wrap(JComponent page) {
        if (page instanceof DataPage) {
            return page;
        } else {
            Wrapper w = new Wrapper();
            w.init(page);
            return w;
        }
    }
    
    static class Wrapper extends JPanel implements DataPage {
        Wrapper() {
            super(new BorderLayout());
        }
        void init(JComponent page) {
            add(page, BorderLayout.CENTER);
        }
    }
    
}
