package org.lsst.ccs.gconsole.agent.command;

import java.time.Duration;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.lsst.ccs.Agent;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Utility class for sending commands to remote subsystems and processing responses.
 * The sender can be configured by calling its setter methods.
 * <p>
 * All methods that send commands return immediately, without waiting for the command to be
 * processed by the messaging service or the target subsystem. Synchronous command execution can be
 * achieved by calling {@link CommandTask#get get()} or {@link CommandTask#getResult() getResult()}
 * on the {@link CommandTask} returned by this method.
 * <p>
 * This class is thread-safe, all methods can be called on any thread.
 *
 * @author onoprien
 */
public class CommandSender {

// -- Fields : -----------------------------------------------------------------
    
    final AgentMessagingLayer messenger;
    final Executor callbackExecutor;
    final Executor workerExecutor;
    final Scheduler timer;
            
    private volatile Duration defaultTimeout = Duration.ofSeconds(10);
    private volatile CommandHandle defaultCommandHandle;


// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Creates an instance of {@code CommandSender}.
     * 
     * @param messenger Messaging access point. If {@code null}, the environment messaging access will be used.
     * @param callbackExecutor Executor for running {@link CommandHandle} methods. If {@code null}, a dedicated single-threaded executor will be created.
     * @param workerExecutor Executor for running potentially time-consuming tasks. If {@code null}, a dedicated executor will be created.
     * @param timer Scheduling executor for timing commands.
     */
    public CommandSender(AgentMessagingLayer messenger, Executor callbackExecutor, Executor workerExecutor, Scheduler timer) {
        
        if (messenger == null) {
            messenger = Agent.getEnvironmentMessagingAccess();
        }
        this.messenger = messenger;
        
        if (callbackExecutor == null || workerExecutor == null) {
            ThreadFactory threadFactory = new ThreadFactory() {
                private final ThreadFactory delegate = Executors.defaultThreadFactory();
                @Override
                public Thread newThread(Runnable r) {
                    Thread thread = delegate.newThread(r);
                    thread.setDaemon(true);
                    return thread;
                }
                
            };
            if (callbackExecutor == null) {
                callbackExecutor = Executors.newSingleThreadExecutor(threadFactory);
            }
            if (workerExecutor == null) {
                workerExecutor = new ThreadPoolExecutor(0, 2, 60L, TimeUnit.SECONDS, new SynchronousQueue<>(), threadFactory);
            }
        }
        this.callbackExecutor = callbackExecutor;
        this.workerExecutor = workerExecutor;
        this.timer = timer;
    }
    
    
// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the default command timeout for this sender.
     * The specified timeout will be used for command execution requests that do not provide their own timeout values.
     * If this method has never been called, the default timeout value is 10 seconds.
     * 
     * @param defaultTimeout Default timeout.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @throws IllegalArgumentException If the supplied value is {@code null}.
     */
    public void setTimeout(Duration defaultTimeout) {
        if (defaultTimeout == null) {
            throw new IllegalArgumentException("Illegal default timeout value: "+ defaultTimeout);
        }
        this.defaultTimeout = defaultTimeout;
    }
    
    /**
     * Sets the default command handle for this sender.
     * The specified handle will be used for command execution requests that do not provide their own handle.
     * If this method has never been called, the default handle does nothing.
     * 
     * @param defaultCommandHandle Default command handle.
     */
    public void setCommandHandle(CommandHandle defaultCommandHandle) {
        this.defaultCommandHandle = defaultCommandHandle;
    }
    

// -- Legacy methods for sending commands : ------------------------------------
    
    /**
     * Sends a command to a remote subsystem.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param command Command in "subsystem/component0/.../componentN/commandName" format.
     * @param args Command arguments.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask execute(CommandHandle handle, Duration timeout, String command, Object... args) {
        if (handle == null) handle = defaultCommandHandle;
        int i = command.lastIndexOf("/");
        String destination = command.substring(0, i);
        command = command.substring(i + 1);
        CommandTask task = new CommandTask(this, handle, timeout, destination, command, args);
        task.send();
        return task;
    }
    
    /**
     * Sends a command to a remote subsystem, using the default {@code CommandHandle} of this sender to process responses.
     * If no default handle has been set, responses will be ignored.
     * 
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param command Command in "subsystem/component0/.../componentN/commandName"} format.
     * @param args Command arguments.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask execute(Duration timeout, String command, Object... args) {
        return execute(defaultCommandHandle, timeout, command, args);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout.
     * If no default timeout has been set for this sender, the timeout is 10 seconds,
     * unless the target subsystem returns a custom timeout with ACK.
     * 
     * @param handle Handle for processing the command responses.
     *               Cannot be {@code null}. If no processing is required, use {@code CommandHandle.NONE}.
     * @param command Command in "subsystem/component0/.../componentN/commandName" format.
     * @param args Command arguments.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask execute(CommandHandle handle, String command, Object... args) {
        return execute(handle, defaultTimeout, command, args);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout and the default callback handle.
     * The default timeout is 10 seconds, unless the target subsystem returns a custom timeout with ACK.
     * If no default handle has been set, responses will be ignored.
     * 
     * @param command Command in "subsystem/component0/.../componentN/commandName" format.
     * @param args Command arguments.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask execute(String command, Object... args) {
        return execute(defaultCommandHandle, defaultTimeout, command, args);
    }

    
// -- Sending commands with raw arguments : ------------------------------------
    
    /**
     * Sends a command to a remote subsystem.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command name.
     * @param args Command arguments as objects that should be passed directly to the command implementation.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask sendRaw(CommandHandle handle, Duration timeout, String destination, String command, Object... args) {
        if (handle == null) handle = defaultCommandHandle;
        if (timeout == null) throw new IllegalArgumentException();
        CommandTask task = new CommandTask(this, handle, timeout, destination, command, args);
        task.send();
        return task;
    }
    
    /**
     * Sends a command to a remote subsystem, using the default {@code CommandHandle} of this sender to process responses.
     * If no default handle has been set, responses are ignored.
     * 
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command name.
     * @param args Command arguments as objects that should be passed directly to the command implementation.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask sendRaw(Duration timeout, String destination, String command, Object... args) {
        return sendRaw(defaultCommandHandle, timeout, destination, command, args);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout.
     * If no default timeout has been set for this sender, the timeout is 10 seconds,
     * unless the target subsystem returns a custom timeout with ACK.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command name.
     * @param args Command arguments as objects that should be passed directly to the command implementation.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask sendRaw(CommandHandle handle, String destination, String command, Object... args) {
        return sendRaw(handle, defaultTimeout, destination, command, args);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout and the default callback handle.
     * If no default timeout has been set for this sender, the timeout is 10 seconds,
     * unless the target subsystem returns a custom timeout with ACK.
     * If no default handle has been set, responses will be ignored.
     * 
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command name.
     * @param args Command arguments as objects that should be passed directly to the command implementation.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask sendRaw(String destination, String command, Object... args) {
        return sendRaw(defaultCommandHandle, defaultTimeout, destination, command, args);
    }

    
// -- Sending commands with arguments encoded as strings : ---------------------
    
    /**
     * Sends a command to a remote subsystem.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command name.
     * @param args Command arguments encoded as strings.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask sendEncoded(CommandHandle handle, Duration timeout, String destination, String command, String... args) {
        if (handle == null) handle = defaultCommandHandle;
        if (args == null) throw new IllegalArgumentException();
        StringBuilder sb = new StringBuilder(command);
        for (String arg : args) {
            if (arg.isEmpty() || (arg.contains(" ") && !(arg.startsWith(" \"")) && !(arg.startsWith(" \"")))) {
                sb.append(" ").append(" \"").append(arg).append("\"");
            } else {
                sb.append(" ").append(arg);
            }
        }
        CommandTask task = new CommandTask(destination, sb.toString(), this, handle, timeout);
        task.send();
        return task;
    }
    
    /**
     * Sends a command to a remote subsystem, using the default {@code CommandHandle} of this sender to process responses.
     * If no default handle has been set, responses are ignored.
     * 
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command name.
     * @param args Command arguments encoded as strings.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask sendEncoded(Duration timeout, String destination, String command, String... args) {
        return sendEncoded(defaultCommandHandle, timeout, destination, command, args);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout.
     * If no default timeout has been set for this sender, the timeout is 10 seconds,
     * unless the target subsystem returns a custom timeout with ACK.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command name.
     * @param args Command arguments encoded as strings.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask sendEncoded(CommandHandle handle, String destination, String command, String... args) {
        return sendEncoded(handle, defaultTimeout, destination, command, args);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout and the default callback handle.
     * If no default timeout has been set for this sender, the timeout is 10 seconds,
     * unless the target subsystem returns a custom timeout with ACK.
     * If no default handle has been set, responses will be ignored.
     * 
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command name.
     * @param args Command arguments encoded as strings.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask sendEncoded(String destination, String command, String... args) {
        return sendEncoded(defaultCommandHandle, defaultTimeout, destination, command, args);
    }
        
    
// -- Sending commands with destination, command, arguments : ------------------
    
    /**
     * Sends a command to a remote subsystem.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param commandName Command name.
     * @param arguments Command arguments encoded as a single string.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(CommandHandle handle, Duration timeout, String destination, String commandName, String arguments) {
        if (handle == null) handle = defaultCommandHandle;
        String command = arguments == null || arguments.isEmpty() ? commandName : commandName +" "+ arguments;
        CommandTask task = new CommandTask(destination, command, this, handle, timeout);
        task.send();
        return task;
    }
    
    /**
     * Sends a command to a remote subsystem, using the default {@code CommandHandle} of this sender to process responses.
     * If no default handle has been set, responses are ignored.
     * 
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param commandName Command name.
     * @param arguments Command arguments encoded as strings.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(Duration timeout, String destination, String commandName, String arguments) {
        return send(defaultCommandHandle, timeout, destination, commandName, arguments);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout.
     * If no default timeout has been set for this sender, the timeout is 10 seconds,
     * unless the target subsystem returns a custom timeout with ACK.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param commandName Command name.
     * @param arguments Command arguments encoded as a single string.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(CommandHandle handle, String destination, String commandName, String arguments) {
        return send(handle, defaultTimeout, destination, commandName, arguments);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout and the default callback handle.
     * If no default timeout has been set for this sender, the timeout is 10 seconds,
     * unless the target subsystem returns a custom timeout with ACK.
     * If no default handle has been set, responses will be ignored.
     * 
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param commandName Command name.
     * @param arguments Command arguments encoded as a single string.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(String destination, String commandName, String arguments) {
        return send(defaultCommandHandle, defaultTimeout, destination, commandName, arguments);
    }
        
    
// -- Sending commands with destination, command : -----------------------------
    
    /**
     * Sends a command to a remote subsystem.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command with arguments, encoded as a single string.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(CommandHandle handle, Duration timeout, String destination, String command) {
        if (handle == null) handle = defaultCommandHandle;
        CommandTask task = new CommandTask(destination, command, this, handle, timeout);
        task.send();
        return task;
    }
    
    /**
     * Sends a command to a remote subsystem, using the default {@code CommandHandle} of this sender to process responses.
     * If no default handle has been set, responses are ignored.
     * 
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command with arguments, encoded as a single string.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(Duration timeout, String destination, String command) {
        return send(defaultCommandHandle, timeout, destination, command);
    }
        
    /**
     * Sends a command to a remote subsystem with the default timeout.
     * If no default timeout has been set for this sender, the timeout is 10 seconds,
     * unless the target subsystem returns a custom timeout with ACK.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command with arguments, encoded as a single string.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(CommandHandle handle, String destination, String command) {
        return send(handle, defaultTimeout, destination, command);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout and the default callback handle.
     * If no default timeout has been set for this sender, the timeout is 10 seconds,
     * unless the target subsystem returns a custom timeout with ACK.
     * If no default handle has been set, responses will be ignored.
     * 
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     * @param command Command with arguments, encoded as a single string.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(String destination, String command) {
        return send(defaultCommandHandle, defaultTimeout, destination, command);
    }


// -- Sending commands with shell command : ------------------------------------
    
    /**
     * Sends a command to a remote subsystem.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param command Command with destination and arguments, as it would be given to a command shell.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(CommandHandle handle, Duration timeout, String command) {
        if (handle == null) handle = defaultCommandHandle;
        String[] ss = command.split("\\s", 2);
        CommandTask task = new CommandTask(ss[0], ss[1], this, handle, timeout);
        task.send();
        return task;
    }
    
    /**
     * Sends a command to a remote subsystem, using the default {@code CommandHandle} of this sender to process responses.
     * If no default handle has been set, responses are ignored.
     * 
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param command Command with destination and arguments, as it would be given to a command shell.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(Duration timeout, String command) {
        return send(defaultCommandHandle, timeout, command);
    }
        
    /**
     * Sends a command to a remote subsystem with the default timeout.
     * If no default timeout has been set for this sender, the timeout is 10 seconds,
     * unless the target subsystem returns a custom timeout with ACK.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     * @param command Command with destination and arguments, as it would be given to a command shell.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(CommandHandle handle, String command) {
        return send(handle, defaultTimeout, command);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout and the default callback handle.
     * If no default timeout has been set for this sender, the timeout is 10 seconds,
     * unless the target subsystem returns a custom timeout with ACK.
     * If no default handle has been set, responses will be ignored.
     * 
     * @param command Command with destination and arguments, as it would be given to a command shell.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask send(String command) {
        return send(defaultCommandHandle, defaultTimeout, command);
    }

}
