package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.*;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.ListCellRenderer;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.gconsole.base.filter.AbstractChannelsFilter;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.base.filter.PersistableAgentChannelsFilter;
import org.lsst.ccs.gconsole.services.command.CommandService;

/**
 * Monitoring data view that displays basic info for the monitored subsystems and provides monitoring controls.
 * For each subsystem, the panel displays:<ul>
 * <li>Time stamp of the last update.
 * <li>Subsystem name is color-coded to reflect configuration state.
 * <li>Save and reset buttons.
 * </ul>
 *
 * @author onoprien
 */
public class ConfigView extends AbstractMonitorView {

// -- Fields : -----------------------------------------------------------------
    
    private final Descriptor descriptor = new Descriptor();
    
    static private final String STATE_PATH = "/state/ConfigurationState";
    static private final String TIME_PATH = "/runtimeInfo/current time";
    
    private List<String> agentList;
    
    private final AgentDataALL ALL;
    private final TreeMap<String,AgentData> agentsData = new TreeMap<>();
    private AgentData[] model;
    
    private final GUI panel;

// -- Life cycle : -------------------------------------------------------------
    
//    @Create(category = MonitorView.CATEGORY,
//            name = "Configuration View",
//            path = "Built-In/Configuration",
//            description = "Monitoring data view that displays configuration state of currently subsystems. The view provides controls for saving and resetting configuration.")
    public ConfigView() {
        
        ALL = new AgentDataALL();
        
        model = new AgentData[] {ALL};
        
        filter = new AbstractChannelsFilter() {
            @Override
            public List<String> getAgents() {
                return agentList;
            }
            @Override
            public List<String> getOriginChannels() {
                return Arrays.asList(STATE_PATH, TIME_PATH);
            }
        };
        
        panel = new GUI();    
    }
    
    
// -- Wrapping other views : ---------------------------------------------------
    
    /**
     * Decorates an arbitrary view by adding {@code ConfigView} to its bottom.
     * 
     * @param view View to decorate.
     * @return Decorated view.
     */
    static public PersistableMonitorView decorateView(PersistableMonitorView view) {
        CompoundView out = new CompoundView(new JPanel(new BorderLayout()));
        out.addView(view, CompoundView.Mode.INDEPENDENT);
        out.getPanel().add(view.getPanel(), BorderLayout.CENTER);
        ConfigView controlView = new ConfigView();
        AgentChannelsFilter filter = view.getFilter();
        if (filter instanceof PersistableAgentChannelsFilter) {
            controlView.setFilter(filter);
        }
        out.addView(controlView, CompoundView.Mode.INDEPENDENT);
        out.getPanel().add(controlView.getPanel(), BorderLayout.SOUTH);
        return out;
    }
    
    
// -- Implementing MonitorView : -----------------------------------------------

    @Override
    public JComponent getPanel() {
        return panel;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        if (filter == null) {
            agentList = null;
        } else {
            List<String> aa = filter.getAgents();
            agentList = aa == null ? null : new ArrayList<>(aa);
        }
    }

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }
    

// -- Updating : ---------------------------------------------------------------
    
    @Override
    public void addChannels(List<AgentInfo> agents, List<Map.Entry<String,AgentChannel>> channels) {
        channels.forEach(e -> {
            AgentChannel channel = e.getValue();
            String agent = channel.getAgentName();
            AgentData agentData = agentsData.get(agent);
            if (agentData == null) {
                agentData = new AgentData();
                agentsData.put(agent, agentData);
            }
            if (e.getKey().endsWith(STATE_PATH)) {
                agentData.configState = e.getValue();
            } else if (e.getKey().endsWith(TIME_PATH)) {
                agentData.time = e.getValue();
            }
        });
        updateModel();
        panel.reset();
    }

    @Override
    public void removeChannels(List<AgentInfo> agents, List<Map.Entry<String,AgentChannel>> channels) {
        for (AgentInfo a : agents) {
            agentsData.remove(a.getName());
        }
        updateModel();
        panel.reset();
    }

    @Override
    public void updateChannels(List< Map.Entry<String,Map.Entry<AgentChannel,List<String>>> > channels) {
        String selection = panel.getSelectedItem().getName();
        boolean stateChange;
        if (channels == null) {
            stateChange = true;
            panel.updateTime();
        } else {
            stateChange = false;
            for (Map.Entry<String, Map.Entry<AgentChannel, List<String>>> e : channels) {
                AgentChannel channel = e.getValue().getKey();
                String agentName = channel.getAgentName();
                if (channel.getPath().endsWith(STATE_PATH)) {
                    stateChange = true;
                } else {
                    if (agentName.equals(selection)) {
                        panel.updateTime();
                    }
                }
            }
        }
        if (stateChange) {
            combineStates();
            panel.updateState();
        }
        ALL.time = Instant.now();
        if (ALL.getName().equals(selection)) {
            panel.updateTime();
        }
    }

    
// -- GUI panel class : --------------------------------------------------------
    
    private final class GUI extends JPanel {
        
        private final JComboBox<AgentData> agentCombo;
        private final JLabel timeLabel;
//        private final JFormattedTextField freqField;
        private final JButton saveButton;
        private final JButton resetButton;
        
        GUI() {
            setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            add(Box.createVerticalGlue());
            Box box = Box.createHorizontalBox();
            add(box);
            box.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
            
            agentCombo = new JComboBox(model);
            agentCombo.setEditable(false);
            agentCombo.setMaximumSize(new Dimension(Integer.MAX_VALUE, agentCombo.getPreferredSize().height));
            agentCombo.setForeground(Color.red);
            agentCombo.addActionListener(e -> {
                AgentData ad = (AgentData) agentCombo.getSelectedItem();
                if (ad == null) return;
                onSelection(ad);
            });
            ListCellRenderer renderer = agentCombo.getRenderer();
            agentCombo.setRenderer((list, value, index, isSelected, cellHasFocus) -> {
                Component c = renderer.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
                if (!isSelected) {
                    boolean dirty = ConfigurationState.DIRTY.toString().equals(((AgentData)value).getState());
                    if (dirty) c.setForeground(Color.BLUE);
                }
                return c;
            });
            box.add(agentCombo);
            box.add(Box.createRigidArea(Const.HDIM));
            
            box.add(new JLabel("Last updated: "));
            timeLabel = new JLabel("Never");
            box.add(timeLabel);
            box.add(Box.createRigidArea(new Dimension(3*Const.HSPACE, Const.VSPACE)));
            box.add(Box.createHorizontalGlue());
            
            saveButton = new JButton("Save Limits");
            saveButton.setToolTipText("Save current alarm and warning limit values to the current configuration");
            saveButton.addActionListener(e -> {
                AgentData selectedAgent = (AgentData) agentCombo.getSelectedItem();
                if (selectedAgent != null && selectedAgent != ALL && ConfigurationState.DIRTY.toString().equals(selectedAgent.getState())) {
                    CommandService.getService().sendEncoded(selectedAgent.getName(), "saveCategories", "Limits");
                }
            });
            box.add(saveButton);
            box.add(Box.createRigidArea(Const.HDIM));
            resetButton = new JButton("Reset Limits");
            resetButton.setToolTipText("Reset alarm and warning limits to values from the currently loaded configuration");
            resetButton.addActionListener(e -> {
                AgentData selectedAgent = (AgentData) agentCombo.getSelectedItem();
                if (selectedAgent != null && selectedAgent != ALL && ConfigurationState.DIRTY.toString().equals(selectedAgent.getState())) {
                    CommandService.getService().sendEncoded(selectedAgent.getName(), "dropChanges", "Limits");
                }
            });
            box.add(resetButton);
            
            box.setMaximumSize(new Dimension(Integer.MAX_VALUE, box.getPreferredSize().height));
            add(box);

        }
        
        void reset() {
            AgentData selected = getSelectedItem();
            agentCombo.removeAllItems();
            agentCombo.setModel(new DefaultComboBoxModel<>(model) );
            agentCombo.setSelectedItem(selected);
            if (agentCombo.getSelectedIndex() == -1) {
                agentCombo.setSelectedItem(ALL);
            }
        }
        
        private void onSelection(AgentData item) {
            timeLabel.setText(formatTime(item.getTime()));
            Boolean b = ConfigurationState.DIRTY.toString().equals(item.getState());
            saveButton.setEnabled(b && item != ALL);
            resetButton.setEnabled(b && item != ALL);
            boolean dirty = ConfigurationState.DIRTY.toString().equals(item.getState());
            agentCombo.setForeground(dirty ? Color.BLUE : Color.BLACK);
        }
        
        void updateState() {
            onSelection(getSelectedItem());
            agentCombo.repaint();
        }
        
        void updateTime() {
            AgentData item = getSelectedItem();
            if (item != null) {
                timeLabel.setText(formatTime(item.getTime()));
            }
        }
        
        AgentData getSelectedItem() {
            return (AgentData) agentCombo.getSelectedItem();
        }
        
    }

// -- Local methods : ----------------------------------------------------------
    
    private void combineStates() {
        ALL.state = ConfigurationState.CONFIGURED.toString();
        String dirty = ConfigurationState.DIRTY.toString();
        for (AgentData ad : model) {
            if (dirty.equals(ad.getState())) {
                ALL.state = dirty;
                return;
            }
        }
    }
    
    private void updateModel() {
        model = new AgentData[agentsData.size() + 1];
        model[0] = ALL;
        int i = 1;
        for (AgentData item : agentsData.values()) {
            model[i++] = item;
        }
        ALL.time = Instant.now();
        combineStates();
    }
    
    private String formatTime(Instant time) {
        try {
            return String.format("%1$tF %1$tT.", ZonedDateTime.ofInstant(time, ZoneId.systemDefault()));
        } catch (IllegalFormatException | DateTimeException | NullPointerException x) {
            return "unknown.";
        }
    }

    
// -- Local classes : ----------------------------------------------------------
    
    private class AgentData {
        
        AgentChannel configState;
        AgentChannel time;
        
        String getState() {
            if (configState == null) {
                return ConfigurationState.CONFIGURED.toString();
            } else {
                Object v = configState.get();
                return v == null ? ConfigurationState.CONFIGURED.toString() : v.toString();
            }
        }
        
        Instant getTime() {
            return time == null ? null : time.get();
        }
        
        String getName() {
            return time == null ? configState.getAgentName() : time.getAgentName();
        }

        @Override
        public String toString() {
//            return ConfigurationState.DIRTY.toString().equals(getState()) ? "<html><font color=\"blue\">"+ getName() : getName();
            return getName();
        }
        
    }
    
    private class AgentDataALL extends AgentData {
        
        String state = ConfigurationState.CONFIGURED.toString();
        Instant time;

        @Override
        String getName() {
            return "ALL";
        }

        @Override
        Instant getTime() {
            return time;
        }

        @Override
        String getState() {
            return state;
        }
        
        void updateTime(AgentData agent) {
            Instant agentTime = agent.getTime();
            if (agentTime != null && (time == null || agentTime.isAfter(time))) {
                time = agentTime;
            }
        }
        
    }
    
}
