package org.lsst.ccs.gconsole.base.filter;

import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import java.util.*;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusAggregator;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.gconsole.services.persist.Persistable;
import org.lsst.ccs.gconsole.services.persist.Savable;

/**
 * A filter that selects a set of data channels and defines their presentation.
 * Original channels are identified by their unique paths (slash-separated string,
 * the first segment is the subsystem name). This filter maps each original channel
 * to zero or more display paths in {@code [group//]path1/.../pathN} format. The
 * display path determines how the channel is presented in the graphical console.
 * <p>
 * The exact interpretation of filter output depends on the tool using the filter. 
 * In general, a data channel is accepted by this filter if all of the following is true:
 * <ul>
 * <li>{@code getOriginChannels()} method returns {@code null}, or the original channel
 *      satisfies one of the channel selectors returned by this method.
 * <li>{@code getDisplayPaths(AgentChannel channel)} method returns a non-empty list.
 * <li>{@code getDisplayChannels()} method returns {@code null}, or at least one of
 *     the display paths returned by {@code getDisplayPaths(AgentChannel channel)}
 *     is contained by the list returned by {@code getDisplayChannels()}.
 * </ul><p>
 * <p>
 * Some tools might not have access to {@code AgentChannel} instances. They will use<br>
 * {@code getDisplayPaths(String originalPath)} and<br>
 * {@code static List<String> getDisplayPaths(AgentChannelsFilter filter, String originalPath)}<br>
 * methods instead.
 * <p>
 * {@code List<String> getOriginChannels()} may return channel selectors of several types (note
 * that interpretation of some selectors depends on the list returned by {@code getAgents()} method):
 * <ul>
 * <li>Complete path.
 * <li>Path template in the "{@code [agent]/[partial path][/]}" format. 
 *     Templates that omit the agent name are expanded against all
 *     agents whose names are in the {@code getAgents()} list (or all agents present
 *     on the buses if that list is {@code null}). Templates that end with "/"
 *     match all channels that start with "{@code agent/partial path/}".
 * <li>Arbitrary regular expression (to be matched against the entire original path).
 * <li>Attribute selector strings in the
 *     "{@code [agent.name=value&][agent.type=value&][agent.key[=value]&…&agent.key[=value]&][key[=value]&…&key[=value]]}"
 *     format, where {@code key} and {@code value} are names and values of agent
 *     properties or static channel attributes. If {@code value} is omitted, the
 *     existence of the attribute is checked.
 * </ul><p>
 * <h4>Implementing notes:</h4>
 * Only {@code List<String> getDisplayPaths(AgentChannel channel)} and {@code List<String> getDisplayPaths(String originPath)}
 * methods are called by clients that support mapping a single original channel to several display paths. However, since most
 * filters map to a single display path, convenience methods
 * {@code String getDisplayPath(AgentChannel channel)} and {@code String getDisplayPath(String originPath)} are provided to be
 * implemented by such filters. Default implementations are as follows:
 * <ul>
 * <li>{@code List<String> getDisplayPaths(AgentChannel channel)} forwards to {@code String getDisplayPath(AgentChannel channel)}.
 * <li>{@code List<String> getDisplayPaths(String originPath)} forwards to {@code String getDisplayPath(String originPath)}.
 * <li>{@code String getDisplayPath(AgentChannel channel)} forwards to {@code String getDisplayPath(String originPath)}.
 * <li>{@code String getDisplayPath(String originPath)} leaves the path unchanged.
 * </ul><p>
 * If neither of these methods is overridden, this filter will leave the paths unchanged but 
 * {@code getOriginChannels()} can be overridden to reject unwanted channels.
 * <p>
 * This interface is somewhat overcomplicated, and some of the method names are unfortunate, for historical reasons:
 * in the original version of the framework, one-to-many mapping was not supported, data dictionary (and, therefore,
 * an ability to look channels up by path) was not available, etc. In later development, it was necessary to keep
 * this interface backward-compatible with any existing filters.
 *
 * @author onoprien
 */
public interface AgentChannelsFilter extends Savable {
    
    /**
     * Category for general purpose {@link Persistable Persistable} filters.
     */
    static public String CATEGORY = "AgentChannelsFilter";
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns the name of this filter.
     * The default implementation returns {@code null}.
     * 
     * @return Name of this filter, or {@code null} if this filter has no name.
     */
    default String getName() {
        return null;
    }
    
    /**
     * Returns a list of agent names accepted by this filter. 
     * The list returned by this method affects interpretation of channel selectors returned by 
     * {@code getOriginChannels()}: every selector that does not explicitly specify agent is
     * only applied to agents from the list.
     * <p>
     * The default implementation returns {@code null}.
     * 
     * @return Agent names accepted by this filter, or {@code null} if any agent is accepted.
     */
    default List<String> getAgents() {
        return null;
    }
    
    /**
     * Returns a list of original channel selectors that might be accepted by this filter.
     * <p>
     * The list returned by this method may contain channel selectors of
     * several types (note that interpretation of some selectors depends on the
     * list returned by {@code getAgents()} method):
     * <ul>
     * <li>Complete path.
     * <li>Path template in the "{@code [agent]/[partial path][/]}" format.
     * Templates that omit the agent name are expanded against all agents whose
     * names are in the {@code getAgents()} list (or all agents present on the
     * buses if that list is {@code null}). Templates that end with "/" match
     * all channels that start with "{@code agent/partial path/}".
     * <li>Arbitrary regular expression (to be matched against the entire
     * original path).
     * <li>Attribute selector strings in the
     * "{@code [agent.name=value&][agent.type=value&][agent.key[=value]&…&agent.key[=value]&][key[=value]&…&key[=value]]}"
     * format, where {@code key} and {@code value} are names and values of agent
     * properties or static channel attributes. If {@code value} is omitted, the
     * existence of the attribute is checked.
     * </ul><p>
     * <p>
     * The default implementation returns {@code null}.
     * 
     * @return Original channel paths accepted by this filter, or {@code null} if any
     * original path for which {@link #getDisplayPaths} returns a non-empty list is accepted.
     */
    default List<String> getOriginChannels() {
        return null;
    }
    
    /**
     * Returns a list of display paths produced by this filter.
     * Typically, if this method returns a non-{@code null} list, components downstream from
     * this filter will display this fixed set of channels, whether or not the data is actually available.
     * If this method returns {@code null}, all display paths produced by {@code getDisplayPaths(AgentChannel channel)}
     * for original channels accepted by {@code getOriginChannels()} should be displayed.
     * <p>
     * The default implementation returns {@code null}.
     * 
     * @return Display paths produced by this filter, or {@code null} if the set of display
     *         paths is not constrained by this filter.
     */
    default List<String> getDisplayChannels() {
        return null;
    }
    
    /**
     * Returns a list of attributes to display for each channel.
     * The default implementation returns {@code null}.
     * 
     * @param compact If {@code true}, returns a list of columns in a compact view.
     * @return Display fields, or {@code null} if this filter does not provides any hints on what attributes should be displayed.
     */
    default List<String> getFields(boolean compact) {
        return null;
    }
    
    /**
     * Returns a list of groups of display paths produced by this filter.
     * The default implementation returns {@code null}.
     * 
     * @return Display groups, or {@code null} if the set of groups is not constrained by this filter.
     */
    default List<String> getGroups() {
        return null;
    }
    
    
// -- Mapping original channels to display paths : -----------------------------
    
    /**
     * Returns a list of display channels for the specified original channel.
     * This method is called by clients that have access to original {@code AgentChannel}
     * instances (like monitoring views).
     * <p>
     * Not that if the specified original channel is not accepted by this filter,
     * this method should return an empty list. {@code null} should never be returned.
     * <p>
     * The default implementation returns a singleton list that contains a string
     * returned by {@link #getDisplayPath(AgentChannel)} method.
     * 
     * @param channel Original data channel.
     * @return List of display channels.
     */
    default List<String> getDisplayPaths(AgentChannel channel) {
        String displayPath = getDisplayPath(channel);
        return displayPath == null ? Collections.emptyList() : Collections.singletonList(displayPath);
    }
    
    /**
     * Returns a list of display channels for the specified original path.
     * This method is called by clients that do not have access to original {@code AgentChannel}
     * instances (like the trending tool).
     * <p>
     * If the specified original channel is not accepted by this filter,
     * this method should return an empty list. {@code null} should never be returned.
     * <p>
     * The default implementation returns a singleton list that contains a string
     * returned by {@link #getDisplayPath(String)} method.
     * 
     * @param originPath Original path.
     * @return List of display channels.
     */
    default List<String> getDisplayPaths(String originPath) {
        String displayPath = getDisplayPath(originPath);
        return displayPath == null ? Collections.emptyList() : Collections.singletonList(displayPath);
    }
    
    /**
     * Returns the display path corresponding to the specified original channel.
     * This method is used directly by clients that expect each original channel to be mapped to
     * no more than one display path. It can also be overridden to define a path returned in a 
     * singleton list by 
     * If the original channel is mapped to multiple display paths, this method may
     * return one of them, or {@code null} if the clients that rely on mapping to a single
     * display path should ignore the specified channel.
     * <p>
     * The default implementation forwards the call to {@link #getDisplayPath(String)}
     * method. Subclasses may override it to take into account channel attributes and metadata.
     * 
     * @param channel Original data channel.
     * @return Display path, or {@code null} if the provided channel is not accepted by this filter.
     */
    default String getDisplayPath(AgentChannel channel) {
        return getDisplayPath(channel.getPath());
    }
    
    /**
     * Returns the display channel corresponding to the specified original path.
     * If the original channel is mapped to multiple display paths, this method may
     * return one of them, or {@code null}.
     * <p>
     * If the specified original channel is not accepted by this filter,
     * this method should return an empty list. {@code null} should never be returned.
     * <p>
     * The default implementation returns an unmodified original path.
     * 
     * @param originPath Original path.
     * @return List of display channels.
     */
    default String getDisplayPath(String originPath) {
        return originPath;
    }
    
    
// -- Mapping display paths back to original paths : ---------------------------
    
    /**
     * Returns the list of original paths mapped to the specified display channel.
     * <p>
     * The default implementation returns a list that contains one element - the
     * path returned by {@link #getOriginPath getOriginPath(...)} -  or an
     * empty list if {@link #getOriginPath getOriginPath(...)} returns {@code null}.
     * 
     * @param displayPath Display channel path.
     * @return Original paths, or an empty list if the specified path does not correspond to any original channel.
     */
    default List<String> getOriginPaths(String displayPath) {
        String path = getOriginPath(displayPath);
        return path == null ? Collections.emptyList() : Collections.singletonList(path);
    }
    
    /**
     * Returns the original path corresponding to the specified display channel.
     * If multiple original paths are mapped to the specified display path, this
     * method may return one of them, or {@code null}.
     * <p>
     * The default implementation returns the display path provided as an argument.
     * 
     * @param displayPath Display channel path.
     * @return Original path, or {@code null} if the specified path does not correspond to any original channel.
     */
    default String getOriginPath(String displayPath) {
        return displayPath;
    }


// -- Static methods that apply a filter : -------------------------------------
    
    /**
     * Returns a list of display paths for the specified original path, applying the provided filter.
     * This method takes into account the output of
     * {@link #getDisplayPaths(String) getDisplayPaths(originPath)},
     * {@link #getDisplayChannels()}, {@link getAgents()}, and {@link #getOriginChannels()}.
     * Templates in the list of origin paths returned by the filter are honored but attribute selectors are ignored. 
     * 
     * @param filter Filter to apply.
     * @param originPath Origin path.
     * @return List of display path.
     */
    @Deprecated
    static List<String> getDisplayPath(AgentChannelsFilter filter, String originPath) {
        
        List<String> paths;
        try {
            paths = filter.getDisplayPaths(originPath);
        } catch (RuntimeException x) {
            return Collections.emptyList();
        }
        if (paths.isEmpty()) return Collections.emptyList();
        
        List<String> dps = filter.getDisplayChannels();
        if (dps != null) {
            paths.retainAll(dps);
        }
        
        List<String> agents = filter.getAgents();
        List<String> channels = filter.getOriginChannels();
        int i = originPath.indexOf("/");
        if (i == -1) return Collections.emptyList();
        String agent = originPath.substring(0, i);
        
        if (channels == null) {
            if (agents == null || agents.contains(agent)) {
                return paths;
            }
        } else {
            for (String channel : channels) {
                if (!isSelector(channel)) {
                    if ((agents == null || !channel.startsWith("/") || agents.contains(agent)) && matchTemplate(originPath, channel)) {
                        return paths;
                    }
                }
            }
        }
        return Collections.emptyList();
    }
    
    
// -- Static utility methods : -------------------------------------------------
    
    /**
     * Checks whether the path matches the template.
     * 
     * @param path Origin path.
     * @param template Path template.
     * @return True if the path matches.
     */
    @Deprecated
    static boolean matchTemplate(String path, String template) {
        if (template.startsWith("/")) {
            path = path.substring(path.indexOf("/"));
        }
        return template.endsWith("/") ? path.startsWith(template) : path.equals(template);
    }
    
    /**
     * Checks whether the specified string is in the channel selector format (as opposed to explicit path or template).
     *
     * @param code String to test.
     * @return {@code true} if the specified string is a channel selector.
     */
    @Deprecated
    static boolean isSelector(String code) {
        if (code.contains("=")) return true;
        return ! code.contains("/");
    }
    
}
