package org.lsst.ccs.gconsole.plugins.jgroups;

import java.io.Serializable;
import java.util.ArrayList;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.SwingWorker;
import org.jgroups.tests.Probe;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.gconsole.services.persist.Savable;
import org.lsst.ccs.gconsole.util.swing.BoxedComboBox;
import org.lsst.ccs.gconsole.util.swing.BoxedTextField;

/**
 *
 * @author onoprien
 */
public final class ProbePage extends JScrollPane implements Savable {

// -- Fields : -----------------------------------------------------------------
    
    JTextField probeCommand, timeoutField, matchField;
    JButton probeSend, probeHelp;
    JCheckBox verboseBox, weedOutDuplicatesBox, timeoutBox, busBox, matchBox;
    JComboBox<String> busCombo;

// -- Life cycle : -------------------------------------------------------------
    
    public ProbePage() {
        
        Box root = Box.createVerticalBox();
        setViewportView(root);
        
        Box probe = Box.createVerticalBox();
        root.add(probe);
        probe.setBorder(BorderFactory.createTitledBorder("Probe"));
        
        // Probe command and help
        
        Box line = Box.createHorizontalBox();
        probe.add(line);
        line.add(Box.createRigidArea(Const.HDIM));
        probeCommand = new BoxedTextField();
        line.add(probeCommand);
        line.add(Box.createRigidArea(Const.HDIM));
        probeSend = new JButton("Send");
        line.add(probeSend);
        probeSend.addActionListener(e -> sendProbe());
        line.add(Box.createRigidArea(Const.HDIM));
        probeHelp = new JButton("Help");
        line.add(probeHelp);
        probeHelp.addActionListener(e -> Console.getConsole().showInBrowser("http://www.jgroups.org/manual4/index.html#Probe"));
        line.add(Box.createRigidArea(Const.HDIM));
        line.add(Box.createHorizontalGlue());
        probe.add(Box.createRigidArea(Const.VDIM));
        
        // Verbose and duplicates
        
        line = Box.createHorizontalBox();
        probe.add(line);
        line.add(Box.createRigidArea(Const.HDIM));
        verboseBox = new JCheckBox("Verbose");
        line.add(verboseBox);
        line.add(Box.createRigidArea(Const.HDIM2));
        weedOutDuplicatesBox = new JCheckBox("Weed out duplicates");
        line.add(weedOutDuplicatesBox);        
        line.add(Box.createRigidArea(Const.HDIM));
        line.add(Box.createHorizontalGlue());
        probe.add(Box.createRigidArea(Const.VDIM));
        
        // Timeout
        
        line = Box.createHorizontalBox();
        probe.add(line);
        line.add(Box.createRigidArea(Const.HDIM));
        timeoutBox = new JCheckBox("Timeout : ");
        line.add(timeoutBox);
        timeoutBox.addActionListener(e -> timeoutField.setEnabled(timeoutBox.isSelected()));
        timeoutField = new BoxedTextField(0);
        timeoutField.setColumns(4);
        timeoutField.setMaximumSize(timeoutField.getPreferredSize());
        line.add(timeoutField);
        line.add(new JLabel(" seconds."));
        line.add(Box.createRigidArea(Const.HDIM));
        line.add(Box.createHorizontalGlue());
        probe.add(Box.createRigidArea(Const.VDIM));
        
        // Bus
        
        String tip = "Select CCS bus, or enter an arbitrary JGroups cluster name";
        line = Box.createHorizontalBox();
        probe.add(line);
        line.add(Box.createRigidArea(Const.HDIM));
        busBox = new JCheckBox("Bus : ");
        line.add(busBox);
        busBox.setToolTipText(tip);
        busBox.addActionListener(e -> busCombo.setEnabled(busBox.isSelected()));
        line.add(Box.createRigidArea(Const.HDIM));
        busCombo = new BoxedComboBox();
        line.add(busCombo);
        busCombo.setToolTipText(tip);
        busCombo.setEditable(true);
        busCombo.addItem("");
        for (Bus bus : Bus.values()) {
            busCombo.addItem(bus.toString());
        }
        line.add(Box.createRigidArea(Const.HDIM));
        line.add(Box.createHorizontalGlue());
        probe.add(Box.createRigidArea(Const.VDIM));
        
        // Match
        
        tip = "Only responces that contain the specified pattern are printed";
        line = Box.createHorizontalBox();
        probe.add(line);
        line.add(Box.createRigidArea(Const.HDIM));
        matchBox = new JCheckBox("Match : ");
        line.add(matchBox);
        matchBox.setToolTipText(tip);
        matchBox.addActionListener(e -> matchField.setEnabled(matchBox.isSelected()));
        matchField = new BoxedTextField();
        line.add(matchField);
        matchField.setToolTipText(tip);
        line.add(Box.createRigidArea(Const.HDIM));
        line.add(Box.createHorizontalGlue());
        probe.add(Box.createRigidArea(Const.VDIM));
        
        root.add(Box.createVerticalGlue());
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    private void sendProbe() {
        ArrayList<String> args = new ArrayList<>();
        String command = probeCommand.getText().trim();
        if (verboseBox.isSelected()) {
            args.add("-v ");
        }
        if (weedOutDuplicatesBox.isSelected()) {
            args.add("-weed_out_duplicates ");
        }
        if (timeoutBox.isSelected()) {
            try {
                String s = timeoutField.getText().trim();
                int t = Integer.parseInt(s);
                if (t <= 0) throw new NumberFormatException();
                args.add("-timeout");
                args.add(s);
            } catch (NumberFormatException x) {
                Console.getConsole().error("Timeout value should be a positive integer", x);
                return;
            }
        }
        if (busBox.isSelected()) {
            String s = busCombo.getSelectedItem().toString().trim();
            if (s.isEmpty()){
                Console.getConsole().error("Empty bus name");
                return;
            }
            args.add("-cluster");
            args.add(s);
        }
        if (matchBox.isSelected()) {
            String s = matchField.getText().trim();
            if (s.isEmpty()){
                Console.getConsole().error("Empty match pattern");
                return;
            }
            args.add("-match");
            args.add(s);
        }
        args.add(command);
        (new SwingWorker() {
            @Override
            protected Object doInBackground() throws Exception {
                try {
                    System.out.println("\nProbe " + String.join(" ", args));
                    Probe.main(args.toArray(new String[0]));
                } catch (Exception x) {
                    Console.getConsole().error("Probe failure", x);
                }
                return null;
            }
        }).execute();
    }

// -- Savin/restoring : --------------------------------------------------------

    @Override
    public Descriptor save() {
        Descriptor d = new Descriptor();
        d.setCommand(probeCommand.getText().trim());
        d.setVerbose(verboseBox.isSelected());
        d.setWeedOutDuplicates(weedOutDuplicatesBox.isSelected());
        String time = timeoutField.getText();
        try {
            int seconds = Integer.parseInt(time);
            if (seconds < 1) throw new NumberFormatException();
            if (!timeoutBox.isSelected()) seconds = - seconds;
            d.setTimeout(seconds);
        } catch (NumberFormatException x) {
            d.setTimeout(0);
        }
        String bus = busCombo.getSelectedItem().toString().trim();
        if (!bus.isEmpty()) {
            if (!busBox.isSelected()) bus = "^"+ bus;
            d.setBus(bus);
        }
        String m = matchField.getText().trim();
        if (!m.isEmpty()) {
            if (!matchBox.isSelected()) m = "^"+ m;
            d.setMatch(m);
        }
        return d;
    }

    @Override
    public void restore(Serializable descriptor) {
        Descriptor d = descriptor instanceof Descriptor ? (Descriptor) descriptor : new Descriptor();
        probeCommand.setText(d.getCommand() == null ? "" : d.getCommand());
        verboseBox.setSelected(d.isVerbose());
        weedOutDuplicatesBox.setSelected(d.isWeedOutDuplicates());
        int t = d.getTimeout();
        if (t <= 0) {
            timeoutBox.setSelected(false);
            timeoutField.setText(Integer.toString(-t));
            timeoutField.setEnabled(false);
        } else {
            timeoutBox.setSelected(true);
            timeoutField.setText(Integer.toString(t));
            timeoutField.setEnabled(true);
        }
        String bus = d.getBus();
        if (bus == null) {
            busBox.setSelected(false);
            busCombo.setSelectedItem("");
            busCombo.setEnabled(false);
        } else {
            if (bus.startsWith("^")) {
                busBox.setSelected(false);
                busCombo.setSelectedItem(bus.substring(1));
                busCombo.setEnabled(false);
            } else {
                busBox.setSelected(true);
                busCombo.setSelectedItem(bus);
                busCombo.setEnabled(true);
            }
        }
        String m = d.getMatch();
        if (m == null) {
            matchBox.setSelected(false);
            matchField.setText("");
            matchField.setEnabled(false);
        } else {
            if (m.startsWith("^")) {
                matchBox.setSelected(false);
                matchField.setText(m.substring(1));
                matchField.setEnabled(false);
            } else {
                matchBox.setSelected(true);
                matchField.setText(m);
                matchField.setEnabled(true);
            }
        }
    }
    
    static public class Descriptor implements Serializable, Cloneable {

        private DataPanelDescriptor page;

        private String command;
        private boolean verbose;
        private boolean weedOutDuplicates;
        private String bus;
        private int timeout = -10;
        private String match;

        public String getCommand() {
            return command;
        }

        public void setCommand(String command) {
            this.command = command;
        }

        public String getMatch() {
            return match;
        }

        public void setMatch(String match) {
            this.match = match;
        }

        public int getTimeout() {
            return timeout;
        }

        public void setTimeout(int timeout) {
            this.timeout = timeout;
        }

        public String getBus() {
            return bus;
        }

        public void setBus(String bus) {
            this.bus = bus;
        }

        public boolean isWeedOutDuplicates() {
            return weedOutDuplicates;
        }

        public void setWeedOutDuplicates(boolean weedOutDuplicates) {
            this.weedOutDuplicates = weedOutDuplicates;
        }

        public boolean isVerbose() {
            return verbose;
        }

        public void setVerbose(boolean verbose) {
            this.verbose = verbose;
        }

        public DataPanelDescriptor getPage() {
            return page;
        }

        public void setPage(DataPanelDescriptor page) {
            this.page = page;
        }

    }
    
}
