package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.services.lock.LockService;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.gconsole.util.ThreadUtil;
import org.lsst.ccs.messaging.AgentPresenceListener;

/**
 * LSST Command Browser Plugin.
 * The original version of the command dictionary tool was written by Etienne Marin-Matholaz.
 * 
 * @author onoprien
 */
@Plugin(name="LSST Commands Browser Plugin",
        id="command",
        description="LSST Commands Browser allows listing available commands for a subsystem and executing them.",
        shortDescription="LSST Remote Subsystem Commands Browser")
public class LsstCommandBrowserPlugin extends ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------
    
    static private final String TOOL_NAME = "Command and Lock Browser";
    static public String[] DEFAULT_TYPES = new String[] {"WORKER", "SERVICE", "MCM", "OCS_BRIDGE", "MMM"};
    static private final String OPT_TIME = "time";
            
    private final ArrayList<Browser> browsers = new ArrayList<>(0);
    private final AgentPresenceListener agentConnectionListener;
    private LockService lockService = LockService.getService();
    
    private boolean displayTiming;
    
// -- Life cycle : -------------------------------------------------------------
    
    public LsstCommandBrowserPlugin() {
        
        // Listening to agents connecting/disconnecting :
        
        agentConnectionListener = new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) {
                for (AgentInfo agent : agents) {
                    AgentInfo.AgentType type = agent.getType();
                    if (!(AgentInfo.AgentType.CONSOLE.equals(type) || AgentInfo.AgentType.LISTENER.equals(type))) {
                        SwingUtilities.invokeLater(() -> {
                            Action action = new AbstractAction(agent.getName()) {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    createBrowser("COM:" + agent.getName(), agent.getName());
                                }
                            };
                            getServices().addMenu(action, "400: CCS Tools :-1:5", TOOL_NAME +":100:100", "Subsystems:1");
                            action = new AbstractAction(TOOL_NAME) {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    createBrowser("Commands:" + agent.getName(), agent.getName());
                                }
                            };
                            getServices().addMenu(action, "CCS Subsystems", agent.getName() + ":-10:5");
                        });
                    }
                }
            }
            @Override
            public void disconnected(AgentInfo... agents) {
                SwingUtilities.invokeLater(() -> {
                    for (AgentInfo agent : agents) {
                        Console.getConsole().removeMenu(" CCS Tools ", TOOL_NAME, "Subsystems", agent.getName());
                        Console.getConsole().removeMenu("CCS Subsystems", agent.getName(), TOOL_NAME);
                    }
                });
            }
        };
        
    }

    @Override
    public void initialize() {
        
        lockService = LockService.getService();
        
        getServices().addProperty(OPT_TIME, false);
        getServices().addPreference(new String[] {"LSST",TOOL_NAME}, null, "${"+ OPT_TIME +"} Display command execution time.");
        
        Action newDictionaryAction = new AbstractAction("Browse...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                BrowserFull.Descriptor descriptor = new BrowserFull.Descriptor();
                descriptor.setName("Commands");
                descriptor.setFilterTypes(DEFAULT_TYPES);
                LsstCommandBrowserPlugin.this.createBrowser(descriptor);
            }
        };
        getServices().addMenu(newDictionaryAction, "400: CCS Tools :-1:5", TOOL_NAME +":0");
    }

    @Override
    public void start() {
        displayTiming = (Boolean) getServices().getProperty(OPT_TIME);
        getConsole().getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(agentConnectionListener);
    }

    @Override
    public void stop() {
        getConsole().getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(agentConnectionListener);
    }

    @Override
    public void propertiesChanged(Object source, Map<String, Object> changes) {
        Object value = changes.get(OPT_TIME);
        if (value instanceof Boolean) {
            displayTiming = (Boolean) value;
        }
    }

    
// -- Creating browsers : ------------------------------------------------------

    /**
     * Creates and displays a generic (un-filtered, default settings) command browser page.
     * May be called on any thread.
     * 
     * @param name Page title.
     * @param subsystem Subsystem name, or {@code null} if the page should be able to display commands for all subsystems.
     */
    public void createBrowser(String name, String subsystem) {
        Browser.Descriptor descriptor;
        if (subsystem == null) {
            descriptor = new BrowserFull.Descriptor();
        } else {
            descriptor = new BrowserOneSubsystem.Descriptor();
            descriptor.setAgent(subsystem);
        }
        descriptor.setName(name);
        createBrowser(descriptor);
    }
    
    /**
     * Creates and displays a command dictionary page customized by the provided descriptor.
     * May be called on any thread.
     * 
     * @param descriptor Descriptor of the page to be created.
     */
    public void createBrowser(Browser.Descriptor descriptor) {
        ThreadUtil.invokeLater(() -> {
            Browser browser;
            if (descriptor instanceof BrowserFull.Descriptor) {
                browser = new BrowserFull();
            } else if (descriptor instanceof BrowserOneSubsystem.Descriptor) {
                browser = new BrowserOneSubsystem();
            } else {
                getConsole().getLogger().error("Unknown type of command browser descriptor: "+ descriptor);
                return;
            }
            browser.restore(descriptor);
            Map<Object, Object> par = new HashMap<>();
            DataPanelDescriptor panDesc = descriptor.getPage();
            if (panDesc != null && panDesc.isOpen()) {
                Map<String, Serializable> data = panDesc.getData();
                if (data != null) {
                    par.putAll(data);
                }
            }
            par.put(Panel.TITLE, browser.getName());
            Consumer<JComponent> onClose = c -> {
                Iterator<Browser> it = browsers.iterator();
                while (it.hasNext()) {
                    Browser b = it.next();
                    if (c == b.getPanel()) {
                        b.shutdown();
                        it.remove();
                        break;
                    }
                }
                browsers.trimToSize();
            };
            par.put(Panel.ON_CLOSE, onClose);
            getConsole().getPanelManager().open(browser.getPanel(), par);
            browsers.add(browser);
            browsers.trimToSize();
        });        
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns {@code true} if the command execution timing should by displayed by browsers, according to the current user preferences.
     * @return True if command execution timing should by displayed.
     */
    boolean isDisplayTiming() {
        return displayTiming;
    }
    
    
// -- Saving/restoring : -------------------------------------------------------

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        if (! (storageBean instanceof Descriptor)) return true;
        
        Descriptor desc = (Descriptor) storageBean;
        if (!browsers.isEmpty()) {
            ArrayList<Browser> copy = new ArrayList<>(browsers);
            for (Browser d : copy) {
                JComponent panel = d.getPanel();
                if (panel != null) {
                    getConsole().getPanelManager().close(panel);
                }
            }
            browsers.clear();
        }
        
        lockService.login(desc.getUserID(), ""); // FIXME: handle credentials once implemented on the toolkit side
        
        Browser.Descriptor[] dd = desc.getBrowsers();
        if (dd != null) {
            for (Browser.Descriptor d : dd) {
                createBrowser(d);
            }
        }
        return true;
    }

    @Override
    public Descriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        if (!browsers.isEmpty()) {
            Browser.Descriptor[] descriptors = new Browser.Descriptor[browsers.size()];
            for (int i = 0; i < descriptors.length; i++) {
                descriptors[i] = browsers.get(i).save();
            }
            desc.setBrowsers(descriptors);
        }
        desc.setUserID(lockService.getUserId());
        return desc;
    }

    static public class Descriptor extends ComponentDescriptor {

        private Browser.Descriptor[] browsers;
        private String userID;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public Browser.Descriptor[] getBrowsers() {
            return browsers;
        }

        public void setBrowsers(Browser.Descriptor[] browsers) {
            this.browsers = browsers;
        }

        public String getUserID() {
            return userID;
        }

        public void setUserID(String userID) {
            this.userID = userID;
        }
        
    }




    
}
