package org.lsst.ccs.gconsole.base.filter;

import java.util.*;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;

/**
 * Filter that selects configuration channels.
 *
 * @author onoprien
 */
public class ConfigurationFilter extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    static public final String CREATOR_PATH = "Built-In/Basic/Configuration";
    static public final String CREATOR_PATH_WHOLECAMERA = "Built-In/Whole Camera/Configuration";
    
    private final String name;
    private final List<String> agents;
    private final String agent;
    private final String replaceTarget;

// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = AgentChannelsFilter.CATEGORY,
            name = "Configuration",
            path = CREATOR_PATH,
            description = "Selects subsystem configuration channels, and maps them to display paths in \"[subsystem/]category//rest/of/path\" format. Subsystem name is included only if multiple subsystems are accepted.")
    public ConfigurationFilter(
            @Par(def = "Config", desc = "Filter name.") String name,
            @Par(def = Par.NULL, desc = "List of subsystem names accepted by this filter. If null, all subsystems are accepted.") List<String> subsystems)
    {
        this.name = name;
        if (subsystems == null) {
            agent = null;
            agents = null;
            replaceTarget = "configuration/";
        } else if (subsystems.size() == 1) {
            agent = subsystems.get(0);
            agents = null;
            replaceTarget = agent +"/configuration/";
        } else {
            agent = null;
            agents = Collections.unmodifiableList(new ArrayList<>(subsystems));
            replaceTarget = "configuration/";
        }
    }
    
    @Create(category = AgentChannelsFilter.CATEGORY,
            name = "Configuration",
            path = CREATOR_PATH_WHOLECAMERA,
            description = "Selects configuration channels of the whole camera, and maps them to display paths in \"subsystem/category//rest/of/path\" format.")
    public ConfigurationFilter() {
        this("Configuration", null);
    }

    
// -- Implementing filter : ----------------------------------------------------

    @Override
    public List<String> getAgents() {
        return agent == null ? agents : Collections.singletonList(agent);
    }

    @Override
    public List<String> getOriginChannels() {
        return Collections.singletonList("/configuration/");
    }

    @Override
    public String getDisplayPath(AgentChannel channel) {
        String category = channel.get(AgentChannel.Key.CONFIG_CATEGORY);
        return category == null ? null : channel.getPath().replace(replaceTarget, category +"//");
    }

    @Override
    public String getOriginPath(String displayPath) {
        if (agent == null) { // agent/category//localpath
            int iA = displayPath.indexOf("/");
            int iC = displayPath.indexOf("//");
            return displayPath.substring(0, iA) +"/configuration/"+ displayPath.substring(iC+2);
        } else { // category//localpath
            int iC = displayPath.indexOf("//");
            return agent +"/configuration/"+ displayPath.substring(iC+2);
        }
    }

    @Override
    public String getName() {
        return name;
    }

}

