package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import java.util.stream.Collectors;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;

/**
 * Formatter for monitoring data.
 * <p>
 * Instances of this class are used by {@link MonitorView} to produce formatted values for display.
 * By default, calls are simply forwarded to {@link MonitorField} methods. Subclasses can modify
 * this behavior and provide custom formatting.
 * <p>
 * This class also provides convenience methods for formatting monitoring table cells, etc.
 *
 * @author onoprien
 */
public class MonitorFormat {

// -- Default instance : -------------------------------------------------------
    
    static public final MonitorFormat DEFAULT = new MonitorFormat();
    
// -- Formatting : -------------------------------------------------------------
    
    /**
     * Produces formatted value, given the field and the list of contributing channels.
     * The implementation provided by this class forwards the call to {@link MonitorField#format(List)}.
     * 
     * @param field Monitored field.
     * @param channels Contributing channels.
     * @return Formatted value.
     */
    public FormattedValue format(MonitorField field, List<AgentChannel> channels) {
        return field.format(channels);
    }
    
    /**
     * Produces formatted value, given the monitored field and the data channel whose value needs to be formatted.
     * The implementation provided by this class forwards the call to {@link MonitorField#format(AgentChannel)}.
     * 
     * @param field Monitored field.
     * @param channel Data channel.
     * @return Formatted value.
     */
    public FormattedValue format(MonitorField field, AgentChannel channel) {
        return field.format(channel);
    }
    
    
// -- Convenience methods: -----------------------------------------------------
    
    /**
     * Convenience method that updates {@link FormattedValue} associated with the specified cell.
     * The actual formatting is done by {@link #format(MonitorField field, List channels)}.
     * 
     * @param cell Cell to be formatted.
     * @return True if the formatted value associated with the cell has changed as a result of this call.
     */
    public boolean format(MonitorCell cell) {
        List<AgentChannel> channels = cell.getChannels().stream()
                .flatMap(h -> h.getChannels().stream())
                .collect(Collectors.toList());
        FormattedValue fv = format(cell.getField(), channels);
        if (Objects.equals(fv, cell.getFormattedValue())) {
            return false;
        } else {
            cell.setFormattedValue(fv);
            return true;
        }
    }
    
    /**
     * Convenience method that updates {@link FormattedValue} associated with the specified cell,
     * assuming that the only possible reason for a change are changes in the specified channel. <br>
     * The implementation provided by this class forwards the call to {@link #format(MonitorCell cell)}, 
     * ignoring the {@code channelHandle} argument. Subclasses might implement more efficient formatting
     * for cells that depend on multiple channels.
     * 
     * @param cell Cell to be formatted.
     * @param channelHandle Handle of the data channel that might have changed.
     * @return  True if the formatted value associated with the cell has changed as a result of this call.
     */
    public boolean format(MonitorCell cell, DisplayChannel channelHandle) {
        return format(cell);
    }
    
    /**
     * Convenience method that produces formatted value, given the field and the channel handle. <br>
     * The actual formatting is done by {@link #format(MonitorField field, AgentChannel channel)}.
     * 
     * @param field Monitored field.
     * @param channelHandle Handle of a data channel whose value needs to be formatted.
     * @return Formatted value.
     */
    public FormattedValue format(MonitorField field, DisplayChannel channelHandle) {
        return format(field, channelHandle.getChannels());
    }
        
}
