package org.lsst.ccs.gconsole.services.persist;

import java.lang.reflect.Constructor;
import java.lang.reflect.Executable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;

/**
 * {@code Creator} that wraps an {@code Executable}.
 *
 * @author onoprien
 */
public class CreatorExecutable implements Creator {

// -- Fields : -----------------------------------------------------------------
    
    private final Executable exec;
    private final Object object;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Construct the factory.
     * 
     * @param exec Executable to be invoked.
     * @param object Object the executable belongs to, if any. If the executable is a static 
     *               method or a constructor, this object is ignored and can be {@code null}.
     */
    public CreatorExecutable(Executable exec, Object object) {
        if (!(exec instanceof Method || exec instanceof Constructor) || 
                exec.getAnnotation(Create.class) == null || 
                exec.getAnnotation(Create.class).path().trim().isEmpty()) {
            throw new IllegalArgumentException();
        }
        this.exec = exec;
        this.object = object;
    }
    
    
// -- Implement Creator : ------------------------------------------------------

    @Override
    public String getCategory() {
        return exec.getAnnotation(Create.class).category();
    }

    @Override
    public String getPath() {
        return exec.getAnnotation(Create.class).path();
    }

    @Override
    public String getName() {
        return exec.getAnnotation(Create.class).name();
    }

    @Override
    public String getDescription() {
        return exec.getAnnotation(Create.class).description();
    }

    @Override
    public Parameter[] getParameters() {
        return exec.getParameters();
    }

    @Override
    public String[] getParameterDescriptions() {
        Parameter[] pp = getParameters();
        String[] dd = new String[pp.length];
        for (int i=pp.length; i-->0; ) {
            Par an = pp[i].getAnnotation(Par.class);
            dd[i] = an == null ? "" : an.desc();
        }
        return dd;
    }

    /**
     * Creates an instance.
     * 
     * @param parameters Parameter values.
     * @return Created instance.
     * @throws IllegalAccessException If the underlying executable is inaccessible due to Java access control.
     * @throws IllegalArgumentException If the provided parameter values are incompatible with the underlying executable.
     * @throws InvocationTargetException If the underlying executable throws an exception.
     * @throws InstantiationException If if the initialization provoked by this method fails.
     */
    @Override
    public Persistable make(Object... parameters) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException, InstantiationException {
        Persistable out;
        if (exec instanceof Method) {
            out = (Persistable) ((Method) exec).invoke(object, parameters);
        } else {
            out = (Persistable) ((Constructor) exec).newInstance(parameters);
        }
        Creator.Descriptor creDesc = new Creator.Descriptor();
        creDesc.setCategory(getCategory());
        creDesc.setPath(getPath());
        int n = parameters.length;
        if (n > 0) {
            Parameter[] pars = getParameters();
            String[] sp = new String[n];
            for (int i=0; i<n; i++) {
                sp[i] = Creator.encode(parameters[i], pars[i].getParameterizedType());
            }
            creDesc.setParameters(sp);
        }
        Persistable.Descriptor desc = out.getDescriptor();
        desc.setCreator(creDesc);
        if (desc.getName() == null) {
            desc.setName(getName());
        }
        return out;
    }

    /**
     * Creates an instance.
     * 
     * @param parametersAsStrings Encoded parameter values.
     * @return Created instance.
     * @throws IllegalAccessException If the underlying executable is inaccessible due to Java access control.
     * @throws IllegalArgumentException If the provided parameter values are incompatible with the underlying executable.
     * @throws InvocationTargetException If the underlying executable throws an exception.
     * @throws InstantiationException If if the initialization provoked by this method fails.
     */
    @Override
    public Persistable make(String... parametersAsStrings) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException, InstantiationException {
        Parameter[] pars = getParameters();
        int n = pars.length;
        Object[] values = new Object[n];
        for (int i = 0; i < n; i++) {
            values[i] = Creator.decode(parametersAsStrings[i], pars[i].getParameterizedType());
        }
        return make(values);
    }

}
