package org.lsst.ccs.gconsole.plugins.trending;

import java.util.*;
import org.lsst.ccs.localdb.statusdb.server.ChannelMetaData;

/**
 * Time history dataset for a specific channel and a specific time window.
 * Includes all metadata.
 *
 * @author onoprien
 */
public class TrendData {
    
// -- Fields : -----------------------------------------------------------------
    
    /** Key that identifies main data. */
    static final String VALUE_KEY = "value";
    
    private final long[] times; // time stamps for data and on-point metadata
    private final Map<String, double[]> values; // key to array of values for data and on-point metadata
    private final long[] timeRange; // {begin, end}
    private final boolean raw; // true if unbinned raw data
    
    private final Map<String, ArrayList<MetaValue>> meta; // key to off-point metadata
    
// -- Life cycle : -------------------------------------------------------------
    
    TrendData(long[] times, Map<String, double[]> onPointValues, Map<String, ArrayList<MetaValue>> offPointMetadata, long[] timeRange, boolean raw) {
        this.times = times;
        this.values = onPointValues;
        this.timeRange = timeRange;
        this.raw = raw;
        this.meta = offPointMetadata;
    }
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns an array of timestamps for points in this dataset.
     * Shared by data and on-point metadata.
     * 
     * @return An array of times, in milliseconds.
     */
    public long[] getTimes() {
        return times;
    }
    
    /**
     * Returns an array of values for data in this dataset.
     * 
     * @return Data values.
     */
    public double[] getValues() {
        return values.get(VALUE_KEY);
    }

    /**
     * Returns an array of timestamps available in this dataset for the specified data or metadata.
     * 
     * @param key String that identifies the data or metadata.
     * @return An array of times, in milliseconds, or {@code null} if there is no such data or metadata.
     */
    public long[] getTimes(String key) {
        if (values.containsKey(key)) {
            return times;
        } else {
            List<MetaValue> metaList = meta.get(key);
            if (metaList == null) return null;
            int n = metaList.size();
            long[] out = new long[n * 2];
            for (int i = 0; i < n; i++) {
                int k = 2 * i;
                MetaValue mp = metaList.get(i);
                out[k] = mp.getStart();
                out[k + 1] = mp.getStop();
            }
            return out;
        }
    }
    
    /**
     * Returns an array of values contained in this dataset for the specified metadata type.
     * 
     * @param key String that identifies the metadata.
     * @return An array of values for the specified metadata, or {@code null} if there is no
     *         such data or metadata that could be converted to double values.
     */
    public double[] getValues(String key) {
        double[] out = values.get(key);
        if (out == null) {
            List<MetaValue> metaList = meta.get(key);
            if (metaList == null) return null;
            int n = metaList.size();
            out = new double[n * 2];
            try {
                for (int i = 0; i < n; i++) {
                    int k = 2 * i;
                    double v = metaList.get(i).getDoubleValue();
                    out[k] = v;
                    out[k + 1] = v;
                }
            } catch (NumberFormatException x) {
                return null;
            }
        }
        return out;
    }
    
    /**
     * Returns an array of string values contained in this dataset for the specified off-point metadata type.
     * 
     * @param key String that identifies the metadata.
     * @return An array of metadata values, or {@code null} if there is no such metadata in this dataset.
     */
    public String[] getMetadata(String key) {
        ArrayList<MetaValue> metaList = meta.get(key);
        if (metaList == null) return null;
        int n = metaList.size();
        String[] out = new String[n * 2];
        for (int i=0; i<n; i++) {
            int k = 2*i;
            MetaValue mp = metaList.get(i);
            out[k] = mp.getValue();
            out[k+1] = mp.getValue();
        }
        return out;
    }
    
    /**
     * Returns off-point metadata value for the specified key if it is valid for the entire time window covered by this dataset.
     * Note that the current implementation does not require validity throughout time window, it returns a non-null value
     * for any single value metadata. This is to work around LSSTCCS-2253.
     * 
     * @param key String that identifies the metadata.
     * @return Value, or {@code null} if there is no such single-value metadata.
     */
    public String getSingleValueMetadata(String key) {
        ArrayList<MetaValue> metaList = meta.get(key);
        if (metaList == null || metaList.isEmpty()) return null;
        String out = metaList.get(0).getValue();
        for (int i=1; i<metaList.size(); i++) {
            if (!Objects.equals(out, metaList.get(i).getValue())) {
                return null;
            }
        }
        return out;
    }
    
    /**
     * Returns the set of keys for on-point data and metadata present in this dataset.
     * The set includes {@code VALUE_KEY}.
     * 
     * @return Set of keys.
     */
    public Set<String> getOnPointKeys() {
        HashSet<String> out = new HashSet<>(values.keySet());
        return out;
    }
    
    /**
     * Returns earliest time for data points contained in this dataset.
     * @return Lowest time for data points contained in this dataset.
     */
    public long getLowT() {
        return (times == null || times.length == 0) ? -1 : times[0];
    }
    
    /**
     * Returns latest time for data points contained in this dataset.
     * @return Highest time for data points contained in this dataset. 
     */
    public long getHighT() {
        return (times == null || times.length == 0) ? -1 : times[times.length - 1];
    }
    
    /**
     * Returns the time range of this data set.
     * Callers should not modify the returned array.
     * @return two-element array containing the bounds of the time range of this data set.
     */
    public long[] getTimeRange() {
        return timeRange;
    }

    /**
     * Reports the type of data.
     * @return {@code true} if unbinned raw data.
     */
    public boolean isRaw() {
        return raw;
    }
    
    
// -- Package-private access to off-point metadata : ---------------------------
    
    static class MetaValue {
        
        private final long start;
        private final long stop;
        private final String value;
        
        MetaValue(long start, long stop, String value) {
            this.start = start;
            this.stop = stop;
            this.value = value;
        }
        
        long getStart() {
            return start;
        }
        
        long getStop() {
            return stop;
        }
        
        String getValue() {
            return value;
        }
        
        double getDoubleValue() {
            return Double.parseDouble(value);
        }
        
        static MetaValue trim(MetaValue in, long begin, long end) {
            if (in.start >= end || in.stop <= begin) {
                return null;
            } else if (in.start < begin || in.stop > end) {
                return new MetaValue(Math.max(in.start, begin), Math.min(in.stop, end), in.value);
            } else {
                return in;
            }
        }
        
        static MetaValue trim(ChannelMetaData in, long begin, long end) {
            long start = in.getTstart();
            long stop = in.getTstop();
            if (stop == -1L) stop = Long.MAX_VALUE;
            if (start >= end || stop <= begin) {
                return null;
            } else {
                return new MetaValue(Math.max(start, begin), Math.min(stop, end), in.getValue());
            }
        }
        
        static MetaValue merge(MetaValue prev, MetaValue next) {
            if (!prev.value.equals(next.value)) return null;
            if (prev.stop >= next.start || prev.stop + 1L == next.start) {
                return new MetaValue(prev.start, next.stop, prev.value);
            } else {
                return null;
            }
        }
        
    }
    
    Map<String, ArrayList<MetaValue>> getOffPointMetadata() {
        return meta;
    }
    
}
