package org.lsst.ccs.gconsole.plugins.tracer;

import java.io.Serializable;
import java.util.Arrays;

/**
 * Message filter that applies a sequence of {@link FilterStep}s.
 * Instances of this class are immutable.
 *
 * @author onoprien
 */
public class MultistepFilter implements MessageFilter {

// -- Private parts : ----------------------------------------------------------
    
    final FilterStep[] steps;
    final boolean[] ors;

// -- Construction and initialization : ----------------------------------------
    
    /**
     * Constructs a multi-step filter.
     * Objects supplied to this constructor will be owned by this filter.
     * 
     * @param filters The sequence of steps.
     * @param or OR flags for steps in the sequence.
     */
    public MultistepFilter(FilterStep[] filters, boolean[] or) {
        this.steps = filters;
        this.ors = or;
    }
    
    public MultistepFilter(Descriptor descriptor) {
        ors = descriptor.getOrs();
        int n = descriptor.getSteps().length;
        steps = new FilterStep[n];
        for (int i = 0; i<n; i++) {
            steps[i] = new FilterStep(descriptor.getSteps()[i]);
        }
    }

// -- Implementing MessageFilter : ---------------------------------------------

    @Override
    public FilteredMessage apply(FilteredMessage filteredMessage) {
        boolean failedOR = false;
        boolean goodOR = false;
        for (int i=0; i<steps.length; i++) {
            FilterStep filter = steps[i];
            if (ors[i]) {
                if (goodOR) continue;
                FilteredMessage fm = filter.apply(filteredMessage);
                if (fm == null) {
                    failedOR = true;
                } else {
                    failedOR = false;
                    goodOR = true;
                    filteredMessage = fm;
                }
            } else {
                if (failedOR) return null;
                filteredMessage = filter.apply(filteredMessage);
                if (filteredMessage == null) return null;
                goodOR = false;
            }
        }
        return failedOR ? null : filteredMessage;
    }

    
// -- Saving and restoring : ---------------------------------------------------

    public Descriptor save() {
        Descriptor desc = new Descriptor();
        int n = steps.length;
        desc.steps = new FilterStep.Descriptor[n];
        for (int i = 0; i < n; i++) {
            desc.steps[i] = steps[i].save();
            
        }
        desc.ors = Arrays.copyOf(ors, ors.length);
        return desc;
    }
    
    static public class Descriptor implements Serializable, Cloneable {

        private FilterStep.Descriptor[] steps;
        private boolean[] ors;

        public boolean[] getOrs() {
            return ors;
        }

        public void setOrs(boolean[] ors) {
            this.ors = ors;
        }

        public FilterStep.Descriptor[] getSteps() {
            return steps;
        }

        public void setSteps(FilterStep.Descriptor[] steps) {
            this.steps = steps;
        }

        @Override
        protected Descriptor clone() {
            try {
                Descriptor desc = (Descriptor) super.clone();
                if (steps != null) {
                    int n = steps.length;
                    desc.steps = new FilterStep.Descriptor[n];
                    for (int i=0; i<n; i++) {
                        desc.steps[i] = steps[i].clone();
                    }
                }
                if (ors != null) desc.ors = Arrays.copyOf(ors, ors.length);
                return desc;
            } catch (CloneNotSupportedException x) {
                return null; // never
            }
        }
        
    }

}
