package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.BorderLayout;
import java.awt.Component;
import java.util.*;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.services.persist.Creator;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;

/**
 * {@link MonitorView} that divides channels into groups and displays one group at a time, using tabs to select the group.
 *
 * @author onoprien
 */
public final class TabbedGroupView extends GroupView {

// -- Fields : -----------------------------------------------------------------
    
    static public final String CREATOR_PATH = "Built-In/Grouped/Tabbed";
    
    static private final String GROUP_KEY = "_GROUP_";
    
    private JPanel root;
    private final HashMap<String,JPanel> groupToPane = new HashMap<>();


// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = MonitorView.CATEGORY,
            name = "Group Tabbed",
            path = CREATOR_PATH,
            description = "Monitoring data view that divides channels into groups, and displays one group at a time. Tabs are used to select the group to display, and tree views are used for each group.")
    public TabbedGroupView(
            @Par(def = "0", desc = "Depth of group hierarchy. If 0, the filter is expected to provide explicit group names (separated by '//' from the rest of the display path. Otherwise, should be 1 or 2, and the first 1 or 2 slash-separtated segments of the display path will be used as group names.") int depth,
            @Par(def = "false", desc = "Place tabs horizontally (left and right from the view) insted of vertically (at the top and bottom).") boolean horizontalTabs
    )
    {
        super();
        descriptor = new Descriptor();
        getDescriptor().setName("Group Tabbed");
        if (depth < 0 || depth > 2) {
            throw new IllegalArgumentException("Illegal group tree depth, must be between 0 and 2.");
        }
        getDescriptor().setDepth(depth);
        getDescriptor().setHorizontalTabs(horizontalTabs);
    }

    @Override
    public JComponent getPanel() {
        if (root == null) {
            root = new JPanel(new BorderLayout());
            rebuild();
            selectGroup(null);
        }
        return root;
    }
    
    private void rebuild() {
        if (root == null) return;
        groupToPane.clear();
        root.removeAll();
        buildSubgroups(root, null, "");
        selectGroup(currentGroup);
    }
    
    private void buildSubgroups(JComponent parent, String prefix, String segment) {
        String[] gg = getSubgroups(prefix);
        if (gg != null) {
            if (gg.length == 0) {
                JPanel comp = new JPanel(new BorderLayout());
                groupToPane.put(prefix, comp);
                comp.putClientProperty(GROUP_KEY, prefix);
                if (parent instanceof JTabbedPane) {
                    ((JTabbedPane)parent).addTab(segment, comp);
                } else {
                    parent.add(comp, BorderLayout.CENTER);
                }
            } else {
                JTabbedPane tp;
                if (parent instanceof JTabbedPane) {
                    tp = new JTabbedPane(getDescriptor().isHorizontalTabs() ? JTabbedPane.RIGHT : JTabbedPane.BOTTOM, JTabbedPane.WRAP_TAB_LAYOUT);
                    ((JTabbedPane)parent).addTab(segment, tp);
                } else {
                    tp = new JTabbedPane(getDescriptor().isHorizontalTabs() ? JTabbedPane.LEFT : JTabbedPane.TOP, JTabbedPane.WRAP_TAB_LAYOUT);
                    parent.add(tp, BorderLayout.CENTER);
                }
                tp.addChangeListener(e -> {
                    try {
                        JTabbedPane sourcePane = (JTabbedPane) e.getSource();
                        Component c = sourcePane.getSelectedComponent();
                        while (c instanceof JTabbedPane) {
                            c = ((JTabbedPane) c).getSelectedComponent();
                        }
                        String group = (String) ((JComponent)c).getClientProperty(GROUP_KEY);
                        selectGroup(group);
                    } catch (ClassCastException | NullPointerException x) {
                    }
                });
                for (String g : gg) {
                    if (g.isEmpty()) {
                        JPanel comp = new JPanel(new BorderLayout());
                        groupToPane.put(prefix, comp);
                        comp.putClientProperty(GROUP_KEY, prefix);
                        tp.addTab("     ", comp);
                    } else {
                        buildSubgroups(tp, prefix == null ? g : prefix +"/"+ g, g);
                    }
                    
                }
            }
        }
    }


// -- Customizing GroupView: ---------------------------------------------------

    @Override
    protected String[] getPath(String group) {
        String[] ss = super.getPath(group);
        if (ss.length > 2) {
            StringJoiner sj = new StringJoiner("/");
            for (int i=1; i<ss.length; i++) {
                sj.add(ss[i]);
            }
            return new String[] {ss[0], sj.toString()};
        } else {
            return ss;
        }
    }

    @Override
    protected void selectGroup(String group) {
        if (root == null) return;
        super.selectGroup(group);
    }

    
// -- Responding to changes : --------------------------------------------------
    
    @Override
    protected void groupsAdded(List<String> groups) {
        rebuild();
    }
    
    @Override
    protected void groupsRemoved(List<String> groups) {
        rebuild();
    }

    @Override
    protected void groupSelected() {
        if (root == null || (currentGroup != null && currentGroup.isEmpty())) return;
        JPanel pane = groupToPane.get(currentGroup);
        AbstractMonitorView view = getView(currentGroup);
        JComponent viewPanel = view.getPanel();
        if (pane.getComponentCount() != 1 || viewPanel != pane.getComponent(0)) {
            pane.removeAll();
            pane.add(viewPanel, BorderLayout.CENTER);
        }
    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * JavaBean that contains information required to re-create this view in its current state.
     */
    static public class Descriptor extends GroupView.Descriptor {

        private boolean horizontalTabs;

        public boolean isHorizontalTabs() {
            return horizontalTabs;
        }

        public void setHorizontalTabs(boolean horizontalTabs) {
            this.horizontalTabs = horizontalTabs;
        }

        @Override
        public Descriptor clone() {
            return (Descriptor) super.clone();
        }
        
    }

    @Override
    public Descriptor getDescriptor() {
        return (Descriptor) descriptor;
    }
    
    
// -- Pre-defined specific view : ----------------------------------------------
    
    @Create(category = MonitorView.CATEGORY,
            name = "Tabbed Tables",
            path = "Built-In/Grouped/Tabbed, Explicit Groups, Tables",
            description = "Tabbed view that expects explicitly defined groups from the filter, and displays each group in a sectioned table.")
    static public TabbedGroupView createTabbedExplicitGroupsSectionedTables() {
        TabbedGroupView.Descriptor viewDesc = new TabbedGroupView.Descriptor();
        Creator.Descriptor creDesc = new Creator.Descriptor(MonitorView.CATEGORY, TabbedGroupView.CREATOR_PATH, "0", "false");
        viewDesc.setCreator(creDesc);
        viewDesc.setDepth(0);
        viewDesc.setDefaultView(new SectionedTableView.Descriptor());
        return (TabbedGroupView) PersistenceService.getService().make(viewDesc);
    }
    
}
