
package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.*;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeListener;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.CommandState;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.bus.states.OperationalState;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusAggregator;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusEvent;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.services.lock.Locker;
import org.lsst.ccs.gconsole.services.lock.LockService;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;

/**
 * A {@link Browser} that works with one particular subsystem.
 *
 * @author onoprien
 */
public class BrowserOneSubsystem extends Browser {

// -- Fields : -----------------------------------------------------------------
    
    private Descriptor descriptor; // at least agent name is always set

    private AgentPanel agentPanel;
    private LockLabel lockLabel;
    private JLabel levelLabel;
    private JFormattedTextField levelField;
    
    private Locker agent; // null if the agent is offline and unlocked
        
    private LockService.Listener lockServiceProxyListener; // added/removed agents
    private ChangeListener agentHandleListener; // lock-related changes in the agent this browser commands, used to update lock panel
    
// -- Life cycle : -------------------------------------------------------------
    
    @Override
    public JComponent getPanel() {
        if (browserPanel == null) {
            super.getPanel();
            
            // Agent dictionary panel:
            
            agentPanel = new AgentPanel();
            browserPanel.add(agentPanel, BorderLayout.CENTER);
            
            // Lock panel:
            
            Box lockPanel = Box.createHorizontalBox();
            browserPanel.add(lockPanel, BorderLayout.NORTH);
            
            lockPanel.add(Box.createRigidArea(Const.HDIM));
            lockLabel = new LockLabel();
            lockPanel.add(lockLabel);
            lockPanel.add(Box.createRigidArea(Const.HDIM));
            levelLabel = new JLabel("Level:");
            lockPanel.add(levelLabel);
            lockPanel.add(Box.createRigidArea(Const.HDIM));
            levelField = new JFormattedTextField(0);
            lockPanel.add(levelField);
            levelField.setColumns(3);
            levelField.setMaximumSize(new Dimension(levelField.getPreferredSize().width, Integer.MAX_VALUE));
            levelField.addActionListener(e -> {
                try {
                    String s = levelField.getText();
                    int level = Integer.parseInt(s);
                    service.executeOperation(LockService.Operation.LEVEL, getAgentName(), level);
                } catch (NumberFormatException x) {
                    Console.getConsole().error("Illegal level format", x);
                }
            });
            lockPanel.add(Box.createRigidArea(Const.HDIM));
            lockPanel.add(stateLabel);
            lockPanel.add(Box.createHorizontalGlue());
            
            // Start listening:
            
            agentHandleListener = e -> updateLockPanel();

            lockServiceProxyListener = new LockService.Listener() {
                @Override
                public void agentsAdded(Locker... agentHandles) {
                    if (agent == null) {
                        String name = getAgentName();
                        for (Locker agentHandle : agentHandles) {
                            if (name.equals(agentHandle.getName())) {
                                agent = agentHandle;
                                agent.addListener(agentHandleListener);
                                updateLockPanel();
                                agentPanel.setAgent(agent);
                                break;
                            }
                        }
                    }
                }
                @Override
                public void agentsRemoved(Locker... agentHandles) {
                    if (agent != null) {
                        for (Locker agentHandle : agentHandles) {
                            if (agent.equals(agentHandle)) {
                                agentPanel.setAgent(null);
                                agent.removeListener(agentHandleListener);
                                agent = null;
                                updateLockPanel();
                                break;
                            }
                        }
                    }
                }
            };
            service.addListener(lockServiceProxyListener);
                        
        }
        return browserPanel;
    }

    @Override
    public void shutdown() {
        if (agent != null) {
            agent.removeListener(agentPanel);
        }
        LockService.getService().removeListener(lockServiceProxyListener);
        super.shutdown();
    }

    @Override
    public String getName() {
        return descriptor.getName();
    }

    @Override
    public String getAgentName() {
        return descriptor.getAgent();
    }
    
// -- Updating : ---------------------------------------------------------------
    
    private void updateLockPanel() {
        if (agent == null) stateLabel.setText("OFFLINE");
        RenderedLock rl = renderLock(agent);
        lockLabel.update(rl);
        levelField.setValue(agent == null ? 0 : agent.getLevel());
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private final class LockLabel extends JLabel implements HasPopupItems {
        
        private RenderedLock renderedLock;
        
        LockLabel() {
            setMinimumSize(new Dimension(ICON_UNLOCKED_BLACK.getIconWidth(), ICON_UNLOCKED_BLACK.getIconHeight()));
            addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    if (e.getClickCount() == 2 && renderedLock != null) {
                        Action act = renderedLock.defaultAction;
                        if (act != null) {
                            act.actionPerformed(null);
                        }
                    }
                }
            });            
        }
        
        void update(RenderedLock renderedLock) {
            this.renderedLock = renderedLock;
            setIcon(renderedLock.icon);
            setToolTipText(renderedLock.tooltip);
            setEnabled(renderedLock.enabled);
        }

        @Override
        public JPopupMenu modifyPopupMenu(JPopupMenu menu, Component componentt, Point point) {
            if (renderedLock != null && renderedLock.availableActions != null) {
                renderedLock.availableActions.forEach(a -> menu.add(new JMenuItem(a)));
            }
            return menu;
        }
        
    }

// -- Saving/restoring : -------------------------------------------------------
    
    @Override
    public Descriptor save() {
        if (agentPanel == null) {
            descriptor.setAgentPanel(null);
        } else {
            descriptor.setAgentPanel(agentPanel.save());
        }
        if (browserPanel != null) {
            descriptor.setPage(DataPanelDescriptor.get(browserPanel));
        }
        return descriptor;
    }
    
    @Override
    public void restore(Serializable descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
            if (agentPanel != null) {
                AgentPanel.Descriptor apDesc = this.descriptor.getAgentPanel();
                if (apDesc != null) agentPanel.restore(apDesc);
            }
        } else {
        }
    }
    
    static public class Descriptor extends Browser.Descriptor {
        
    }
    
}
