package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import java.util.AbstractMap.SimpleEntry;
import java.util.stream.Collectors;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.PersistableAgentChannelsFilter;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusEvent;

/**
 * An adapter that simplifies implementing {@code MonitorView}.
 * <p>
 * Provided implementation of the {@link AgentStatusListener} should not be overridden.
 * Subclasses should override one of the two sets of methods (both called on EDT):
 * <ul>
 * <li>addChannels(...), updateChannels(...), and removeChannels(...)
 * <li>resetChannels(), update(), and, optionally, createChannel(...). Implementations can use 
 *     {@code data} field that contains a map of display paths to display channels.
 * </ul>
 *
 * @author onoprien
 */
abstract public class AbstractMonitorView implements PersistableMonitorView {

// -- Fields : -----------------------------------------------------------------
    
    protected PersistableAgentChannelsFilter filter;
    protected MonitorFormat formatter;
    protected List<MonitorField> fields;
    protected List<MonitorField> compactFields;
    
    /** Map of display paths to display channels, to be used by subclasses to populate the view. In addition order. */
    protected final LinkedHashMap<String,DisplayChannel> data = new LinkedHashMap<>();
    
    /** Policy on dealing with removed channels. */
    private volatile boolean discardRemovedChannels = false;
    
    private ArrayList<String> disconnectedAgents; // agents to be purged when they reconnect; null if empty.

    
// -- Setters : ----------------------------------------------------------------

    /**
     * Sets the policy on dealing with channels reported as removed by the status aggregator.
     * If {@code true}, the channels are immediately removed from the {@code data} map.
     * If {@code false} (default), the channels are retained until and unless an identically
     * named agent appears in future status aggregator events.
     * 
     * @param discardRemovedChannels Policy value.
     */
    public void setDiscardRemovedChannels(boolean discardRemovedChannels) {
        this.discardRemovedChannels = discardRemovedChannels;
    }
    
    
// -- Implement AgentStatusListener : ------------------------------------------

    @Override
    public void connect(AgentStatusEvent event) {
        // ignore
    }

    @Override
    public void configure(AgentStatusEvent event) {
        if (!event.isEmpty()) {
            SwingUtilities.invokeLater(() -> onConnect(event));
        }
    }

    @Override
    public void statusChanged(AgentStatusEvent event) {
        if (!event.isEmpty()) {
            SwingUtilities.invokeLater(() -> onChange(event));
        }
    }

    @Override
    public void disconnect(AgentStatusEvent event) {
        if (!event.isEmpty()) {
            SwingUtilities.invokeLater(() -> onDisconnect(event));
        }
    }
    
    private void onConnect(AgentStatusEvent event) {
        List<AgentInfo> agents = event.getAgents();
        List<Map.Entry<String,AgentChannel>> channels = event.getAddedChannels().stream()
                .flatMap(channel -> filter.getDisplayPaths(channel).stream().map(displayPath -> new SimpleEntry<>(displayPath,channel)))
                .collect(Collectors.toList());
        addChannels(agents, channels);
    }
    
    private void onChange(AgentStatusEvent event) {
        List< Map.Entry<String,Map.Entry<AgentChannel,List<String>>> > channels = event.getStatusChanges().entrySet().stream()
                .flatMap(e -> filter.getDisplayPaths(e.getKey()).stream().map(displayPath -> new SimpleEntry<>(displayPath,e)))
                .collect(Collectors.toList());
        updateChannels(channels);
    }
    
    private void onDisconnect(AgentStatusEvent event) {
        List<AgentInfo> agents = event.getAgents();
        List<Map.Entry<String,AgentChannel>> channels = event.getRemovedChannels().stream()
                .flatMap(channel -> filter.getDisplayPaths(channel).stream().map(displayPath -> new SimpleEntry<>(displayPath,channel)))
                .collect(Collectors.toList());
        removeChannels(agents, channels);
    }
    
    
// -- Level 1 SPI for subclasses (called on EDT) : -----------------------------
    
    /**
     * Adds channels to this view.
     * Any previously existing channels from the same agents are removed.
     * 
     * @param agents Newly connected agents these channels belong to.
     * @param channels New channels, each entry contains display path and channel.
     */
    public void addChannels(List<AgentInfo> agents, List<Map.Entry<String,AgentChannel>> channels) {
        boolean needReset = false;
        LinkedHashSet<DisplayChannel> updatedChannels = new LinkedHashSet<>();
        
        // Remove channels from added agents if they were not discarded earlier
        
        if (!discardRemovedChannels) {
            updatedChannels = doRemove(agents.stream().map(a -> a.getName()).collect(Collectors.toList()), false);
        }
        
        // Add agents

        for (Map.Entry<String,AgentChannel> e : channels) {
            String dp = e.getKey();
            AgentChannel c = e.getValue();
            DisplayChannel dc = data.get(dp);
            if (dc == null) {
                dc = createChannel(dp);
                data.put(dp, dc);
                needReset = true;
            } else {
                updatedChannels.add(dc);
            }
            dc.appendChannel(c);
        }
        
        // Remove display channels that have no agent channels
        
        Iterator<DisplayChannel> it = updatedChannels.iterator();
        while (it.hasNext()) {
            DisplayChannel dc = it.next();
            if (dc.getChannels().isEmpty()) {
                data.remove(dc.getPath());
                it.remove();
                needReset = true;
            }
        }
        
        // Call SPI hooks
        
        if (needReset) {
            resetChannels();
        } else {
            for (DisplayChannel dc : updatedChannels) {
                dc.update(null);
            }
            update();
        }        
    }
    
    /**
     * Updates channels in this view.
     * 
     * @param channels Updated channels, each element contains: {display path : {channel : list of modified attributes}}.
     */
    public void updateChannels(List< Map.Entry<String,Map.Entry<AgentChannel,List<String>>> > channels) {
        doUpdate(channels);
    }
    
    /**
     * Removes channels from this view.
     * If the {@code discardRemovedChannels} flag is not set, the channels are updated but not removed.
     * 
     * @param agents Names of disconnected agents.
     * @param channels Removed channels, each entry contains display path and channel.
     */
    public void removeChannels(List<AgentInfo> agents, List<Map.Entry<String,AgentChannel>> channels) {
        List<String> agentNames = agents.stream().map(a -> a.getName()).collect(Collectors.toList());
        if (discardRemovedChannels) {
            if (!doRemove(agentNames, true).isEmpty()) {
                resetChannels();
            } else {
                update();
            }
        } else {
            if (disconnectedAgents == null) {
                disconnectedAgents = new ArrayList<>(agentNames);
            } else {
                disconnectedAgents.addAll(agentNames);
            }
            doUpdate(channels.stream()
                    .map(e -> new SimpleEntry<String, Map.Entry<AgentChannel, List<String>>>( e.getKey(), new SimpleEntry<AgentChannel,List<String>>(e.getValue(), null) ))
                    .collect(Collectors.toList()));
        }
    }
    
    
// -- Local methods used for going from stage 1 to stage 2 : -------------------
    
    /**
     * Removes data channels that belong to the specified agents from {@code data}.
     * 
     * @param agents Agents to remove.
     * @return True if display channels have been removed from {@code data}.
     */    
    private LinkedHashSet<DisplayChannel> doRemove(List<String> agents, boolean purge) {
        LinkedHashSet<DisplayChannel> out = new LinkedHashSet<>(data.size()*2);
        Iterator<Map.Entry<String,DisplayChannel>> it = data.entrySet().iterator();
        while (it.hasNext()) {
            DisplayChannel dc = it.next().getValue();
            List<AgentChannel> oldChannels = dc.getChannels();
            if (oldChannels.stream().anyMatch(a -> agents.contains(a.getAgentName()))) {
                ArrayList<AgentChannel> newChannels = new ArrayList<>(oldChannels.size());
                for (AgentChannel c : oldChannels) {
                    if (!agents.contains(c.getAgentName())) {
                        newChannels.add(c);
                    }
                }
                dc.setChannels(newChannels);
                if (newChannels.isEmpty()) {
                    out.add(dc);
                    if (purge) it.remove();
                } else {
                    if (purge) dc.update(null);
                }
            }
        }
        return out;
    }
    
    private void doUpdate(List< Map.Entry<String,Map.Entry<AgentChannel,List<String>>> > channels) {
        if (channels == null) {
            data.values().forEach(dc -> {
                dc.update(null);
            });
        } else {
            for (Map.Entry<String, Map.Entry<AgentChannel, List<String>>> e : channels) {
                String displayPath = e.getKey();
                DisplayChannel dc = data.get(displayPath);
                List<String> attributes = e.getValue().getValue();
                if (dc != null) dc.update(attributes);
            }
        }
        update();
    }

    
// -- Level 2 SPI for subclasses (called on EDT) : -----------------------------
    
    /**
     * Called on EDT whenever channels have been added or removed.
     * Should be implemented to completely rebuild the view based on {@code data} field.
     * Empty implementation is provided.
     */
    protected void resetChannels() {}
    
    /**
     * Called on EDT at the end of each update, after all DisplayChannel.update(...) methods.
     * Empty implementation is provided.
     */
    protected void update() {}
    
    /**
     * Called on EDT to create new display channels whenever this view is informed of new display paths.
     * The default implementation constructs an instance of {@link DisplayChannelMulti} without any {@link Updatable} target.
     * 
     * @param displayPath Display path.
     * @return An instance of {@code DisplayChannel} to be put into the {@code data} map for the specified display path key.
     */
    protected DisplayChannel createChannel(String displayPath) {
        return new DisplayChannelMulti(displayPath);
    }


// -- Implement MonitorView : --------------------------------------------------
    
    /**
     * Returns {@code true} if this view has no channels to display.
     * Implemented to return {@code true} if the {@code data} map is empty.
     * 
     * @return True if this view has no display channels.
     */
    @Override
    public boolean isEmpty() {
        return data.isEmpty();
    }
    
    /**
     * Computes group name based on the display path returned by the filter.
     * The default implementation splits around the last "/".
     * 
     * @param displayPath Display path.
     * @return Group name, or {@code null} if this display path does not belong to any group.
     */
    public String getGroup(String displayPath) {
        int i = displayPath.lastIndexOf("/");
        return displayPath.substring(i+1);
    }

    public List<String> getGroups() {
        List<String> out = null;
        
        // try to parse filter.getDisplayChannels()
        
        AgentChannelsFilter f = getFilter();
        if (f != null) {
            List<String> channels = f.getDisplayChannels();
            if (channels != null) {
                LinkedHashSet<String> groups = new LinkedHashSet<>();
                channels.forEach(displayPath -> {
                    String group = getGroup(displayPath);
                    if (group != null) {
                        groups.add(group);
                    }
                });
                out = new ArrayList<>(groups);
            }
        }
        
        // parse current list of channels

        if (out == null) {
            LinkedHashSet<String> groups = new LinkedHashSet<>();
            data.keySet().forEach(displayPath -> {
                String group = getGroup(displayPath);
                if (group != null) groups.add(group);
            });
            return new ArrayList<>(groups);
        }
        
        return out;
    }

    @Override
    public PersistableAgentChannelsFilter getFilter() {
        return filter;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        if (!(filter instanceof PersistableAgentChannelsFilter)) throw new IllegalArgumentException(getClass().getName() +" only accepts AbstractChannelsFilter filters");
        
        // Save reference to the filter for future use
        
        this.filter = (PersistableAgentChannelsFilter) filter;
        
        // Set {@code compactFields}
        
        Descriptor desc = getDescriptor();
        List<String> ff;
        
        String[] ss = desc == null ? null : desc.getCompactFields();
        if (ss == null) {
            ff = filter.getFields(true);
        } else {
            ff = Arrays.asList(ss);
        }
        if (ff == null) {
            compactFields = MonitorTable.DEFAULT_COMPACT_FIELDS;
        } else {
            compactFields = ff.stream().map(f -> MonitorField.getInstance(f)).filter(mf -> mf != null).collect(Collectors.toList());
        }
        
        // Set {@code fields}
        
        ss = desc == null ? null : desc.getFields();
        if (ss == null) {
            ff = filter.getFields(false);
        } else {
            ff = Arrays.asList(ss);
        }
        if (ff == null) {
            fields = MonitorTable.DEFAULT_FIELDS;
        } else {
            fields = ff.stream().map(f -> MonitorField.getInstance(f)).filter(mf -> mf != null).collect(Collectors.toList());
        }
    }

    @Override
    public MonitorFormat getFormater() {
        return formatter;
    }

    @Override
    public void setFormatter(MonitorFormat formatter) {
        this.formatter = formatter;
    }
    
    
// -- Utility methods : --------------------------------------------------------

    /**
     * Retrieves a list {@code MonitorField} for display channel groups from the filter.
     * The default implementation calls {@code getFilter().getGroupFields(true)} and
     * then applies {@code stringToField(...)}.
     * 
     * @return List of fields for display channel groups, or {@code null} if the filter does not provide the list.
     */
    public List<MonitorField> getGroupFields() {
        AgentChannelsFilter f = getFilter();
        if (f == null) return null;
        List<String> channels = f.getDisplayChannels();
        if (channels == null) return null;
        List<String> fields = f.getFields(true);
        if (fields == null || fields.size() != channels.size()) return null;
        return fields.stream().map(s -> MonitorFormat.stringToField(s)).collect(Collectors.toList());
    }

    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * JavaBean that contains information required to re-create this view in its current state.
     */
    static public class Descriptor extends PersistableMonitorView.Descriptor {

        private String[] fields;
        private String[] compactFields;

        public String[] getCompactFields() {
            return compactFields;
        }

        public void setCompactFields(String[] compactFields) {
            this.compactFields = compactFields;
        }

        public String[] getFields() {
            return fields;
        }

        public void setFields(String[] fields) {
            this.fields = fields;
        }

        @Override
        public Descriptor clone() {
            return (Descriptor) super.clone();
        }
        
    }

    @Override
    public abstract Descriptor getDescriptor();

    @Override
    public Descriptor save() {
        return (Descriptor) PersistableMonitorView.super.save();
    }

}
