package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.util.*;
import javax.swing.DefaultListCellRenderer;
import javax.swing.DefaultListModel;
import javax.swing.JList;
import javax.swing.ListSelectionModel;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.command.annotations.Command.CommandCategory;

/**
 * Graphics component that displays a list of commands for the currently selected component.
 *
 * @author onoprien
 */
public class CommandListPanel extends JList<DictionaryCommand> implements TreeSelectionListener {

// -- Fields : -----------------------------------------------------------------
    
    private final AgentPanel aPanel;
    private final ComponentTree componentTree;
    private final CommandComparator comparator = new CommandComparator();
    
    private Dictionary currentDictionary;
    private DictionaryCommand currentCommand;

// -- Life cycle : -------------------------------------------------------------
    
    public CommandListPanel(AgentPanel agentPanel){
        aPanel = agentPanel;
        componentTree = agentPanel.componentTree;
        getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        setCellRenderer(new CommandListRenderer());
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    public boolean isVisibleCommandSelected() {
        DictionaryCommand dc = getSelectedValue();
        return dc == null ? false : !Browser.isCommandLocked(dc, componentTree.getAgent());
    }
    
    
// -- Responding to external events : ------------------------------------------

    /**
     * Called when the user selects a component.
     * @param tse Event.
     */
    @Override
    public void valueChanged(TreeSelectionEvent tse) {
        update();
    }
    
    /**
     * Updates this panel based on current settings and selected component.
     */
    public void update() {
        if (currentDictionary != null) {
            currentCommand = getSelectedValue();
        }
        currentDictionary = componentTree.getSelectedUserObject();
        DefaultListModel<DictionaryCommand> model = new DefaultListModel<>();
        if (currentDictionary != null) {
            ArrayList<DictionaryCommand> commandList = new ArrayList<>(currentDictionary.size());
            for (DictionaryCommand dc : currentDictionary) {
                if (!((aPanel.descriptor.isHideSystem() && dc.getCategory().equals(CommandCategory.SYSTEM)) || 
                      (aPanel.descriptor.isHideLocked() && Browser.isCommandLocked(dc, componentTree.getAgent())))) {
//                if ((!descriptor.isHideLocked() || !Browser.isCommandLocked(dc, currentDictionary)) && (!descriptor.isHideSystem() || dc.getCategory() != CommandCategory.SYSTEM)) {
                    commandList.add(dc);
                }
            }
            commandList.sort(comparator);
            commandList.forEach(dc -> model.addElement(dc));
        }
        setModel(model);
        if (currentDictionary != null && currentCommand != null) {
            setSelectedValue(currentCommand, true);
        }
    }
    
    
// -- Local methods and classes : ----------------------------------------------
    
    /**
     * Renderer that displays command name and grays out invisible commands.
     */
    private class CommandListRenderer extends DefaultListCellRenderer {
        @Override
        public Component getListCellRendererComponent(JList<?> jlist, Object e, int i, boolean isSelected, boolean cellHasFocus) {
            super.getListCellRendererComponent(jlist, e, i, isSelected, cellHasFocus);
            DictionaryCommand dc = (DictionaryCommand)e;
            setText(dc.getCommandName());
            int fontStyle;
            if (Browser.isCommandLocked(dc, componentTree.getAgent())) {
                setForeground(Color.GRAY);
                fontStyle = Font.ITALIC;
            } else {
                fontStyle = Font.PLAIN;
            }
            if (dc.getCategory() == CommandCategory.USER) {
                fontStyle |= Font.BOLD;
            }
            setFont(getFont().deriveFont(fontStyle));
            return this;
        }
    }
    
    private class CommandComparator implements Comparator<DictionaryCommand> {
        @Override
        public int compare(DictionaryCommand c1, DictionaryCommand c2) {
            boolean v1 = Browser.isCommandLocked(c1, componentTree.getAgent());
            boolean v2 = Browser.isCommandLocked(c2, componentTree.getAgent());
            if (v1 == v2) {
                int out = c1.getCommandName().compareTo(c2.getCommandName());
                return out == 0 ? c1.getArguments().length - c2.getArguments().length : out;
            } else {
                return v1 ? 1 : -1;
            }
        }
    }
}
