package org.lsst.ccs.gconsole.services.aggregator;

import java.time.Instant;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.DataProviderInfo.Attribute;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.EmbeddedObjectDeserializationException;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel.Key;
import org.lsst.ccs.gconsole.base.Console;

/**
 * Handles status aggregator data for a specific {@code Agent}.
 * 
 * Implementation notes:<ul>
 * <li>
 * A handle exists for every agent present on the buses, created when either the data dictionary of the first status message is received.
 * The handle if destroyed when the agent goes OFFLINE or disconnects.
 * <li>
 * A handle is configured once the data dictionary is received from an agent service. At this point,
 * existing status aggregator listeners are added to this handle and receive "connect" and "configure" events;
 * the list of local listener handles ({@code listeners}) becomes non-null (empty if there are no listeners);
 * the map of paths to channels (processed dictionary, {@code channels}) becomes non-null.
 * <li>
 * {@code noWatchedChannels} flag reflects the absence of listeners interested in channels from this agent.
 * If {@code true}, all channel values except non-monitoring trending are {@code null} and not updated.
 * <li>
 * All methods except getters are executed on the status aggregator thread. Listeners are notified on that thread as well.
 * <li>
 * The state of this handle is modified while holding its monitor lock.
 * Getters are synchronized on this lock as well.
 * Objects returned by the getters are unaffected by future changes in this handle.
 * </ul>
 */
class AgentHandle {
    
// -- Fields : -----------------------------------------------------------------
    
    private final AgentInfo agentInfo;
    final AgentStatusAggregator aggregator;
        
    private volatile StateBundle state; // last recordered agent state
    private volatile ConfigurationInfo configuration; // last received configuration
    
    /* Listeners; null if the data dictionary has not been received yet. */
    private ArrayList<ListenerHandle> listeners;
    /* Map of paths to channels; null if the data dictionary has not been received yet. */
    LinkedHashMap<String, AgentChannel> channels;
    /* Map that stores alternate keys that map published values to paths */
    LinkedHashMap<String, String> alternatePublishedKeyToPathMap;
    
    /* List to keep track of which data does not have an AgentChannel */
    ArrayList<String> orphanedData;
    
    /* True if this handle has no listeners interested in channels. */
    private boolean noWatchedChannels = true;
    
    final String AGENT_PREFIX;
    final String TIME_PATH;
    final String STATE_PREFIX;
    final String CONFIG_PREFIX;

    private static final Logger LOG = Logger.getLogger(AgentHandle.class.getName());

// -- Life cycle : -------------------------------------------------------------
    
    /** Constructs an instance */
    AgentHandle(AgentInfo agentInfo, AgentStatusAggregator aggregator) {
        this.aggregator = aggregator;
        this.agentInfo = agentInfo;
        String agent = agentInfo.getName();
        AGENT_PREFIX = agent +"/";
        TIME_PATH = agent +"/runtimeInfo/current time";
        STATE_PREFIX = agent +"/state/";
        CONFIG_PREFIX = agent +"/configuration/";
    }


// -- Getters : ----------------------------------------------------------------
    
    /** Returns the agent descriptor. */
    AgentInfo getAgent() {
        return agentInfo;
    }
    
    /** Returns the channel specified by the given path, or {@code null} if there is no channel with this path. */
    synchronized AgentChannel getChannel(String path) {
        return channels == null ? null : channels.get(path);
    }
    
    /** Returns the last recorded state. */
    StateBundle getState() {
        return state;
    }
    
    /** Returns the last recorded configuration. */
    ConfigurationInfo getConfiguration() {
        return configuration;
    }
    
    /** Returns {@code true} if the dictionary has already been processed. */
    boolean noDictionary() {
        return channels == null;
    }
    
    /**
     * Returns a collection of all channels obtained from the dictionary. 
     */
    synchronized List<AgentChannel> getDictionaryChannels() {
        return channels == null ? Collections.emptyList() : new ArrayList<>(channels.values());
    }
    
    /**
     * Returns a list of watched channels. 
     */
    synchronized List<AgentChannel> getWatchedChannels() {
        return noWatchedChannels ? Collections.emptyList() : new ArrayList<>(channels.values());
    }
    
    
// -- Processing messages, disconnects, and dictionary : -----------------------
    
    void onMessage(StatusMessage mess) {
        
        // always keep references to latest state and configuration

        boolean firstState = state == null;
        boolean firstConfig = configuration == null;        
        state = mess.getState();
        if (mess instanceof StatusConfigurationInfo) {
            try {
                configuration = ((StatusConfigurationInfo) mess).getConfigurationInfo();
            } catch (EmbeddedObjectDeserializationException x) {
                LOG.log(Level.FINE, x, () -> "Failed to deserialize ConfigurationInfo from "+ agentInfo.getName());
            }
        }
        
        // No dictionary yet, ignore the message
        
        if (noDictionary()) {
            return;
        }
        
        // No channels are being watched - ignore unless non-monitoring, infrequently updated channels:
        
        if (noWatchedChannels) {
            if (mess instanceof StatusSubsystemData && ((StatusSubsystemData) mess).getDataAttributes().get("taskName") == null) {
                synchronized (this) {
                    readTrendingData(mess, null);
                }
            }
            return;
        }

        // Detect status changes and accumulate into event:
        
        MutableAgentStatusEvent event = new MutableAgentStatusEvent(aggregator, mess);

        synchronized (this) { // keep lock while modifying data

            // Agent and channel states:

            if (mess instanceof StatusStateChangeNotification) {
                if (firstState) {
                    readState(state, event);
                } else {
                    StateBundle diff = state.diffState(((StatusStateChangeNotification) mess).getOldState());
                    readState(diff, event);
                }
            }
            
            // Configuration :
            
            else if (mess instanceof StatusConfigurationInfo) {
                ConfigurationInfo config = ((StatusConfigurationInfo)mess).getConfigurationInfo();
                readConfiguration(firstConfig ? config.getAllParameterInfo() : config.getLatestChanges(), event);
            }

            // Trending data :
            
            else {
                readTrendingData(mess, event);
            }
            
            // Time stamp:
            
            {
                Instant time = mess.getCCSTimeStamp().getUTCInstant();
                AgentChannel channel = channels.get(TIME_PATH);
                if (channel != null) {
                    boolean changed = channel.set(time);
                    if (changed) {
                        event.addChange(channel, AgentChannel.Key.VALUE);
                    }
                }
            }
        
        } // finished modifying data, release lock on Agent handle

        // Notify listeners:
        
        if (!event.isEmpty()) {
            listeners.forEach(lh -> {
                AgentStatusEvent e = lh.filter(event);
                if (!e.isEmpty()) {
                    try {
                        lh.global.listener.statusChanged(e);
                    } catch (RuntimeException x) {
                        Console.getConsole().getLogger().warn("Error processing status aggregator event by " + lh.global.listener.getClass(), x);
                    }
                }
            });
        }
        
    }

    /** Called when the agent disconnects, and this agent handle needs to be discarded. */
    void onDisconnect() {
        if (channels == null) return; // no dictionary yet, no connects were sent
        AgentChannel phase = channels.get(STATE_PREFIX + PhaseState.class.getSimpleName());
        if (phase != null) {
            phase.set(PhaseState.OFF_LINE.toString());
        }
        List<AgentChannel> removed = new ArrayList<>(channels.values());
        removed.forEach(ch -> ch.set(AgentChannel.Key.STATE, DataProviderState.OFF_LINE));
        AgentStatusEvent event = new AgentStatusEvent(aggregator, Collections.singletonList(agentInfo), null, Collections.emptyMap(), Collections.emptyList(), removed);
        listeners.forEach(lh -> {
            AgentStatusEvent e = lh.filter(event);
            try {
                lh.global.listener.disconnect(e);
            } catch (RuntimeException x) {
                Console.getConsole().getLogger().warn("Error processing status aggregator event by "+ lh.global.listener.getClass(), x);
            }
        });
    }

    synchronized AgentChannel getChannelForPublishedData(String publishedPath, String prefix, boolean canBeNull) {
        String fullInternalPath = prefix + publishedPath;
        fullInternalPath = alternatePublishedKeyToPathMap.getOrDefault(fullInternalPath,fullInternalPath);
        AgentChannel channel = channels.get(fullInternalPath);
        if ( channel == null && ! canBeNull ) {
            if ( !orphanedData.contains(fullInternalPath)) {
                orphanedData.add(fullInternalPath);
                LOG.log(Level.FINE, "No channel found for published data: {0} {1}", new Object[]{prefix, publishedPath});
            }
        }
        return channel;
    }
    
    /** Called when the data dictionary is received from the service. */
    synchronized void onDictionary(DataProviderDictionary dictionary) {
        if (noDictionary()) {
            
            // Process dictionary:

            //Check if the remote agent is using full paths
            //When full paths are used we don't need to evaluate the
            //published data path
            boolean useFullPaths = "true"
                .equals(agentInfo.getAgentProperty("org.lsst.ccs.use.full.paths", "false").toLowerCase());

            List<DataProviderInfo> dictChannels = dictionary.getDataProviderInfos();
            channels = new LinkedHashMap<>(dictChannels.size() * 2, .99f);
            alternatePublishedKeyToPathMap = new LinkedHashMap<>(dictChannels.size() * 2, .99f);
            orphanedData = new ArrayList<>();
            String section = null;
            for (DataProviderInfo dc : dictChannels) {
                String path;
                String trendingKey;
                String publishedKey = null;                
                DataProviderInfo.Type type;
                try {
                    type = DataProviderInfo.Type.valueOf(dc.getAttributeValue(Attribute.DATA_TYPE));
                } catch (IllegalArgumentException x) {
                    continue;
                }
                
                //Check if the Dictionary entry provides a published path attribute.
                //If it does we have to use it as the definitive key to map
                //published values and Channel paths.
                String publishedPathAttribute = getPublishedDataAttribute(dc);
                switch (type) {
                    case TRENDING: 
                    case MONITORING:
                        path = dc.getPath();
                        trendingKey = dc.getKey();
                        if (trendingKey == null || trendingKey.equals(path) || trendingKey.isEmpty()) {
                            trendingKey = null;
                        }
                        path = AGENT_PREFIX + path;

                        if ( publishedPathAttribute != null ) {
                            publishedKey = AGENT_PREFIX + publishedPathAttribute;
                        } else if ( trendingKey != null && !useFullPaths ) {
                            publishedKey = AGENT_PREFIX + trendingKey;
                        }
                        break;
                    case CONFIGURATION:
                        path = dc.getPath();
                        path = CONFIG_PREFIX + (path.isEmpty() ? "" : (path +"/")) + dc.getKey();
                        trendingKey = null;

                        if ( publishedPathAttribute != null ) {
                            publishedKey = CONFIG_PREFIX + publishedPathAttribute;
                        } else if (!useFullPaths) {
                            String dataPath = dc.getPath();
                            int indx = dataPath.lastIndexOf("/") + 1;
                            publishedKey = dataPath.isEmpty() ? null : CONFIG_PREFIX + (indx > 0 ? dataPath.substring(indx) : dataPath) + "/" + dc.getKey();
                        }
                        break;
                    case STATE:
                        String key = dc.getKey();
                        key = key.substring(key.lastIndexOf('.') + 1);
                        path = dc.getPath();
                        path = STATE_PREFIX + (path.isEmpty() ? "" : (path +"/")) + key;
                        trendingKey = null;
                        break;
                    default:
                        continue;
                }
                
                if ( publishedKey != null && !publishedKey.equals(path)) {
                    alternatePublishedKeyToPathMap.put(publishedKey, path);
                }
                
                HashMap<String, Object> attributes;
                DataProviderInfo.Attribute[] aa = dc.getAttributes();
                if (aa.length == 0 && trendingKey == null) {
                    attributes = null;
                } else {
                    attributes = new HashMap<>(aa.length * 2);
                    if (trendingKey != null) {
                        attributes.put(Key.TRENDING, trendingKey);
                    } 
                    for (DataProviderInfo.Attribute a : aa) {
                        String av = dc.getAttributeValue(a);
                        if (a == Attribute.DESCRIPTION && DataProviderInfo.Type.MONITORING.name().equals(dc.getAttributeValue(Attribute.DATA_TYPE))) {
                            int ii = av.indexOf("\\");
                            if (ii != -1) {
                                section = av.substring(0, ii);
                                av = av.substring(ii + 1);
                            }
                            if (section != null) {
                                attributes.put(Key.SECTION, section);
                            }
                        }
                        attributes.put(a.getName(), av);
                    }
                }
                BasicChannel channel = new BasicChannel(path, agentInfo, attributes, null);
                channels.put(path, channel);
            }
            BasicChannel timeChannel = new BasicChannel(TIME_PATH, agentInfo, null, null);
            channels.put(TIME_PATH, timeChannel);
            channels = new LinkedHashMap<>(channels);
            
            // Add existing global listeners, send "connect" and "configure"

            listeners = new ArrayList<>();
            for (AgentStatusAggregator.GlobalListenerHandle global : aggregator.listeners) {
                MutableAgentStatusEvent event = new MutableAgentStatusEvent(aggregator);
                addListener(global, event); // "connect" sent and event is moddified
                if (!event.isEmpty()) {
                    event.validate();
                    try {
                        global.listener.configure(event);
                    } catch (RuntimeException x) {
                        Console.getConsole().getLogger().warn("Error processing status aggregator event by " + global.listener.getClass(), x);
                    }
                }
            }
            listeners.trimToSize();
        }
    }
    
    //This method is overwritten by the corresponding test class.
    String getPublishedDataAttribute(DataProviderInfo dc) {                    
        return dc.getAttributeValue(Attribute.PUBLISHED_PATH);
    }

    

// -- Adding/removing listeners : ----------------------------------------------
    
    /**
     * Called to add a listener. Does the following, all items optional, depending on agents and channels the listener is interested in:<ul>
     * <li> Creates a local (agent-specific) handle for the listener and  adds it to {@code listeners} list. Sends "connect" event.
     * <li> Creates new watched channels and adds them to {@code channels} map.
     * <li> Updates the event with relevant channels.</ul>
     * Notice that "configure" event is not sent from this method.
     * 
     * @param global Global handle for the listener.
     * @param event Status aggregator "configure" event to be updated with channels from this agent handle.
     */
    void addListener(AgentStatusAggregator.GlobalListenerHandle global, MutableAgentStatusEvent event) {
        if (noDictionary()) return;
        String agentName = agentInfo.getName();
        boolean watchAgent = global.agents == null || global.agents.contains(agentName);
        
        // Compile a list of relevant channel selectors
        
        List<ChannelSelector> selectors;
        if (global.channelSelectors == null) {
            selectors= null;
        } else {
            selectors = new ArrayList<>(global.channelSelectors.size());
            for (ChannelSelector selector : global.channelSelectors) {
                String agent = selector.getAgent();
                if (agentName.equals(agent) || (watchAgent && agent == null)) {
                    selectors.add(selector);
                }
            }
        }
                
        // Send connect, or return if the listener is not interested in this agent
        
        if (watchAgent || (selectors != null && !selectors.isEmpty())) {
            try {
                global.listener.connect(new AgentStatusEvent(aggregator, Collections.singletonList(agentInfo), null, null, null, null));
            } catch (RuntimeException x) {
                Console.getConsole().getLogger().warn("Error processing status aggregator event by "+ global.listener.getClass(), x);
            }
        } else {
            return;
        }
        
        // Compile set of channel paths the listener is interested in, and add channels to event
        
        Set<String> paths;
        if (selectors == null) {
            paths = null;
            event.addAddedChannels(channels.values());
        } else if (selectors.isEmpty()) {
            paths = Collections.emptySet();
        } else {
            paths = new HashSet<>();
            for (AgentChannel ch : channels.values()) {
                for (ChannelSelector s : selectors) {
                    if (s.match(ch)) {
                        paths.add(ch.getPath());
                        if (event != null) {
                            event.addAddedChannel(ch);
                        }
                        break;
                    }
                }
            }
        }
        if (noWatchedChannels && (paths == null || !paths.isEmpty())) {
            noWatchedChannels = false;
            if (state != null) {
                readState(state, null);
            }
            if (configuration != null) {
                try {
                    readConfiguration(configuration.getAllParameterInfo(), null);
                } catch (EmbeddedObjectDeserializationException x) {
                    Console.getConsole().getLogger().warn("Failed to read configuration for " + this.agentInfo.getName(), x);
                }
            }
        }
        
        // Create local listener handle
        
        ListenerHandle listenerHandle = new ListenerHandle(global, paths);
        listeners.add(listenerHandle);
    }
    
    void removeListener(AgentStatusAggregator.GlobalListenerHandle global) {
        
        // remove local listener handles
        
        Iterator<ListenerHandle> it = listeners.iterator();
        boolean changed = false;
        while (it.hasNext()) {
            ListenerHandle local = it.next();
            if (local.global == global) {
                it.remove();
                changed = true;
            }
        }
        
        // if any handles have been removed, update {@code noWatchedChannels}.
        
        if (changed && !noWatchedChannels) {
            noWatchedChannels = true;
            for (ListenerHandle lh : listeners) {
                if (lh.hasChannels()) {
                    noWatchedChannels = false;
                    break;
                }
            }
        }
        
        // remove current values from all channels
        
        if (noWatchedChannels) {
            for (AgentChannel c : channels.values()) {
                c.set(null);
                c.set(AgentChannel.Key.STATE, null);
            }
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    void readState(StateBundle bundle, MutableAgentStatusEvent changeEvent) {
        if (bundle == null) return;
        Map<String, String> out = bundle.getAllStatesAsStrings();
        for (String component : bundle.getComponentsWithStates()) {
            addAllStates(bundle.getComponentStateBundle(component), component + "/", out);
        }
        for (Map.Entry<String, String> e : out.entrySet()) {
            AgentChannel channel = channels.get(STATE_PREFIX + e.getKey());
            if (channel == null) {
//                System.out.println("Missing state: "+ STATE_PREFIX + e.getKey());
            } else {
                if (channel.set(e.getValue()) && changeEvent != null) {
                    changeEvent.addChange(channel, Key.VALUE);
                }
            }
        }
    }    
    
    private static void addAllStates(StateBundle bundle, String prefix, Map<String,String> out) {
        bundle.getInternalStates().forEach((key, value) -> {
            key = key.substring(key.lastIndexOf('.') + 1);
            out.put(prefix + key, value);
        });
        bundle.getDecodedStates().forEach((key, value) -> {
            if (value != null && !(value instanceof DataProviderState)) {
                key = key.substring(key.lastIndexOf('.') + 1);
                out.put(prefix + key, value.toString());
            }
        });
        bundle.getComponentsWithStates().forEach(component -> {
            addAllStates(bundle.getComponentStateBundle(component), prefix + component +"/", out);
        });
    }
    
    void readConfiguration(List<ConfigurationParameterInfo> config, MutableAgentStatusEvent changeEvent) {
        if (config == null) return;
        for (ConfigurationParameterInfo conf : config) {

            // Update channel where this is metadata (limits)
            AgentChannel channel = getChannelForPublishedData(conf.getComponentName(), AGENT_PREFIX, true);
            if (channel != null) {
                String attributeName = conf.getParameterName();
                channel.set(attributeName, conf);
                if (changeEvent != null) changeEvent.addChange(channel, attributeName);
            }

            // Updare channel where this is central value
            String p = conf.getPathName();
            channel = getChannelForPublishedData(p.startsWith("/") ? p.substring(1) : p, CONFIG_PREFIX, false);
            if (channel != null) {
                channel.set(conf);
                if (changeEvent != null) changeEvent.addChange(channel, AgentChannel.Key.VALUE);
            }
        }
    }
    
    void readTrendingData(StatusMessage mess, MutableAgentStatusEvent event) {
        try {
            Object o = mess.getEncodedData();
            if (o instanceof KeyValueDataList) {
                KeyValueDataList encodedData = (KeyValueDataList) o;
                for (KeyValueData d : encodedData) {
                    KeyValueData.KeyValueDataType type = d.getType();
                    String attrKey = null;
                    String key = d.getKey();
                    Object value = d.getValue();
                    switch (type) {
                        case KeyValueTrendingData:
                            attrKey = AgentChannel.Key.VALUE;
                            break;
                        case KeyValueMetaData:
                            int lastIndex = key.lastIndexOf('/');
                            attrKey = key.substring(lastIndex + 1);
                            key = key.substring(0, lastIndex);
                            if (AgentChannel.Key.STATE.equals(attrKey) && value != null) { // convert states to enum
                                try {
                                    value = DataProviderState.valueOf(value.toString());
                                } catch (IllegalArgumentException x) {
                                    attrKey = null; // ignore
                                }
                            }
                            break;
                    }
                    if (attrKey != null) {
                        AgentChannel channel = getChannelForPublishedData(key, AGENT_PREFIX, false);
                        if (channel != null) {
                            boolean changed = channel.set(attrKey, value);
                            if (changed && event != null) {
                                event.addChange(channel, attrKey);
                            }
                        }
                    }
                }
            }
        } catch (EmbeddedObjectDeserializationException x) {
            LOG.log(Level.FINE, x, () -> "Failed to read encoded data for " + agentInfo.getName());
        }
    }    
    
// -- Local classes : ----------------------------------------------------------

    /**
     * Local (agent specific) handle for a listener.
     * Created by {@code AgentHandle.createLocalListenerHangle(AgentStatusAggregator.GlobalListenerHandle)}. 
     */
    private class ListenerHandle {

        /** Global listener handle. */
        final AgentStatusAggregator.GlobalListenerHandle global;
        
        /**
         * Paths of watched channels;
         * Empty if this listener is only interested in agent connects and disconnects; {@code null} if this listener is interested in all channels.
         */
        final Set<String> paths;
        
        ListenerHandle(AgentStatusAggregator.GlobalListenerHandle global, Set<String> paths) {
            this.global = global;
            if (paths == null) {
                this.paths = null;
            } else {
                this.paths = paths.isEmpty() ? Collections.emptySet() : new HashSet(paths);
            }
        }        
        
        boolean hasChannels() {
            return paths == null || !paths.isEmpty();
        }
        
        AgentStatusEvent filter(AgentStatusEvent event) { // FIXME: filter list of agents
            if (paths == null) return event;
            List<AgentChannel> added = event.getAddedChannels();
            added = added.isEmpty() ? Collections.emptyList() : added.stream()
                                                                .filter(c -> paths.contains(c.getPath()))
                                                                .collect(Collectors.toList());
            List<AgentChannel> removed = event.getRemovedChannels();
            removed = removed.isEmpty() ? Collections.emptyList() : removed.stream()
                                                                .filter(c -> paths.contains(c.getPath()))
                                                                .collect(Collectors.toList());
            Map<AgentChannel, List<String>> changes = event.getStatusChanges();
            changes = changes.isEmpty() ? Collections.emptyMap() : changes.entrySet().stream()
                                          .filter(e -> paths.contains(e.getKey().getPath()))
                                          .collect(Collectors.toMap(e -> e.getKey(), e -> e.getValue()));
            return new AgentStatusEvent(event.getSource(), event.getAgents(), event.getMessage(), changes, added, removed);
        }
    
        
    }
    
}
