package org.lsst.ccs.gconsole.base.filter;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;

/**
 * Filter that selects channels that reflect CCS state and conditions, and are present in all subsystems.
 * This filter is used by "Whole Camera/CCS Health" monitor page.
 *
 * @author onoprien
 */
public class CCSHealthFilter extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    static public final String CREATOR_PATH = "Built-In/Whole Camera/CCS Health";
    
    static private final String GROUP_STATE = "Core States/";
    static private final String GROUP_VERSION = "Versions/";
    static private final String GROUP_RUNTIME = "Runtime Info/";
    
    static private final List<String> CHANNELS = Collections.unmodifiableList(Arrays.asList(
            "/state/PhaseState",
            "/state/OperationalState",
            "/state/ConfigurationState",
            "/state/AlertState",
            "/state/CommandState",
            "/ccsVersions/",
            "/runtimeInfo/"));
    
    static private final int cut = 5;
    static private final Pattern PAT_VERSION = Pattern.compile("ccsVersions/+([^/]+)/version");
    static private final Pattern PAT_RUNTIME = Pattern.compile("runtimeInfo/+([^/]+)");
    static private final Pattern PAT_REVERSE = Pattern.compile("([^/]+/)([^/]+)/([^/]+)");

// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = "AgentChannelsFilter",
            name = "CCS",
            path = CREATOR_PATH,
            description = "CCS Health Monitor. Selects channels that reflect CCS state and conditions, and are present in all subsystems.")
    public CCSHealthFilter() {
    }
    
// -- Implementing filter : ----------------------------------------------------
    
    @Override
    public String getDisplayPath(AgentChannel channel) {
        String path = channel.getLocalPath();
        Matcher m = PAT_VERSION.matcher(path);
        if (m.matches()) {
            return GROUP_VERSION + channel.getAgentName() +"/"+ m.group(1);
        } else {
            m = PAT_RUNTIME.matcher(path);
            if (m.matches()) {
                return GROUP_RUNTIME + channel.getAgentName() + "/" + m.group(1);
            } else {
                return GROUP_STATE + channel.getAgentName() + path.substring(cut, path.length() - cut);
            }
        }
    }

    @Override
    public String getOriginPath(String displayPath) {
        Matcher m = PAT_REVERSE.matcher(displayPath);
        if (m.matches()) {
            switch (m.group(1)) {
                case GROUP_STATE:
                    return m.group(2) +"/state/"+ m.group(3) +"State";
                case GROUP_VERSION:
                    return m.group(2) +"/ccsVersions/"+ m.group(3) +"/version";
                case GROUP_RUNTIME:
                    return m.group(2) +"/runtimeInfo/"+ m.group(3);
                default:
                    return null;
            }
        } else {
            return null;
        }
    }

    @Override
    public List<String> getOriginChannels() {
        return CHANNELS;
    }

    @Override
    public String getName() {
        return "CCS";
    }

    @Override
    public List<String> getFields(boolean compact) {
       return Collections.singletonList("VALUE");
    }

}

