package org.lsst.ccs.gconsole.plugins.trending.timeselection;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import javax.swing.table.AbstractTableModel;
import org.freehep.application.PropertyUtilities;
import org.freehep.application.studio.Studio;

/**
 * Maintains a list of time window presets known to the application.
 *
 * @author onoprien
 */
public class PresetList extends AbstractTableModel {

// -- Private parts : ----------------------------------------------------------
    
    private final String KEY = "org.lsst.trending.custom";
    private final Studio app;
    
    private ArrayList<TimeWindow> data;

// -- Construction and initialization : ----------------------------------------
    
    public PresetList(Studio application) {
        app = application;
        restore();
    }
    

// -- Getters and setters : ----------------------------------------------------
    
    /** Returns a list of <tt>maxSize</tt> most recently used time windows ordered by name. */
    public List<TimeWindow> getRecent(int maxSize) {
        List<TimeWindow> out = new ArrayList<>(data);
        Collections.sort(out, TimeWindow.compareByTime());
        if (maxSize < out.size()) out = out.subList(0, maxSize);
        Collections.sort(out, TimeWindow.compareByName());
        return out;
    }
    
    /** Returns preset with the specified index. */
    public TimeWindow get(int index) {
        return data.get(index);
    }
    
    /** Returns the index of the specified preset in this list, or -1 if this list does not contain the preset. */
    public int indexOf(TimeWindow timeWindow) {
        return data.indexOf(timeWindow);
    }
    
    /**
     * Returns time window with the specified name.
     * @param name Name of the time window.
     * @return Time window, or {@code null} if this preset list does not contain a time window with the specified name.
     */
    public TimeWindow get(String name) {
        for (TimeWindow tw : data) {
            if (name.equals(tw.getName())) return tw;
        }
        return null;
    }
    
    
// -- Modifying the list : -----------------------------------------------------
    
    /**
     * Adds time window to this list, removing any identically named elements.
     * If the specified element is already in this list, it is removed and re-added.
     */
    public void insert(TimeWindow timeWindow) {
        
        data.remove(timeWindow);
        
        String name = timeWindow.getName();
        int nElements = data.size();
        boolean needSave = timeWindow.isPersistent();
            
        int i = 0;
        for (; i < nElements; i++) {
            TimeWindow e = data.get(i);
            int out = name.compareTo(e.getName());
            if (out == 0) {
                needSave = needSave || e.isPersistent();
                data.set(i, timeWindow);
                break;
            } else if (out < 0) {
                data.add(i, timeWindow);
                break;
            }
        }
        if (i == nElements) {
            data.add(timeWindow);
        }

        if (needSave) save();
        fireTableStructureChanged();
    }
    
    /**
     * Removes the specified window from this list.
     * Updates application properties if necessary.
     */
    public void delete(TimeWindow timeWindow) {
        if (data.remove(timeWindow) && timeWindow.isPersistent()) {
            save();
        }
        fireTableStructureChanged();
    }
    

// -- Implementing TableModel : ------------------------------------------------

    @Override
    public int getRowCount() {
        return data.size();
    }

    @Override
    public int getColumnCount() {
        return 4;
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        switch (columnIndex) {
            case 0: return data.get(rowIndex).getName();
            case 1: return data.get(rowIndex).getLowerEdgeString();
            case 2: return data.get(rowIndex).getUpperEdgeString();
            case 3: return data.get(rowIndex).isPersistent();
            default: throw new IllegalArgumentException();
        }
    }

    @Override
    public Class<?> getColumnClass(int columnIndex) {
        return columnIndex == 3 ? Boolean.class : String.class;
    }

    @Override
    public String getColumnName(int column) {
        switch (column) {
            case 0: return "Name";
            case 1: return "Start";
            case 2: return "End";
            case 3: return "Saved";
            default: throw new IllegalArgumentException();
        }
    }
    
    
// -- Saving to / restoring from properties : ----------------------------------
    
    /**
     * Updates information on the specified time window in application properties.
     */
    private void save() {
        Properties prop = app.getUserProperties();
        if (prop != null) {
            int n = data.size();
            ArrayList<String> savedData = null;
            if (n != 0) {
                savedData = new ArrayList<>(n);
                for (int i=0; i<n; i++) {
                    savedData.add(data.get(i).toCompressedString());
                }
            }
            PropertyUtilities.setStringCollection(prop, KEY, savedData);
        }
    }
    
    /** Restore persistent time selections from application properties. */
    private void restore() {
        Properties prop = app.getUserProperties();
        if (prop == null) {
            data = new ArrayList<>(0);
        } else {
            String[] savedData = PropertyUtilities.getStringArray(prop, KEY, new String[0]);
            data = new ArrayList<>(savedData.length);
            for (String s : savedData) {
                try {
                    data.add(TimeWindow.parseCompressedString(s));
                } catch (IllegalArgumentException x) {
                }
            }
            Collections.sort(data, TimeWindow.compareByName());
        }
    }
    
}
