package org.lsst.ccs.gconsole.agent.command;

import java.time.Duration;
import java.util.HashMap;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.command.TokenizedCommand;
import org.lsst.ccs.messaging.CommandOriginator;
import org.lsst.ccs.messaging.CommandRejectedException;

/**
 * A {@code CommandTask} sends a command to a target subsystem and handles the responses. 
 * Methods are provided to check if the command processing is complete, to wait for its completion, and to retrieve the result.
 *
 * @author onoprien
 */
public class CommandTask {

// -- Fields : -----------------------------------------------------------------
    
    private final CommandSender sender;
        
    private final CommandHandle commandHandle;
    private long timeout;
    private final String destination;
    private final String command;
    private final Object[] args;
    
    private final ReentrantLock lock = new ReentrantLock();
    private final Condition condition = lock.newCondition();

    private ScheduledFuture<?> timeoutTask;
    private boolean done;
    private boolean canceled;
    private Object result;
    
    private HashMap<String,Object> props; // user-attached properties; guarded by this instance monitor

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Creates a {@code CommandTask} with an explicit list of raw (not converted to strings) arguments.
     * 
     * @param sender CommandSender that created this task.
     * @param handle Object to be notified of responses to the command.
     * @param timeout Timeout. Zero means the command will never time out. Negative
     *                timeout means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param destination Command destination, in "subsystem/component0/.../componentN" format.
     *                    If {@code null}, the {@code command} is assumed to contain destination in "destination/method" format.
     * @param command Command name.
     * @param args Command arguments.
     */
    protected CommandTask(CommandSender sender, CommandHandle handle, Duration timeout, String destination, String command, Object[] args) {
        this.sender = sender;
        this.commandHandle = handle;
        this.timeout = timeout.toMillis();
        this.destination = destination;
        this.command = command;
        if (args == null) throw new IllegalArgumentException();
        this.args = args;
    }
    
    /**
     * Creates a {@code CommandTask}.
     * 
     * @param destination Command destination in subsystem/path format. 
     * @param command Command in "method arguments" format.
     * @param sender CommandSender that created this task.
     * @param handle Object to be notified of responses to the command.
     * @param timeout Timeout. Zero means the command will never time out. Negative
     *                timeout means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     */
    protected CommandTask(String destination, String command, CommandSender sender, CommandHandle handle, Duration timeout) {
        this.sender = sender;
        this.commandHandle = handle;
        this.timeout = timeout.toMillis();
        this.destination = destination;
        this.command = command;
        this.args = null;
    }
    
    /**
     * Triggers sending the command to the target subsystem.
     */
    protected void send() {
        sender.workerExecutor.execute(() -> {
            lock.lock();
            try {
                if (done) return;
            } finally {
                lock.unlock();
            }
            try {
                CommandRequest request;
                if (args == null) {
                    TokenizedCommand tc = new TokenizedCommand(command);
                    request = new CommandRequest(destination, tc);
                } else {
                    request = new CommandRequest(destination, command, args);
                }
                sender.messenger.sendCommandRequest(request, new Originator());
                lock.lock();
                try {
                    if (!done && timeout != 0L) {
                        timeoutTask = sender.timer.schedule(CommandTask.this::timeOut, Math.abs(timeout), TimeUnit.MILLISECONDS);
                    }
                } finally {
                    lock.unlock();
                }
            } catch (Throwable x) {
                lock.lock();
                try {
                    done = true;
                    result = new IllegalStateException(x);
                    if (commandHandle != null) {
                        sender.callbackExecutor.execute(() -> {
                            commandHandle.onSendingFailure(x, CommandTask.this);
                        });
                    }
                    condition.signalAll();
                } finally {
                    lock.unlock();
                }
            }
        });
    }


// -- Future : -----------------------------------------------------------------
    
    /**
     * Waits if necessary for the command execution to complete, and then returns its result.
     * 
     * @return The command result.
     * @throws IllegalStateException If the command cannot be sent.
     * @throws InterruptedException If the current thread was interrupted while waiting.
     * @throws CommandRejectedException If the command was rejected by the target subsystem.
     * @throws CancellationException If the command was canceled.
     * @throws TimeoutException If the command timed out.
     * @throws ExecutionException If an exception was thrown by the target subsystem while executing the command,
     *                            or the result of the execution is an instance of {@code Throwable}.
     */
    public Object get() throws IllegalStateException, InterruptedException, CommandRejectedException, CancellationException, TimeoutException, ExecutionException {
        getResult();
        if (result instanceof Throwable) {
            if (result instanceof IllegalStateException) {
                throw (IllegalStateException)result;
            } else if (result instanceof InterruptedException) {
                throw (InterruptedException)result;
            } else if (result instanceof CommandRejectedException) {
                throw (CommandRejectedException)result;
            } else if (result instanceof CancellationException) {
                throw (CancellationException)result;
            } else if (result instanceof TimeoutException) {
                throw (TimeoutException)result;
            } else if (result instanceof ExecutionException) {
                throw (ExecutionException)result;
            } else {
                throw new ExecutionException((Throwable)result);
            }
        }
        return result;
    }
    
    /**
     * Waits if necessary for the command execution to complete, and then returns its result or exception.
     * Unlike the {@code get()} method, this method does not throw command-related exceptions. Instead, 
     * the outcome of this command task is reported through the return value.
     * 
     * @return The command result, or the exception that would be thrown by the {@link get()} method.
     * @throws InterruptedException If the current thread was interrupted while waiting.
     */
    public Object getResult() throws InterruptedException {
        lock.lock();
        try {
            while (!done) {
                condition.await();
            }
        } finally {
            lock.unlock();
        }
        return result;
    }
    
    /**
     * Cancels this command task.
     * Once the command is canceled, {@link CommandHandle} methods will not be called, and the {@code get()} method will throw {@code CancellationException}.
     * The target subsystem will not be notified of cancellation.
     * 
     * @return {@code False} if the command task could not be canceled, typically because it has already completed normally; {@code true} otherwise.
     */
    public boolean cancel() {
        lock.lock();
        try {
            if (done) return false;
            canceled = true;
            done = true;
            if (timeoutTask != null) timeoutTask.cancel(false);
            result = new CancellationException();
            if (commandHandle != null) {
                sender.callbackExecutor.execute(() -> commandHandle.onCancel((CancellationException) result, CommandTask.this));
            }
            condition.signalAll();
        } finally {
            lock.unlock();
        }
        return true;
    }
    
    /**
     * Returns {@code true} if this command task was canceled before it completed normally.
     * 
     * @return True if the command was canceled.
     */
    public boolean isCancelled() {
        lock.lock();
        try {
            return canceled;
        } finally {
            lock.unlock();
        }
    }
    
    /**
     * Returns true if this command task has completed.
     * Completion may be due to normal completion, an exception, or cancellation.
     * 
     * @return {@code True} if this task completed.
     */
    public boolean isDone() {
        lock.lock();
        try {
            return done;
        } finally {
            lock.unlock();
        }
    }
    
    
// -- Other getters and properties : -------------------------------------------
    
    /**
     * Returns the command as a string in subsystem/path/method format.
     * @return Command.
     */
    public String getCommand() {
        return command;
    }
    
    /**
     * Returns the command arguments.
     * @return Command arguments.
     */
    public Object[] getArguments() {
        return args;
    }
    
    /**
     * Attaches a user-supplied property to this task.
     * 
     * @param key Property name.
     * @param value Property value.
     * @return Previous value associated with {@code key}, or {@code null} if there was no mapping for {@code key}.
     */
    synchronized public Object putProperty(String key, Object value) {
        if (props == null) {
            props = new HashMap<>(2);
        }
        return props.put(key, value);
    }
    
    /**
     * Removes a previously attached user-supplied property.
     * 
     * @param key Property name.
     * @return Previous value associated with {@code key}, or {@code null} if there was no mapping for {@code key}.
     */
    synchronized public Object removeProperty(String key) {
        return props.remove(key);
    }
    
    /**
     * Returns the value of the specified user-supplied property.
     * 
     * @param key Property name.
     * @return Property value, or {@code null} if a property with the specified name does not exist.
     */
    synchronized public Object getProperty(String key) {
        return props.get(key);
    }
    
// -- Local methods : ----------------------------------------------------------
    
    private void timeOut() {
        lock.lock();
        try {
            if (done) return;
            done = true;
            timeoutTask = null;
            result = new TimeoutException("Timed out after "+ Math.abs(timeout)/1000L +" seconds.");
            if (commandHandle != null) {
                sender.callbackExecutor.execute(() -> commandHandle.onTimeout((TimeoutException) result, CommandTask.this));
            }
            condition.signalAll();
        } finally {
            lock.unlock();
        }
    }
    
    private boolean areAllArgumentsStrings() {
        for (Object arg : args) {
            if (!(arg instanceof String)) return false;
        }
        return true;
    }
    
// -- Auxiliary classes : ------------------------------------------------------
    
    private class Originator implements CommandOriginator {

        @Override
        public void processNack(CommandNack nack) {
            lock.lock();
            try {
                if (done) return;
                done = true;
                if (timeoutTask != null) timeoutTask.cancel(false);
                result = new CommandRejectedException(nack);
                if (commandHandle != null) {
                    sender.callbackExecutor.execute(() -> commandHandle.onNack(nack, CommandTask.this));
                }
                condition.signalAll();
            } finally {
                lock.unlock();
            }
        }

        @Override
        public void processAck(CommandAck ack) {
            lock.lock();
            try {
                if (done) return;
                if (ack != null && timeout > 0L) {
                    Duration customTimeout = ack.getTimeout();
                    if (customTimeout != null && !(customTimeout.isZero() || customTimeout.isNegative())) {
                        if (timeoutTask != null) timeoutTask.cancel(false);
                        timeout = customTimeout.toMillis();
                        timeout = timeout + timeout/10L + 1000L; // treat custom timeout as an estimate, use longer actual timeout
                        timeoutTask = sender.timer.schedule(CommandTask.this::timeOut, timeout, TimeUnit.MILLISECONDS);
                    }
                }
                if (commandHandle != null) {
                    sender.callbackExecutor.execute(() -> commandHandle.onAck(ack, CommandTask.this));
                }
            } finally {
                lock.unlock();
            }
        }

        @Override
        public void processResult(CommandResult result) {
            lock.lock();
            try {
                if (done) return;
                done = true;
                if (timeoutTask != null) timeoutTask.cancel(false);
                try {
                    CommandTask.this.result = result.getResult();
                } catch (Throwable t) {
                    CommandTask.this.result = t;
                }
                if (commandHandle != null) {
                    if (CommandTask.this.result instanceof Throwable) {
                        sender.callbackExecutor.execute(() -> commandHandle.onExecutionFailure((Throwable)CommandTask.this.result, CommandTask.this));
                    } else {
                        sender.callbackExecutor.execute(() -> commandHandle.onSuccess(CommandTask.this.result, CommandTask.this));
                    }
                }
                condition.signalAll();
            } finally {
                lock.unlock();
            }
        }
        
    }
    

}
