package org.lsst.ccs.gconsole.services.optpage;

import java.util.HashMap;
import javax.swing.JComponent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.services.optpage.LsstOptionalPageService;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;

/**
 * Descriptor of a page that can be automatically opened by the graphical console.
 * <p>
 * A descriptor of this type specifies a set of graphical console pages that can be opened 
 * on console initialization or whenever compatible subsystems connect to the buses.
 * <p>
 * Each page in the set is identified by a slash-separated path returned by {@link #getPage(AgentInfo)} method.
 * The method is first called with {@code null} argument when this descriptor is registered with the console by a call
 * to {@link Console#addOptionalPage(OptionalPage)}. If a non-null path is returned at this point,
 * a page will be created regardless of what other subsystems are present on the buses. After that,
 * the method is called with an appropriate {@link AgentInfo AgentInfo} instance for each of the 
 * subsystems connected to the buses, and whenever a new subsystem joins the buses later.
 * <p>
 * Every time the {@link #getPage(AgentInfo)} method returns a new non-null path, a new graphical console
 * page is enabled by creating a menu with that path. The menu can be used by the user to open or close
 * the page at any time. The return value of the {@link #isAutoOpen()} method determines whether the pages
 * created by this descriptor are opened by default, before the user interacts with the menu.
 * Note that each page created by this descriptor can interact with zero or more subsystems,
 * and each subsystem is expected to contribute to no more than one such page.
 * <p>
 * When a subsystem contributing to one of the pages disconnects from the buses, this descriptor
 * is notified by a call to its {@link #agentDisconnected(AgentInfo)} method. If the method
 * returns {@code true}, the page is closed and the menu is removed. Otherwise, the page will be
 * removed only when all the conditions below are satisfied:
 * <ul><li>There are no more subsystems contributing to it.
 *     <li>The page was created in response to a subsystem connection (in other words, its path was not returned by a call to {@code getPage(null)}.
 *     <li>The page is closed.</ul>
 * <p>
 * The title of the page can be explicitly specified by setting the name of the component returned by
 * the {@link #open(AgentInfo, JComponent)} method. If the name is not specified, the last
 * segment of the page path is used as a title.
 * <p>
 * All methods declared by this interface are called by the framework on the AWT Event Dispatching Thread.<br>
 * See also {@link Console#addOptionalPage(OptionalPage)}.
 *
 * @author onoprien
 */
public interface OptionalPage {
    
    /**
     * Returns the default auto-open value. 
     * If {@code true}, the page(s) will be open automatically on compatible subsystem discovery.
     * <p>
     * The default implementation returns {@code true}.
     * 
     * @return Default {@code auto-open} value.
     */
    default boolean isAutoOpen() {
        return false;
    }
    
    /**
     * Tells the framework whether the state of pages in this set (open or closed) should be 
     * automatically saved or restored when the graphical console session is saved or restored.
     * <p>
     * The default implementation returns {@code false}. It should be used if a plugin
     * prefers saving the state of its optional pages as a part of its own state. This is usually 
     * preferable if those states contain additional information, or if the descriptor might not
     * be active throughout plugin's life time. If this method is overridden to return {@code true},
     * the pages states will be automatically saved and restored (to the extend possible) by the framework.
     * 
     * @return True if the state of pages in this set should be automatically saved or restored as a part of the console session.
     */
    default boolean isAutoSave() {
        return true;
    }
    
    /**
     * Called with {@code null} argument when this descriptor is registered, then whenever a new subsystem is detected on the buses. 
     * This method should return the slash-separated page menu path if the subsystem
     * identified by the supplied parameter is compatible with this optional page descriptor. 
     * Otherwise, it should return {@code null}.
     * <p>
     * If {@code auto-open} is {@code true}, the {@link #open open(...)} function will be called immediately
     * following this call. Otherwise, a menu item will be added to the graphical console to allow opening the
     * page at a later time.
     * 
     * @param agent Agent that has been discovered on the buses, or {@code null} if this is the initial call on descriptor registration.
     * @return Page menu path.
     */
    String getPage(AgentInfo agent);
   
    /**
     * Called when a page is being opened.
     * The function should return the graphical component to be displayed. 
     * If {@code null} is returned, the page is not displayed.
     * If a page with the path returned by {@link #getPage(AgentInfo)} already exists
     * (this can happen if a single page communicates with multiple subsystems,
     * or if a subsystem restarted without closing the page),
     * it is passed to this method as {@code existingComponent}.
     * 
     * @param agent Agent descriptor.
     * @param existingComponent Existing component with the specified path.
     * @return Graphical component to be open.
     */
    JComponent open(AgentInfo agent, JComponent existingComponent);
    
    /**
     * Called whenever an agent potentially contributing to this page disconnects from the buses.
     * The agent is potentially contributing if {@code getPage(AgentInfo)} returns {@code true}.
     * This method may return {@code true} to request closing the page if currently open.
     * <p>
     * The default implementation returns {@code false}.
     * 
     * @param agent Agent.
     * @return {@code true} to close the page.
     */
    default boolean agentDisconnected(AgentInfo agent) {
        return false;
    }
    
    /**
     * Called after the page has been opened and made visible by the graphical console.
     * <p>
     * The default implementation does nothing.
     * 
     * @param page Name of the opened page.
     */
    default void opened(String page) {
    }
    
    /**
     * Called when the page is closed by the user.
     * <p>
     * The default implementation does nothing.
     * 
     * @param page Name of the closed page.
     */
    default void closed(String page) {
    }
    
    
// -- Convenience methods for accessing optional page service : ----------------
    
    static public LsstOptionalPageService getService() {
        return Console.getConsole().getOptionalPageService();
    }
    
    default public void setOpen(AgentInfo agent, boolean isOpen) {
        LsstOptionalPageService service = getService();
        if (service != null) {
            service.setOpen(this, agent, isOpen);
        }
    }
    
    /**
     * Returns JavaBean describing the current state of pages handled by this descriptor.
     * @return Storage bean describing current state.
     */
    default public HashMap<String,DataPanelDescriptor> save() {
        LsstOptionalPageService service = getService();
        if (service != null) {
            return service.save(this);
        } else {
            return null;
        }
    }
    
    /**
     * Restores, to the extent possible, pages handled by this descriptor to the specified state.
     * @param storageBean Storage bean describing state.
     */
    default public void restore(HashMap<String,DataPanelDescriptor> storageBean) {
        LsstOptionalPageService service = getService();
        if (service != null) {
            service.restore(this, storageBean);
        }
    }
    
}
