package org.lsst.ccs.gconsole.base.filter;

import java.util.*;
import java.util.regex.Pattern;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;

/**
 * 
 *
 * @author onoprien
 */
public class GenericFilter extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final String name;
    private final List<String> agents;
    private final List<String> channels;
    private final String delimeter;
    private final boolean agentTabs;
    private final List<String> fields;
    
    private HashMap<String,String> display2origin;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs a filter.
     * 
     * @param name Name of the filter.
     * @param agents Comma-separated list of subsystems accepted by this filter.
     * @param channels Comma-separated list of channels accepted by this filter (may contain templates and selectors).
     * @param delimeter Character in the original channel path that should be replaced by a slash (and therefore used as a path delimeter).
     * @param separatePagesForAgents If {@code true}, the subsystem name is used as a page name.
     * @param fields Comma-separated list of fields to display.
     */
    @Create(category = "AgentChannelsFilter",
            name = "Configurable generic filter",
            path = "Built-In/Generic",
            description = "Channels filter that leaves paths unchanged (except for possibly replacing delimeters with slashes and inserting doupbe slashes to separate pages), but allows selecting desired subsystems, channels, and monitor fields.")
    public GenericFilter(
            @Par(def = Par.NULL, desc = "Filter name. Optional.") String name,
            @Par(def = Par.NULL, desc = "List of subsystems accepted by this filter. If not specified, all subsystems are accepted.") List<String> agents,
            @Par(def = Par.NULL, desc = "List of channels accepted by this filter (may contain templates and selectors). If not specified, all subsystems are accepted.") List<String> channels,
            @Par(def = Par.NULL, desc = "Character in the original channel path that should be replaced by a slash (and therefore used as a path delimeter).") String delimeter,
            @Par(def = "false", desc = "If true, the first slash in the path is replaced by a double slash. This causes some monitoring views display each subsystem data in a separate tab.") boolean separatePagesForAgents,
            @Par(def = Par.NULL, desc = "List of fields to display in monitor views. If not specified, the default set of fields is displayed. The default depends on a specific view. Some views may ignore the list of fields suggested by the filter. Available standard fields: VALUE, UNITS, LOW_ALARM, LOW_WARN, ALERT_LOW, HIGH_ALARM, HIGH_WARN, ALERT_HIGH, DESCR.") List<String> fields)
    {
        
        this.name = name;
        this.agents = agents == null ? null : new ArrayList<>(agents);
        this.channels = channels == null ? null : new ArrayList<>(channels);
        
        if (delimeter == null || delimeter.isEmpty()) {
            this.delimeter = null;
        } else {
            String s = delimeter.trim();
            if (s.length() > 2 && s.startsWith("\"") && s.endsWith("\"")) {
                this.delimeter = s.substring(1, s.length()-1);
            } else {
                this.delimeter = Pattern.quote(delimeter);
            }
            display2origin = new HashMap<>();
        }
        
        this.agentTabs = separatePagesForAgents;
        this.fields = fields == null ? null : new ArrayList<>(fields);
    }
    
    /**
     * Constructs a filter, using comma-separated strings instead of lists.
     * Legacy constructor for compatibility with early GenericFilter version.
     * 
     * @param name Name of the filter.
     * @param agents Comma-separated list of subsystems accepted by this filter.
     * @param channels Comma-separated list of channels accepted by this filter (may contain templates and selectors).
     * @param delimeter Character in the original channel path that should be replaced by a slash (and therefore used as a path delimeter).
     * @param separatePagesForAgents If {@code true}, the subsystem name is used as a page name.
     * @param fields Comma-separated list of fields to display.
     */
    public GenericFilter(String name, String agents, String channels, String delimeter, boolean separatePagesForAgents, String fields) {
        this(name, toList(agents), toList(channels), delimeter, separatePagesForAgents, toFieldsList(fields));
    }
    
    static private List<String> toList(String s) {
        if (s.trim().isEmpty()) {
            return null;
        } else if (s.trim().equals("\"\"")) {
            return Collections.emptyList();
        } else {
            String[] ss = s.split(",");
            List<String> out = new ArrayList<>(ss.length);
            for (String x : ss) {
                out.add(x.trim());
            }
            return out;
        }
    }
    
    static private List<String> toFieldsList(String fields) {
        if (fields == null || fields.trim().isEmpty()) {
            return null;
        } else if (fields.equals("\"\"")) {
            return Collections.singletonList("VALUE");
        } else {
            String[] ss = fields.split(",");
             List<String> out = new ArrayList<>(ss.length);
            for (String s : ss) {
                out.add(s.trim());
            }
            return out;
        }
    }

    
// -- Implementing filter : ----------------------------------------------------

    @Override
    public String getDisplayPath(String originPath) {
        String displayPath;
        if (delimeter == null) {
            if (agentTabs) {
                displayPath = originPath.replaceFirst("/", "//");
            } else {
                displayPath = originPath;
            }
        } else {
            int i = originPath.indexOf("/");
            String agent = originPath.substring(0, i);
            String localPath = originPath.substring(i+1);
            localPath = localPath.replaceAll(delimeter, "/");
            if (agentTabs) {
                displayPath = agent +"//"+ localPath;
            } else {
                displayPath = agent +"/"+ localPath;
            }
            display2origin.putIfAbsent(displayPath, originPath);
        }
        return displayPath;
    }

    @Override
    public String getOriginPath(String displayPath) {
        if (delimeter == null) {
            if (agentTabs) {
                return displayPath.replaceFirst("//", "/");
            } else {
                return displayPath;
            }
        } else {
            return display2origin.get(displayPath);
        }
    }

    @Override
    public List<String> getOriginChannels() {
        return channels;
    }

    @Override
    public List<String> getAgents() {
        return agents;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public List<String> getFields(boolean compact) {
        return compact ? null : fields;
    }

    
// -- Specific filters : -------------------------------------------------------
    
    @Create(category = "AgentChannelsFilter",
            name = "Monitor channels",
            path = "Built-In/Monitor",
            description = "Channels filter that leaves paths unchanged (except for possibly replacing delimeters with slashes and inserting doupbe slashes to separate pages), but accepts only monitored channels.")
    static public AgentChannelsFilter filterMonitorOnly(
            @Par(def = "Monitored channels", desc = "Filter name.") String name,
            @Par(def = Par.NULL, desc = "List of subsystems accepted by this filter. If not specified, all subsystems are accepted.") List<String> agents,
            @Par(def = Par.NULL, desc = "Character in the original channel path that should be replaced by a slash (and therefore used as a path delimeter).") String delimeter,
            @Par(def = "false", desc = "If true, the first slash in the path is replaced by a double slash. This causes some monitoring views display each subsystem data in a separate tab.") boolean separatePagesForAgents,
            @Par(def = Par.NULL, desc = "List of fields to display in monitor views. If not specified, the default set of fields is displayed. The default depends on a specific view. Some views may ignore the list of fields suggested by the filter.") List<String> fields)
    {
        return new GenericFilter(name, agents, Collections.singletonList(AgentChannel.Key.FORMAT), delimeter, separatePagesForAgents, fields);
    }
    
    @Create(category = "AgentChannelsFilter",
            name = "Subsystem Selector",
            path = "Built-In/Subsystem Selector",
            description = "Selects channels from the specified subsystem only. Leaves paths unchanged.")
    static public AgentChannelsFilter agentSelector(
            @Par(desc = "Subsystem name.") String subsystem)
    {
        return new GenericFilter(subsystem +" m-r", Collections.singletonList(subsystem), null, null, false, null);
    }
    
    
    
    
    
    
}
