package org.lsst.ccs.gconsole.base.filter;

import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import java.util.*;
import org.lsst.ccs.gconsole.base.filter.AbstractChannelsFilter;
import org.lsst.ccs.gconsole.services.persist.Savable;

/**
 * A filter that selects a set of data channels and defines their presentation.
 * Original channels are identified by their unique string trending paths (slash-separated,
 * the first segment is the subsystem name).  This filter maps some of the original
 * channels to one or more display paths in <br>
 * {@code [page//]path1/.../pathN/name} format.
 * <p>
 * The exact interpretation of filter output depends on the tool using the filter. 
 * See {@link AgentStatusAggregator#addListener(AgentStatusListener, AgentChannelsFilter)}
 * for more details on using filters to configure agent status aggregator and tools 
 * that rely on it.
 *
 * @author onoprien
 */
public interface AgentChannelsFilter extends Savable {
    
    /**
     * Category for general purpose {@link Persistable Persistable} filters.
     */
    static public String CATEGORY = "AgentChannelsFilter";
    
    /**
     * Trivial filter that accepts all channels and does not modify paths.
     * This filter uses default implementations for all methods.
     */
    static PersistableAgentChannelsFilter ALL = new AbstractChannelsFilter() {
        @Override
        public String getName() {return "Unfiltered";}
    };
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns the name of this filter.
     * <p>
     * The default implementation returns {@code null}.
     * 
     * @return Name of this filter, or {@code null} if this filter has no name.
     */
    default String getName() {
        return null;
    }
    
    /**
     * Returns a list of agent names accepted by this filter.
     * <p>
     * The default implementation returns {@code null}.
     * 
     * @return Agent names accepted by this filter, or {@code null} if any agent might be accepted.
     */
    default List<String> getAgents() {
        return null;
    }
    
    /**
     * Returns a list of original channel paths accepted by this filter.
     * <p>
     * Note that if this filter is used to configure {@code AgentStatusAggregator} or any clients 
     * that rely on it (like monitoring data views), the list returned by this method may contain
     * templates and selectors. See 
     * {@link AgentStatusAggregator#addListener(AgentStatusListener, Collection, Collection) AgentStatusAggregator.addListener(listener, agents, channels)}
     * for details.
     * <p>
     * The default implementation returns {@code null}.
     * 
     * @return Original channel paths accepted by this filter, or {@code null} if any
     * original path for which {@link #getDisplayPaths} returns a non-empty list is accepted.
     */
    default List<String> getOriginChannels() {
        return null;
    }
    
    /**
     * Returns a list of display paths produced by this filter.
     * Typically, if this method returns a non-null list, components downstream from
     * this filter will display this fixed set of channels, whether or not the data
     * is available, regardless of what paths are returned by {@link #getDisplayPaths}
     * applied to accepted original channels. Otherwise, any display channels for
     * which the data is available will be displayed.
     * <p>
     * The default implementation returns {@code null}.
     * 
     * @return Display paths produced by this filter, or {@code null} if the list is not fixed.
     */
    default List<String> getDisplayChannels() {
        return null;
    }
    
    /**
     * Returns the original path corresponding to the specified display channel.
     * If multiple original paths are mapped to the specified display path, this
     * method may return one of them, or {@code null}.
     * <p>
     * The default implementation returns the display path provided as an argument.
     * 
     * @param displayPath Display channel path.
     * @return Original path, or {@code null} if the specified path does not correspond to any original channel.
     */
    default String getOriginPath(String displayPath) {
        return displayPath;
    }
    
    /**
     * Returns the list of original paths mapped to the specified display channel.
     * <p>
     * The default implementation returns a list that contains one element - the
     * path returned by {@link #getOriginPath getOriginPath(...)} -  or an
     * empty list if {@link #getOriginPath getOriginPath(...)} returns {@code null}.
     * 
     * @param displayPath Display channel path.
     * @return Original paths, or an empty list if the specified path does not correspond to any original channel.
     */
    default List<String> getOriginPaths(String displayPath) {
        String path = getOriginPath(displayPath);
        return path == null ? Collections.emptyList() : Collections.singletonList(path);
    }
    
    /**
     * Returns the display channel corresponding to the specified original channel.
     * If the original channel is mapped to multiple display paths, this method may
     * return one of them, or {@code null}.
     * <p>
     * Not that if the specified original channel is not accepted by this filter,
     * this method should return an empty list. {@code null} should never be returned.
     * <p>
     * The default implementation forwards the call to {@link #getDisplayPath(String)}
     * method. Subclasses may override it to take into account channel attributes and metadata.
     * 
     * @param channel Original data channel.
     * @return List of display channels.
     */
    default String getDisplayPath(AgentChannel channel) {
        return getDisplayPath(channel.getPath());
    }
    
    /**
     * Returns the display channel corresponding to the specified original path.
     * If the original channel is mapped to multiple display paths, this method may
     * return one of them, or {@code null}.
     * <p>
     * If the specified original channel is not accepted by this filter,
     * this method should return an empty list. {@code null} should never be returned.
     * <p>
     * The default implementation returns an unmodified original path.
     * 
     * @param originPath Original path.
     * @return List of display channels.
     */
    default String getDisplayPath(String originPath) {
        return originPath;
    }
    
    /**
     * Returns a list of display channels for the specified original channel.
     * <p>
     * Not that if the specified original channel is not accepted by this filter,
     * this method should return an empty list. {@code null} should never be returned.
     * <p>
     * The default implementation returns a singleton list that contains a string
     * returned by {@link #getDisplayPath(AgentChannel)} method.
     * 
     * @param channel Original data channel.
     * @return List of display channels.
     */
    default List<String> getDisplayPaths(AgentChannel channel) {
        String displayPath = getDisplayPath(channel);
        return displayPath == null ? Collections.emptyList() : Collections.singletonList(displayPath);
    }
    
    /**
     * Returns a list of display channels for the specified original path.
     * <p>
     * If the specified original channel is not accepted by this filter,
     * this method should return an empty list. {@code null} should never be returned.
     * <p>
     * The default implementation returns a singleton list that contains a string
     * returned by {@link #getDisplayPath(String)} method.
     * 
     * @param originPath Original path.
     * @return List of display channels.
     */
    default List<String> getDisplayPaths(String originPath) {
        String displayPath = getDisplayPath(originPath);
        return displayPath == null ? Collections.emptyList() : Collections.singletonList(displayPath);
    }
    
    /**
     * Returns additional attributes associated with the specified path.
     *
     * @param path Full or partial display path.
     * @return Map of attribute names to their values.
     */
    default Map<String,Object> getAttributes(String path) {
        return Collections.emptyMap();
    }
    
    /**
     * Returns a list of attributes to display for each channel.
     * 
     * @param compact If {@code true}, returns a list of columns in a compact view.
     * @return Display fields, or {@code null} if this filter does not provides any hints on what attributes should be displayed.
     */
    default List<String> getFields(boolean compact) {
        return null;
    }

    
// -- Static utility methods : -------------------------------------------------
    
    /**
     * Returns a list of display paths for the specified original path, applying the provided filter.
     * This method takes into account the output of
     * {@link #getDisplayPaths(String) getDisplayPaths(originPath)},
     * {@link #getDisplayChannels()}, and {@link #getOriginChannels()}.
     * Templates in the list of origin paths returned by the filter are honored by selectors are ignored. 
     * 
     * @param filter Filter to apply.
     * @param originPath Origin path.
     * @return List of display path.
     */
    static List<String> getDisplayPath(AgentChannelsFilter filter, String originPath) {
        
        List<String> paths = filter.getDisplayPaths(originPath);
        if (paths.isEmpty()) return paths;
        
        List<String> dps = filter.getDisplayChannels();
        if (dps != null) {
            paths.retainAll(dps);
        }
        
        List<String> agents = filter.getAgents();
        List<String> channels = filter.getOriginChannels();
        int i = originPath.indexOf("/");
        if (i == -1) return Collections.emptyList();
        String agent = originPath.substring(0, i);
        
        if (channels == null) {
            if (agents == null || agents.contains(agent)) {
                return paths;
            }
        } else {
            for (String channel : channels) {
                if (!isSelector(channel)) {
                    if ((agents == null || !channel.startsWith("/") || agents.contains(agent)) && matchTemplate(originPath, channel)) {
                        return paths;
                    }
                }
            }
        }
        return Collections.emptyList();
    }
    
    /**
     * Checks whether the path matches the template.
     * 
     * @param path Origin path.
     * @param template Path template.
     * @return True if the path matches.
     */
    static boolean matchTemplate(String path, String template) {
        if (template.startsWith("/")) {
            path = path.substring(path.indexOf("/"));
        }
        return template.endsWith("/") ? path.startsWith(template) : path.equals(template);
    }
    
    /**
     * Checks whether the specified string is in the channel selector format (as opposed to explicit path or template).
     *
     * @param code String to test.
     * @return {@code true} if the specified string is a channel selector.
     */
    static boolean isSelector(String code) {
        if (code.contains("=")) return true;
        return ! code.contains("/");
    }
    
}
