package org.lsst.ccs.gconsole.services.aggregator;

import java.util.*;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Collectors;
import org.lsst.ccs.gconsole.util.MatchPredicate;

/**
 * Checks whether an {@code AgentChannel} satisfies certain criteria.
 *
 * @author onoprien
 */
public interface ChannelSelector {
    
    static final Pattern pRegEx = Pattern.compile("[/+.*?(){}|\\\\]");

    static ChannelSelector compile(String rawSelector) {
        if (pRegEx.matcher(rawSelector).find()) {
            return PathPattern.compile(rawSelector);
        } else {
            return AttributeSelector.compile(rawSelector);
        }
    }
    
    static List<ChannelSelector> compile(Collection<String> rawSelectors) {
        return rawSelectors == null ? null : rawSelectors.stream()
                                                         .map(cs -> ChannelSelector.compile(cs))
                                                         .filter(s -> s != null)
                                                         .collect(Collectors.toList());
    }
    
    static boolean accept(List<String> agents, List<ChannelSelector> channelSelectors, AgentChannel channel) {
        String agentName = channel.getAgentName();
        boolean watchAgent = agents == null || agents.contains(agentName);
        if (channelSelectors == null) {
            return watchAgent;
        } else {
            for (ChannelSelector s : channelSelectors) {
                if (watchAgent || agentName.equals(s.getAgent())) {
                    return s.match(channel);
                }
            }
            return false;
        }
    }
    
    static <T extends AgentChannel> List<T> filter(List<String> agents, List<ChannelSelector> channelSelectors, Collection<T> channels) {
        return channels.stream().filter(ch -> accept(agents, channelSelectors, ch)).collect(Collectors.toList());
    }
    
    /**
     * Appends channels matching selectors to the provided list.
     * The input collection is assumed to contain channels from one agent.
     * 
     * @param <T> Type of input and output lists, subclass of {@code AgentChannel}.
     * @param agents 
     * @param channelSelectors
     * @param channels
     * @param out
     * @return 
     */
    static <T extends AgentChannel> List<T> filter(List<String> agents, List<ChannelSelector> channelSelectors, Collection<T> channels, List<T> out) {
        if (out == null) out = new ArrayList<>();
        if (!channels.isEmpty()) {
            String agentName = channels.iterator().next().getAgentName();
            boolean watchedAgent = agents == null || agents.contains(agentName);
            if (channelSelectors == null) {
                if (watchedAgent) {
                    out.addAll(channels);
                }
            } else {
                for (T ch : channels) {
                    for (ChannelSelector s : channelSelectors) {
                        if (watchedAgent || agentName.equals(s.getAgent())) {
                            if (s.match(ch)) {
                                out.add(ch);
                                break;
                            }
                        }
                    }
                }
            }
        }
        return out;
    }    
    
    String getAgent();
    
    boolean match(AgentChannel channel);
    
    static ChannelSelector ALL = new ChannelSelector() {
        @Override
        public String getAgent() {
            return null;
        }
        @Override
        public boolean match(AgentChannel channel) {
            return true;
        }
    };
    
}

final class AttributeSelector implements ChannelSelector {

    private final String agentName;
    private final String[] data;
    
    private AttributeSelector(String agent, List<String> att) {
        agentName = agent;
        data = att.toArray(new String[att.size()]);
    }

    static AttributeSelector compile(String rawTemplate) {
        String[] ss = rawTemplate.split("&");
        ArrayList<String> out = new ArrayList<>(ss.length * 2);
        String agentName = null;
        for (String s : ss) {
            if (!s.isEmpty()) {
                String[] tt = s.split("=");
                String key = tt[0];
                String value = tt.length == 1 ? null : tt[1];
                if ("agent.name".equals(key)) {
                    agentName = value;
                }
                out.add(key);
                out.add(value);
            }
        }
        return out.isEmpty() ? null : new AttributeSelector(agentName, out);
    }

    public String get(String key) {
        for (int i=0; i<data.length; i += 2) {
            if (data[i].equals(key)) {
                return data[i+1];
            }
        }
        return null;
    }

    @Override
    public String getAgent() {
        return agentName;
    }

    @Override
    public boolean match(AgentChannel channel) {
        for (int i=0; i<data.length; i += 2) {
            String key = data[i];
            Object v = key.startsWith("agent.") ? channel.getAgent().getAgentProperty(key.substring("agent.".length())) : channel.get(key);
            if (v == null) return false;
            String requiredValue = data[i+1];
            if (!(requiredValue == null || requiredValue.equals(v.toString()))) {
                return false;
            }
        }
        return true;
    }
}


final class PathPattern implements ChannelSelector {
    
    static final Pattern pAgent = Pattern.compile("([\\w\\-]+)/.*");

    private String agent;
    private Predicate<String> pattern;

    static PathPattern compile(String rawTemplate) {
        PathPattern out = new PathPattern();
        if (rawTemplate.startsWith("/")) {
            rawTemplate = "[^/]+" + rawTemplate;
        } else {
            Matcher m = pAgent.matcher(rawTemplate);
            if (m.matches()) {
                out.agent = m.group(1);
            }
        }
        if (rawTemplate.endsWith("/")) {
            rawTemplate = rawTemplate + ".+";
        }
        try {
            out.pattern = new MatchPredicate(rawTemplate);
            return out;
        } catch (PatternSyntaxException x) {
            return null;
        }
    }

    @Override
    public String getAgent() {
        return agent;
    }

    @Override
    public boolean match(AgentChannel channel) {
        return pattern.test(channel.getPath());
    }

}
