package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Component;
import java.awt.Dimension;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableColumn;

/**
 * Renderer for monitoring data table cells.
 * Expects an instance of {@link FormattedValue} as the value of the cell.
 *
 * @author onoprien
 */
public class MonitorTableCellRenderer extends DefaultTableCellRenderer {

// -- Fields : -----------------------------------------------------------------
    
    private boolean ensureWidth = false;
    private boolean ensureHeight = false;
    private int addedSpace = 0;
    
    private String additionalSpace;

// -- Life cycle : -------------------------------------------------------------
    
// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the flag that determines whether the table column width should increase if necessary to accommodate new values.
     * The default value is {@code false}.
     * 
     * @param ensureWidth If {@code true}, the table column width will increase if necessary to accommodate new values.
     */
    public void setEnsureWidth(boolean ensureWidth) {
        this.ensureWidth = ensureWidth;
    }

    /**
     * Sets the flag that determines whether the table row height should increase if necessary to accommodate new values.
     * The default value is {@code false}.
     * 
     * @param ensureHeight If {@code true}, the table row height will increase if necessary to accommodate new values.
     */
    public void setEnsureHeight(boolean ensureHeight) {
        this.ensureHeight = ensureHeight;
    }

    /**
     * Sets a flag that determines the number of space characters used to pad the text provided by {@code FormattedValue}.
     * 
     * @param addedSpace 
     * <ul>
     * <li>0 : no padding (default)
     * <li>positive : {@code addedSpace} spaces added before and after the text
     * <li>negative : ensures presence of {@code addedSpace} spaces before and after the text, counting space characters already in {@code FormattedValue}
     * </ul>
     */
    public void setAddedHorizontalSpace(int addedSpace) {
        this.addedSpace = - addedSpace;
        if (this.addedSpace <= 0) {
            additionalSpace = null;
        } else if (this.addedSpace > 0) {
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < this.addedSpace; i++) {
                sb.append(' ');
            }
            additionalSpace = sb.toString();
        }
    }

// -- Rendering : --------------------------------------------------------------
    
    @Override
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {

        if (value instanceof FormattedValue) {
            FormattedValue fv = (FormattedValue) value;
            super.getTableCellRendererComponent(table, addSpace(fv.getText()), false, false, row, column);
            setBackground(fv.getBgColor());
            setForeground(fv.getFgColor());
            setHorizontalAlignment(fv.getHorizontalAlignment());
            setToolTipText(fv.getToolTip());
        } else if (value == null) {
            super.getTableCellRendererComponent(table, addSpace(" "), false, false, row, column);
            TableColumn c = table.getColumnModel().getColumn(column);
            setBackground(MonitorFormat.COLOR_OFF);
            setHorizontalAlignment(SwingUtilities.CENTER);
        } else {
            super.getTableCellRendererComponent(table, addSpace(value.toString()), false, false, row, column);
            setHorizontalAlignment(SwingUtilities.CENTER);
        }

        if (ensureWidth) {
            int w = getMinimumSize().width + 2;
            TableColumn c = table.getColumnModel().getColumn(column);
            if (w > c.getMinWidth()) {
                c.setMinWidth(w);
            }
        }

        if (ensureHeight) {
            int h = getMinimumSize().height;
            if (h > table.getRowHeight(row)) {
                table.setRowHeight(row, h);
            }
        }
        
        return this;
    }

    @Override
    public Dimension getMinimumSize() {
        return getPreferredSize();
    }
    
// -- Local methods : ----------------------------------------------------------
    
    private String addSpace(String text) {
        if (addedSpace == 0) {
            return text;
        } else if (addedSpace < 0) {
            return additionalSpace + text + additionalSpace;
        } else {
            int n = text.length();
            StringBuilder sb = new StringBuilder();
            int space = 0;
            int i = 0;
            while (i<n && text.charAt(i++) == ' ') space++;
            while (space++ < addedSpace) {
                sb.append(' ');
            }
            sb.append(text);
            if (i != n) { // non-space-only original text
                space = 0;
                i = n;
                while (text.charAt(--i) == ' ') space++;
                while (space++ < addedSpace) {
                    sb.append(' ');
                }
            }
            return sb.toString();
        }
    }

}
