package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import java.util.stream.Collectors;
import javax.swing.JComponent;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.gconsole.base.Console;

/**
 * View of monitoring data.
 * <p>
 * {@code AgentStatusListener} methods are called on the {@code AgentStatusAggregator} thread.
 * All other methods declared by this interface should be called on AWT Event Dispatch Thread.
 *
 * @author onoprien
 */
public interface MonitorView extends AgentStatusListener {
    
    /**
     * Category for general purpose {@link Persistable Persistable} views.
     */
    static public String CATEGORY = "MonitorView";
    
    /** Default {@code MonitorField} to be used for displaying a value corresponding to a group of display channels. */
    static public final MonitorField DEFAULT_GROUP_FIELD = MonitorField.AVERAGE_VALUE;
    
    /** Available standard {@code MonitorField} constants for displaying a value corresponding to a group of display channels. */
    static public final MonitorField[] AVAILABLE_GROUP_FIELDS = {
        MonitorField.AVERAGE_VALUE, 
        MonitorField.MAX_VALUE, 
        MonitorField.MIN_VALUE, 
        MonitorField.MEDIAN_VALUE
    };
    
// -- Getters and setters : ----------------------------------------------------
    
    /**
     * Returns the graphical component maintained by this view.
     * @return Graphical component to be displayed by the GUI.
     */
    JComponent getPanel();
    
    /**
     * Returns the name of this view.
     * @return  View name.
     */
    default String getName() {
        JComponent panel = getPanel();
        return panel == null ? "" : panel.getName();
    }
    
    /**
     * Sets the name of this view.
     * @param name View name.
     */
    default void setName(String name) {
        JComponent panel = getPanel();
        if (panel != null) panel.setName(name);
    }
    
    /**
     * Returns the filter associated with this view, or {@code null} if there is none.
     * @return Channels filter.
     */
    AgentChannelsFilter getFilter();
    
    /**
     * Sets the channels filter to be used by this view.
     * The filter determines what channels are displayed by this view, and how they are displayed.
     * @param filter Filter to be associated with this view.
     */
    void setFilter(AgentChannelsFilter filter);
    
    /**
     * Returns formatter associated with this view, if any.
     * @return Formatter associated with this view, or {@code null} if there is no formatter.
     */
    default MonitorFormat getFormater() {
        return null;
    }
    
    /**
     * Associates a formatter with this view.
     * The exact way the formatter is used - if at all - depends on specific implementation.
     * @param formatter Formatter to be associated with this view.
     */
    default void setFormatter(MonitorFormat formatter) {
    }
    
    /**
     * Returns {@code true} if this view has no content to display at the moment.
     * This can be used report that the view can be closed, for example.
     * The default implementation returns {@code false}.
     * @return True if this view has no content to display.
     */
    default boolean isEmpty() {
        return false;
    }
    
    
// -- Grouping display channels (optional) : -----------------------------------
    
    /**
     * Computes group name based on the display path returned by the filter.
     * The default implementation splits around the last "/".
     * 
     * @param displayPath Display path.
     * @return Group name, or {@code null} if this display path does not belong to any group.
     */
    default String getGroup(String displayPath) {
        int i = displayPath.lastIndexOf("/");
        return displayPath.substring(i+1);
    }
    
    /**
     * Returns a list of display channels groups.
     * The default implementation calls {@code getFilter().getDisplayChannels()}
     * and then applies {@code getGroup(displayPath)} to the result.
     * 
     * @return List of groups, or {@code null} is the list is unknown at this time.
     */
    default List<String> getGroups() {
        AgentChannelsFilter filter = getFilter();
        if (filter == null) return null;
        List<String> channels = filter.getDisplayChannels();
        if (channels == null) return null;
        LinkedHashSet<String> groups = new LinkedHashSet<>();
        channels.forEach(displayPath -> {
            String group = getGroup(displayPath);
            if (group != null) groups.add(group);
        });
        return new ArrayList<>(groups);
    }
    
    /**
     * Retrieves a list {@code MonitorField} for display channel groups from the filter.
     * The default implementation calls {@code getFilter().getFields(true)} and
     * then applies {@code stringToField(...)}.
     * 
     * @return List of fields for display channel groups, or {@code null} if the filter does not provide the list.
     */
    default List<MonitorField> getFields() {
        AgentChannelsFilter filter = getFilter();
        if (filter == null) return null;
        List<String> channels = filter.getDisplayChannels();
        if (channels == null) return null;
        List<String> fields = filter.getFields(true);
        if (fields == null || fields.size() != channels.size()) return null;
        return fields.stream().map(s -> stringToField(s)).collect(Collectors.toList());
    }
    
    /**
     * Returns {@code MonitorField} given its string representation.
     * The default implementation selects from {@link #AVAILABLE_GROUP_FIELDS}, and returns {@link #DEFAULT_GROUP_FIELD} if none matches.
     * 
     * @param field String representation of {@code MonitorField}.
     * @return {@code MonitorField} corresponding to the provided string.
     */
    default MonitorField stringToField(String field) {
        for (MonitorField mf : AVAILABLE_GROUP_FIELDS) {
            if (mf.toString().equals(field)) {
                return mf;
            }
        }
        return DEFAULT_GROUP_FIELD;
    }
    
    /**
     * Converts {@code MonitorField} to its string representation.
     * The default implementation uses {@code toString()}.
     * 
     * @param field {@code MonitorField} constant to convert.
     * @return String representation.
     */
    default String fieldToString(MonitorField field) {
        return field.toString();
    }
    

// -- Installation : -----------------------------------------------------------
    
    /**
     * Installs this view, connecting it to the data source.
     */
    default void install() {
        AgentChannelsFilter filter = getFilter();
        if (filter == null) {
            filter = AgentChannelsFilter.ALL;
            setFilter(filter);
        }
        Console.getConsole().getStatusAggregator().addListener(this, filter);
    }
    
    /**
     * Uninstalls this view, disconnecting it from the data source.
     * Uninstalling a view that has not been installed should have no effect.
     */
    default void uninstall() {
        Console.getConsole().getStatusAggregator().removeListener(this);
    }
            
}
