package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridLayout;
import java.util.*;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import org.lsst.ccs.command.DictionaryArgument;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.gconsole.base.Const;

/**
 * Graphics component that displays information on a command and accepts arguments input.
 *
 * @author emarin
 */
public final class ArgInputPanel extends JPanel implements ListSelectionListener {
    
// -- Fields : -----------------------------------------------------------------

    private DictionaryCommand command;
    private ArgInput[] argGetters;
    private final JPanel formPane;
    private final JEditorPane commandDesc;
    private JButton sendCmdButton;
    
    private final Map<String,String[]> history = new LinkedHashMap<String,String[]>(4, 0.8f, true) {
        @Override
        protected boolean removeEldestEntry(Map.Entry<String,String[]> eldest) {
            return size() > 50;
        }
    };
    private boolean valuesValid;

// -- Life cycle : -------------------------------------------------------------
    
    public ArgInputPanel() {
        setLayout(new BorderLayout());
        
        // Command description panel:
        
        commandDesc = new JEditorPane();
        add(new JScrollPane(commandDesc), BorderLayout.CENTER);
        commandDesc.setBackground(new Color(0xed, 0xec, 0xeb)); //TODO : color must match system colors
        commandDesc.setEditable(false);
        commandDesc.setContentType("text/html");
        commandDesc.setText("      ");
        
        // Argument input panel:
        
        formPane = new JPanel();
        formPane.setBorder(BorderFactory.createCompoundBorder(
                      BorderFactory.createEtchedBorder(), 
                      BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE)));
    }
    
    void setCommandButton(JButton button) {
        sendCmdButton = button;
    }
    
    
// -- Updates : ----------------------------------------------------------------
    
    @Override
    public void valueChanged(ListSelectionEvent lse) {
        CommandListPanel commandListPanel = (CommandListPanel) lse.getSource();
        command = commandListPanel.getSelectedValue();
        clear();
        if (command != null) {
            fill();
        }
        boolean enabled = commandListPanel.isVisibleCommandSelected();
        setEnabled(enabled);
        if (enabled) {
            add(formPane, BorderLayout.SOUTH);
            updateValidity(true);
        } else {
            remove(formPane);
            sendCmdButton.setEnabled(false);
        }
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns a pretty string for displaying the values to the user.
     * @return Human-readable representation of argument values.
     */
    public String getValuesAsString() {
        StringBuilder res = new StringBuilder();
        for (ArgInput argGetter : argGetters) {
            String s = argGetter.get();
            if (s.contains(" ")) {
                res.append("\"").append(s).append("\"");
            } else {
                res.append(s);
            }
            res.append(" ");
        }
        return res.toString();
    }
    
    public String[] getValues() {
        String[] res = new String[argGetters.length];
        for (int i = 0; i < argGetters.length; i++) {
            res[i] = argGetters[i].get();
        }
        if (command != null && res.length != 0) {
            history.put(command.getCommandName(), (String[])res.clone());
        }
        return res;
    }
    
// -- Local methods : ----------------------------------------------------------
    
    private void clear() {
        formPane.removeAll();
        commandDesc.setText("       ");
        valuesValid = false;
        sendCmdButton.setEnabled(valuesValid);
    }
    
    private void fill() {
        
        // Argument input panel:
        
        final int nArgs = command.getArguments().length;
        formPane.setLayout(new GridLayout(nArgs, 2));
        argGetters = new ArgInput[nArgs];
        String[] hist = history.get(command.getCommandName());
        if (hist != null && hist.length != nArgs) {
            hist = null;
        }
        for (int i = 0; i < nArgs; i++) {
            DictionaryArgument da = command.getArguments()[i];
            argGetters[i] = new ArgInput(da);
            formPane.add(argGetters[i].getLabel());
            formPane.add(argGetters[i].getEditor());
        }
        if (hist != null) {
            for (int i = 0; i < nArgs; i++) {
                if (hist[i] != null) {
                    argGetters[i].set(hist[i]);
                }
            }
        }

        // Command description panel:
        
        StringBuilder sb = new StringBuilder("<html>");
        sb.append(command.getType()).append(". Level: ").append(command.getLevel());
        sb.append("<h3>").append(command.getDescription()).append("</h3>");
        commandDesc.setText(sb.toString());
        
        // Refresh display:
        
        updateValidity(true);
        setEnabled(isEnabled());
        repaint();
        validate();
    }
    
    private void updateValidity(boolean argValid) {
        if (valuesValid == argValid) return; // no change
        if (argValid) {
            for (ArgInput argInput : argGetters) {
                if (!argInput.isValueValid()) {
                    return;  // overall state is still invalid
                }
            }
            valuesValid = true; // all argument values are now valid
        } else { // overall state is now invalid
            valuesValid = false;
        }
        sendCmdButton.setEnabled(valuesValid);
    }

    @Override
    public void setEnabled(boolean enabled) {
        super.setEnabled(enabled);
        if (enabled) {
            add(formPane, BorderLayout.SOUTH);
        } else {
            this.remove(formPane);
        }
    }
    
    private String getDefaultValueOfArgument(DictionaryArgument da) {
        String out = da.getDefaultValue();
        if (out != null) {
            switch (out) {
                case Argument.NOT_SET:
                    return null;
                case Argument.NULL:
                    return "null";
            }
        }
        return out;
    }
    
    
// -- Single argument input component : ----------------------------------------
    
    private class ArgInput {
        
        private final DictionaryArgument da;
        private boolean valueValid;
        private final JComponent editor;
        private final JLabel label;
        
        ArgInput(DictionaryArgument da) {
            this.da = da;
            
            label = new JLabel(da.getName());
            String tip = da.getDescription();
            if (tip != null && !tip.isEmpty()) label.setToolTipText(tip);

            List<String> allowedValues = da.getAllowedValues();
            String type = da.getSimpleType();
            if (allowedValues.isEmpty() && "boolean".equals(type)) {
                allowedValues = Arrays.asList(new String[] {"true", "false"});
            }
            if(allowedValues.isEmpty()) {
                HintTextField htf = new HintTextField(type);
                String defValue = getDefaultValueOfArgument(da);
                if (defValue == null) {
                    valueValid = false;
                } else {
                    htf.setText(defValue);
                    valueValid = true;
                }
                htf.addCaretListener(e -> validateValue());
                editor = htf;
            } else {
                JComboBox<String> comboBox = new JComboBox<>(allowedValues.toArray(new String[0]));
                String defValue = da.getDefaultValue();
                if (defValue != null) {
                    comboBox.setSelectedItem(defValue);
                }
                valueValid = true;
                editor = comboBox;
            }
        }
        
        void set(String value) {
            if (editor instanceof HintTextField) {
                ((HintTextField)editor).setText(value);
            } else if (editor instanceof JComboBox) {
                ((JComboBox<String>)editor).setSelectedItem(value);
            }
            validateValue();
        }
        
        String get() {
            String out;
            if (editor instanceof HintTextField) {
                out = ((HintTextField)editor).getText();
            } else {
                out = ((JComboBox<String>)editor).getSelectedItem().toString();
            }
            return out;
        }
        
        boolean isValueValid() {
            return valueValid;
        }
        
        private void validateValue() { // FIXME: Ideally, validation should use the same algorithm as the command invocation
            boolean nowValid = true;
            if (editor instanceof HintTextField) {
                String s = ((HintTextField)editor).getText();
                switch (da.getSimpleType()) {
                    case "int":
                        try {
                            Integer.decode(s);
                        } catch (NumberFormatException x) {
                            nowValid = false;
                        }
                        break;
                    case "long":
                        try {
                            Long.decode(s);
                        } catch (NumberFormatException x) {
                            nowValid = false;
                        }
                        break;
                    case "float":
                        try {
                            Float.parseFloat(s);
                        } catch (NumberFormatException x) {
                            nowValid = false;
                        }
                        break;
                    case "double":
                        try {
                            Double.parseDouble(s);
                        } catch (NumberFormatException x) {
                            nowValid = false;
                        }
                        break;
                }
            }
            if (nowValid != valueValid) {
                valueValid = nowValid;
                updateValidity(valueValid);
            }
        }
        
        JComponent getEditor() {
            return editor;
        }
        
        JLabel getLabel() {
            return label;
        }

    }
    
    
    
    
}
