package org.lsst.ccs.gconsole.plugins.trending;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ThreadPoolExecutor;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindowSelector;
import org.lsst.ccs.gconsole.services.rest.LsstRestService;
import org.lsst.ccs.localdb.statusdb.server.ChannelMetaData;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.Datas;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.statusdb.server.TrendingResult;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Source of trending data based on a REST server. Provides getters for retrieving the list
 * of known trending channels and the data for specified channels.
 * <p>
 * This sources watches for external events that can change the list of known channels,
 * and notifies listeners when this happens.
 * <p>
 * <i>Current implementation note. This class maintains a map of paths to {@code DataChannel},
 * but only the list of paths is externally available. Metadata provided by {@code DataChannel}
 * is unused. Accordingly, whenever the map is updated (due to time window or REST server
 * change), listeners are not notified unless the set of available paths changes.</i>
 *
 * @author onoprien
 */
public class RestSource {

// -- Fields : -----------------------------------------------------------------
    
    private final LsstTrendingPlugin plugin;
    private final TrendingPreferences pref;
    private final LsstRestService restService;
    private final ThreadPoolExecutor dataFetcher;
    private final Logger logger;
    
    private final Map<String,DataChannel> channels = new LinkedHashMap<>();
    private volatile int begin;
    
    private final CopyOnWriteArrayList<ChangeListener> listeners = new CopyOnWriteArrayList<>();
    
    private final TimeWindowSelector.Listener timeListener;
    private final ChangeListener restListener;
    

// -- Life cycle : -------------------------------------------------------------
    
//<editor-fold defaultstate="collapsed">
    RestSource(LsstTrendingPlugin plugin) {
        
        this.plugin = plugin;
        pref = plugin.getPreferences();
        restService = (LsstRestService) plugin.getConsole().getConsoleLookup().lookup(LsstRestService.class);
        dataFetcher = plugin.getExecutor();
        logger = plugin.getConsole().getLogger();
        
        timeListener = event -> {
            onTimeWindowChange(event.getTimeWindow());
        };
        restListener = event -> {
            onRestServerChange();
        };
    }
    
    /**  Initialization. Called on EDT. */
    void start() {
        plugin.getTimeWindowSelector().addListener(timeListener);
        restService.addListener(restListener);
        onTimeWindowChange(plugin.getSelectedTimeWindow());
    }
    
    /** Shutdown. Called on EDT. */
    void stop() {
        TimeWindowSelector ts = plugin.getTimeWindowSelector();
        if (ts != null) {
            ts.removeListener(timeListener);
        }
        restService.removeListener(restListener);
    }
//</editor-fold>

// -- Updating this source and notifying listeners : ---------------------------
    
//<editor-fold defaultstate="collapsed">
    /**
     * Refreshes the list of channels by retrieving them from the REST server.
     */
    public void refresh() {
        dataFetcher.execute(this::refreshChannelList);
    }
    
    private void onRestServerChange() {
        refresh();
    }
    
    private void onTimeWindowChange(TimeWindow timeWindow) {
        int time;
        if (timeWindow == null) {
            time = 86400;
        } else {
            time = (int) Math.max(86400, (System.currentTimeMillis() - timeWindow.getLowerEdge())/1000L + 3600L);
        }
        if (time != begin) {
            begin = time;
            refresh();
        }
    }
    
    private void refreshChannelList() {
        List<DataChannel> cc;
        try {
            DataChannel.DataChannelList channelList = restService.getChannelList(begin);
            if (channelList == null || channelList.list == null || channelList.list.isEmpty()) {
                cc = Collections.emptyList();
            } else {
                cc = channelList.list;
            }
        } catch (RuntimeException e) {
            logger.warn("Unable to retrieve the list of channels from the REST server.");
            cc = Collections.emptyList();
        }
        synchronized (channels) {
            if (!(channels.size() == cc.size())) {
                channels.clear();
                cc.forEach(c -> channels.put(String.join("/", c.getPath()), c));
                fireEvent();
            }
        }
    }
    
    /**
     * Registers a listener to be notified if the list of channels available from this trending source changes.
     * This method can be called on any thread. The listeners are notified on the EDT.
     * 
     * @param listener Listener to be added.
     */
    public void addListener(ChangeListener listener) {
        listeners.add(listener);
    }
    
    /**
     * Removes a listener.
     * This method can be called on any thread.
     * 
     * @param listener Listener to be removed.
     */
    public void removeListener(ChangeListener listener) {
        listeners.remove(listener);
    }
    
    /**
     * Removes all listeners.
     * This method can be called on any thread.
     */
    public void removeAllListeners() {
        listeners.clear();
    }
    
    /**
     * Notifies listeners of changes in this source.
     * Can be called on any thread. Listeners are notified on EDT.
     */
    protected void fireEvent() {
        SwingUtilities.invokeLater(() -> {
            ChangeEvent event = new ChangeEvent(RestSource.this);
            listeners.forEach(listener -> {
                try {
                    listener.stateChanged(event);
                } catch (RuntimeException x) {
                    if (logger != null) {
                        logger.error(listener +" threw an exception while processing RestSource event", x);
                    }
                }
            });
        });
    }
//</editor-fold>"

// -- Retrieving data : --------------------------------------------------------

    /**
     * Returns the list of known data channel paths.
     * Calling this method does not retrieve data from the REST server.
     * 
     * @return List of channel paths.
     */
    public List<String> getChannels() {
        synchronized (channels) {
            return new ArrayList<>(channels.keySet());
        }
    }

    /**
     * Retrieves from the REST server and processes historical data for the specified trending channel. 
     * 
     * @param path Channel path.
     * @param begin Beginning of time window.
     * @param end End of time window.
     * @param metadata Set of metadata types to be included.
     * @param history Previously known data.
     * @return Newly created data container, or {@code null} if retrieval fails for any reason.
     */
    public TrendData get(String path, long begin, long end, EnumSet<Trend.Meta> metadata, TrendData history) {
        
        int nBins = pref.getNBins();
        String flavor;
        if (nBins > 0) {
            flavor = pref.isUseRawData() ? null : "stat";
        } else {
            flavor = pref.isUseRawData() ? "raw" : "stat";
        }
        Datas datas;
        try {
            datas = restService.getData(begin, end, flavor, nBins, path);
        } catch (RuntimeException x) {
            return null;
        }
        if (datas == null) return null;
        TrendingResult result = datas.getDataArray()[0].getTrendingResult();
            
        Set<String> onPoint, offPoint;
        if (metadata == null) {
            onPoint = null;
            offPoint = null;
        } else {
            onPoint = new HashSet<>();
            onPoint.add("value");
            offPoint = new HashSet<>();
            metadata.forEach(m -> {
                if (m.isOnPoint()) {
                    onPoint.addAll(m.getKeys());
                } else {
                    offPoint.addAll(m.getKeys());
                }
            });
        }
        
        Map<String, long[]> times = new HashMap<>();
        Map<String, double[]> values = new HashMap<>();
        
        long lowEdge = begin;
        long highEdge = end;
        TrendingData[] dataArray = result.getTrendingDataArray();
        if (dataArray != null) {
            int n = dataArray.length;
            long[] time = new long[n];
            times.put("value", time);
            for (int i=0; i<n; i++) {
               TrendingData td = dataArray[i];
               time[i] = td.getAxisvalue().getValue();
               for (TrendingData.DataValue dv : td.getDatavalue()) {
                   String key = dv.getName();
                   if (onPoint == null || onPoint.contains(key)) {
                       double[] valueArray = values.get(key);
                       if (valueArray == null) {
                           valueArray = new double[n];
                           values.put(key, valueArray);
                       }
                       valueArray[i] = dv.getValue();
                   }
               }
            }
        }
        
        List<ChannelMetaData> metaList = result.getChannelMetadata();
        HashMap<String,ArrayList<MetaPoint>> metaMap = new HashMap<>();
        for (ChannelMetaData meta : metaList) {
            String key = meta.getName();
            if (offPoint == null || offPoint.contains(key)) {
                try {
                    ArrayList<MetaPoint> ps = metaMap.get(key);
                    if (ps == null) {
                        ps = new ArrayList<>();
                        metaMap.put(key, ps);
                    }
                    double value = Double.parseDouble(meta.getValue());
                    long start = meta.getTstart();
                    long stop = meta.getTstop();
                    if ((start <= highEdge || start == -1L) && (stop >= lowEdge || stop == -1L)) {
                        if (start < lowEdge) {
                            start = lowEdge;
                        }
                        ps.add(new MetaPoint(start, value));
                        if (stop > highEdge || stop == -1) {
                            stop = highEdge;
                        }
                        ps.add(new MetaPoint(stop, value));
                    }
                } catch (NumberFormatException x) {
                }
            }
        }
        
        metaMap.forEach((key, points) -> {
            int n = points.size();
            long[] t = new long[n];
            double[] v = new double[n];
            for (int i=0; i<n; i++) {
                MetaPoint p = points.get(i);
                t[i] = p.time;
                v[i] = p.value;
            }
            times.put(key, t);
            values.put(key, v);
        });
        
        return new TrendData(times, values, new long[] {begin, end});
    }
    
    
// Local methods and classes : -------------------------------------------------
    
    static private class MetaPoint {
        MetaPoint(long t, double v) {
            time = t;
            value = v;
        }
        long time;
        double value;
    }
    
}
