package org.lsst.ccs.gconsole.plugins.trending;

import java.awt.Component;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import javax.swing.SwingUtilities;
import org.freehep.application.Application;
import org.freehep.application.mdi.PageEvent;
import org.freehep.application.mdi.PageListener;
import org.freehep.application.studio.Studio;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Handles automatic refreshing of trending plots.
 * The class is thread-safe, all public methods can be called from any thread.
 *
 * @author onoprien
 */
final public class AutoRefreshManager {

// -- Private parts : ----------------------------------------------------------
    
    private final LsstTrendingPlugin plugin;
    
    /**
     * Periodic task that runs on Agent's Scheduler and launches a task on EDT that
     * grabs lists of Threads that might require refreshing.
     * The period of this task is the minimum set through preferences.
     */
    private PeriodicTask launcher;
    
    /**
     * Background refreshing executor.
     */
    private volatile ExecutorService executor;
    private final int EXECUTOR_MAX_THREADS = 2;
    
    /**
     * Listener that triggers refreshing when a page is selected.
     */
    private PageListener pageListener;
    
    private volatile long[] delayBounds; // min/max delay in millis
    

// -- Life cycle : -------------------------------------------------------------
    
    public AutoRefreshManager(LsstTrendingPlugin plugin) {
        this.plugin = plugin;
    }
    
    /**
     * Starts auto-refreshing.
     * Compiles a list of all regions that need refreshing, puts them in the queue,
     * and starts data fetcher thread.
     */
    public synchronized void start() {
        
        TrendingPreferences pref = plugin.getPreferences();
        setDelay(pref.getRefreshMin(), pref.getRefreshMax(), TimeUnit.SECONDS);
        
        // Start executor
        
        ThreadFactory tFactory = new ThreadFactory() {
            ThreadFactory delegate = Executors.defaultThreadFactory();
            int id = 0;
            @Override
            public Thread newThread(Runnable r) {
                Thread thread = delegate.newThread(r);
                thread.setName("Auto-Refresh Executor " + id++);
                return thread;
            }
        };
        executor = Executors.newFixedThreadPool(EXECUTOR_MAX_THREADS, tFactory);
        
        // submit launcher task
        
        Scheduler scheduler = plugin.getConsole().getScheduler();
        launcher = new PeriodicTask(scheduler, this::launch, false, "Auto Refresh Launcher", Level.WARNING, delayBounds[0], TimeUnit.MILLISECONDS);
        launcher.start(delayBounds[0], TimeUnit.MILLISECONDS);
        
        // listen to page selections

        pageListener = pe -> {
            if (pe.getID() == PageEvent.PAGESELECTED) {
                try {
                    TrendPage page = (TrendPage) pe.getPageContext().getPage();
                    SwingUtilities.invokeLater(() -> {
                        try {
                            int n = page.numberOfRegions();
                            for (int i=0; i<n; i++) {
                                PlotRegion region = page.region(i);
                                Plotter plotter = region.currentPlot();
                                if (plotter != null && plotter instanceof TrendPlotter) {
                                    executor.execute(() -> refresh(((TrendPlotter)plotter).getTrends()));
                                }
                            }
                        } catch (Throwable t) {
                        }
                    });
                } catch (ClassCastException|NullPointerException x) {
                }
            }
        };
        ((Studio)Application.getApplication()).getPageManager().addPageListener(pageListener);
        
    }
    
    /**
     * Stops auto-refreshing.
     * Cancels auto-refreshing tasks and stops data fetcher executor.
     */
    public synchronized void stop() {
        
        ExecutorService exec = executor;
        executor = null;
        
        launcher.cancel(true);
        launcher = null;

        ((Studio)Application.getApplication()).getPageManager().removePageListener(pageListener);
        pageListener = null;
        
        exec.shutdownNow();
    }
    
    /**
     * Sets the allowed range of setLoading frequency.
     * 
     * @param min Minimum setLoading interval.
     * @param max Maximum setLoading interval.
     * @param unit Time unit used to specify minimum and maximum intervals.
     */
    private void setDelay(long min, long max, TimeUnit unit) {
        if (min > max) {
            throw new IllegalArgumentException();
        } else if (min == max) {
            delayBounds = new long[] {TimeUnit.MILLISECONDS.convert(min, unit)};
        } else {
            delayBounds = new long[] {TimeUnit.MILLISECONDS.convert(min, unit), TimeUnit.MILLISECONDS.convert(max, unit)};
        }
    }


// -- Operation : --------------------------------------------------------------
    
    public synchronized void updatePreferences() {
        TrendingPreferences pref = plugin.getPreferences();
        long minDelay = delayBounds[0];
        setDelay(pref.getRefreshMin(), pref.getRefreshMax(), TimeUnit.SECONDS);
        if (minDelay != delayBounds[0] && launcher != null) {
            launcher.setPeriod(delayBounds[0], TimeUnit.MILLISECONDS);
        }
    }


// -- Methods executed by tasks : ----------------------------------------------

    /** Executed on Agent's scheduler to periodically launch findRegionsToRefresh() on EDT. */
    private void launch() {
        SwingUtilities.invokeLater(this::findRegionsToRefresh);
    }
    
    /** Executed on EDT to grab lists of Trends that might need to be refreshed. */
    private void findRegionsToRefresh() {
        try {
            plugin.getPages().forEach(page -> {
                if (((Component) page).isShowing()) {
                    int n = page.numberOfRegions();
                    for (int i = 0; i < n; i++) {
                        PlotRegion region = page.region(i);
                        Plotter plotter = region.currentPlot();
                        if (plotter != null && plotter instanceof TrendPlotter) {
                            executor.execute(() -> refresh(((TrendPlotter) plotter).getTrends()));
                        }
                    }
                }
            });
        } catch (NullPointerException x) { // executor == null, auto-refresh stopped
        }
    }
    
    /** Executed by background executor to fetch data from the database. */
    private void refresh(List<Trend> trends) {
        boolean needRefresh = trends.stream().anyMatch(trend -> needsRefresh(trend));
        if (needRefresh) {
            long time = System.currentTimeMillis();
            trends.forEach(trend -> trend.setTimestamp(time));
            plugin.refresh(trends);
        }
    }
    
    

// -- Local methods : ----------------------------------------------------------
    
    private boolean needsRefresh(Trend trend) {
        TimeWindow tw = trend.getTimeWindow();
        if (tw.isFixed() && (tw.getUpperEdge() < tw.getLastUseTime())) return false;
//        if ((tw.isFixed() && (tw.getUpperEdge() < tw.getLastUseTime())) || trend.getData() == null) return false;
        long delay;
        if (delayBounds.length == 1 || trend.getData() == null) {
            delay = delayBounds[0];
        } else {
            delay = Math.round(timeToNextPoint(trend.getData())*1.1);
            delay = Math.max(delay, delayBounds[0]);
            delay = Math.min(delay, delayBounds[1]);
        }
        return trend.getTimestamp() + delay < System.currentTimeMillis();
    }
    
    private long timeToNextPoint(TrendData data) {
        long[] time = data.getTime();
        if (time == null) return delayBounds[0];
        int n = time.length;
        if (n<2) {
            return delayBounds[0];
        } else {
            return (time[n-1] - time[0]) / n;
        }
    }
    
}
