 package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.Serializable;
import java.util.*;
import java.util.concurrent.CancellationException;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import org.freehep.application.Application;
import org.freehep.application.mdi.PageContext;
import org.freehep.application.mdi.PageEvent;
import org.freehep.application.mdi.PageListener;
import org.freehep.application.mdi.PageManager;
import org.freehep.application.studio.Studio;
import org.freehep.graphicsbase.util.export.ExportDialog;
import org.freehep.util.commanddispatcher.CommandProcessor;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.PersistableAgentChannelsFilter;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.services.persist.Creator;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.gconsole.services.persist.Persistable;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.AgentPresenceManager;
import org.lsst.ccs.services.AgentService;

/**
 * Graphical console plugin that supports display of monitoring data.
 * Unless otherwise specified, methods of this class can be called on any thread.
 *
 * @author onoprien
 */
@Plugin(name="LSST Monitoring Plugin",
        id="monitor",
        description="LSST CCS monitoring data display service.")
public class LsstMonitorPlugin extends ConsolePlugin {

// -- Fields : -----------------------------------------------------------------
    
    static public String MENU_NAME = "Monitoring";
    
    private final Studio app = (Studio) Application.getApplication();
    private final ArrayList<MonitorPage> currentViews = new ArrayList<>(0);
    private final ArrayDeque<MonitorPage.Descriptor> recentViews = new ArrayDeque<>(0);

// -- Life cycle : -------------------------------------------------------------
    
    public LsstMonitorPlugin() {
    }

    /** Initialization. Called by the framework on EDT. */
    @Override
    public void initialize() {

        // Monitoring menu:
        
        Action newView = new AbstractAction("New...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                openManagedPage();
            }
        };
        getServices().addMenu(newView, "400: CCS Tools :-1:15", MENU_NAME +":1");
        
        Action loadPage = new AbstractAction("Load...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                PersistenceService service = getConsole().getSingleton(PersistenceService.class);
                try {
//                    Persistable.Descriptor desc = service.load(MonitorPage.CATEGORY, "Load monitoring page", getConsole().getWindow());
                    MonitorPage page = (MonitorPage) service.make(null, "Load monitoring page", getConsole().getWindow(), MonitorPage.CATEGORY);
                    if (page != null) {
                        openManagedPage(page);
                    }
                } catch (RuntimeException x) {
                }
            }
        };
        getServices().addMenu(loadPage, "400: CCS Tools :-1:15", MENU_NAME +":2");
        
        AgentPresenceManager apMan = getConsole().getMessagingAccess().getAgentPresenceManager();
        apMan.addAgentPresenceListener(new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) {
                for (AgentInfo info : agents) {
                    AgentInfo.AgentType type = info.getType();
                    if (!(AgentInfo.AgentType.CONSOLE.equals(type) || AgentInfo.AgentType.LISTENER.equals(type))) {
                        SwingUtilities.invokeLater(() -> {
                            Action act = new AbstractAction(info.getName()) {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    openDefaultAgentView(e.getActionCommand());
                                }
                            };
                            getServices().addMenu(act, "400: CCS Tools :-1:15", MENU_NAME + ":3", "Subsystems:1");
                            act = new AbstractAction("Default Monitor") {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    openDefaultAgentView(info.getName());
                                }
                            };
                            getServices().addMenu(act, "CCS Subsystems", info.getName() + ":-10:15");
                        });
                    }
                }
            }

            @Override
            public void disconnected(AgentInfo... agents) {
                SwingUtilities.invokeLater(() -> {
                    for (AgentInfo agent : agents) {
                        getConsole().removeMenu(" CCS Tools ", MENU_NAME, "Subsystems", agent.getName());
                        getConsole().removeMenu("CCS Subsystems", agent.getName(), "Default Monitor");
                    }
                });
            }
        });
        
        // Enabling dumping monitoring data into files:
        
        PageManager pm = app.getPageManager();
        pm.addPageListener(new Saver());
    }


// -- Opening monitoring pages : -----------------------------------------------
    
    /**
     * Opens a new graphical console page that displays the specified view.
     * Views opened by this method are NOT automatically saved and restored by this plugin.
     * This method should be called on the AWT Event Dispatch Thread.
     * 
     * @param view View to display.
     */
    public void openPage(MonitorView view) {
        JComponent panel = view.getPanel();
        if (panel != null) {
            HashMap<Object, Object> par = new HashMap<>();
            par.put(Panel.TITLE, view.getName());
            Consumer<JComponent> onClose = c -> view.uninstall();
            par.put(Panel.ON_CLOSE, onClose);
            Console.getConsole().getPanelManager().open(panel, par);
            view.install();
        }
    }
    
    /**
     * Closes the console page that displays the specified view.
     * This method should be called on the AWT Event Dispatch Thread.
     * 
     * @param view View to close.
     */
    public void closePage(MonitorView view) {
        PanelManager pm = getConsole().getPanelManager();
        JComponent panel = view.getPanel();
        if (panel != null) pm.close(panel);
        view.uninstall();
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    /**
     * Called in response to menu selection.
     * Pages opened by this method are automatically saved and restored by this plugin.
     * 
     * @param desc Monitor page descriptor.
     */
    public void openManagedPage(MonitorPage.Descriptor desc) {
        MonitorPage monPage = null;
        if (desc == null) {
            openManagedPage();
        } else {
            monPage = new MonitorPage(desc); 
        }
        openManagedPage(monPage);
    }
    
    /**
     * Called in response to menu selection.
     * Pages opened by this method are automatically saved and restored by this plugin.
     * 
     * @param desc Monitor page descriptor.
     */
    private void openManagedPage() {
        try {
            MonitorPage monPage = new MonitorPage();
            monPage.getDescriptor().setCategory(MonitorPage.CATEGORY);
            monPage.getDescriptor().setName("Monitor");
            PersistableMonitorView.Descriptor viewDesc = new PersistableMonitorView.Descriptor();
            Creator.Descriptor creDesc = new Creator.Descriptor(MonitorView.CATEGORY, "Built-In/Default");
            viewDesc.setCreator(creDesc);
            monPage.getDescriptor().setView(viewDesc);
            PersistableAgentChannelsFilter.Descriptor filterDesc = new PersistableAgentChannelsFilter.Descriptor();
            creDesc = new Creator.Descriptor(AgentChannelsFilter.CATEGORY, "Built-In/Unfiltered");
            filterDesc.setCreator(creDesc);
            monPage.getDescriptor().setFilter(filterDesc);
            monPage = monPage.edit("Open Monitor Page", null);
            openManagedPage(monPage);
        } catch (CancellationException x) {
        } catch (RuntimeException x) {
            getConsole().error("Error opening monitor page", x);
        }
        
    }

    /**
     * Opens a managed monitor page.
     * Pages opened by this method are automatically saved and restored by this plugin.
     * 
     * @param page Monitor page to display.
     */
    public void openManagedPage(MonitorPage page) {
        if (page != null) {
            JComponent panel = page.getPanel();
            if (panel != null) {
                MonitorPage.Descriptor desc = page.getDescriptor();
                HashMap<Object, Object> par = new HashMap<>();
                if (desc != null) {
                    DataPanelDescriptor panDesc = desc.getPanel();
                    if (panDesc != null && panDesc.isOpen()) {
                        Map<String, Serializable> data = panDesc.getData();
                        if (data != null) par.putAll(data);
                    }
                }
                par.put(Panel.TITLE, page.getName());
                Consumer<JComponent> onClose = c -> onClose(c);
                par.put(Panel.ON_CLOSE, onClose);
                Consumer<JComponent> onSaveAs = c -> {
                    for (MonitorPage p : currentViews) {
                        if (c == p.getPanel()) {
                            PersistenceService service = getConsole().getSingleton(PersistenceService.class);
                            service.saveAs(p.save(), "Save monitor page", null);
                            break;
                        }
                    }
                };
                par.put(Panel.ON_SAVE_AS, onSaveAs);
                Consumer<JComponent> onEdit = c -> {
                    for (MonitorPage p : currentViews) {
                        if (c == p.getPanel()) {
                            try {
                                MonitorPage editedPage = p.edit("Edit page", null);
                                if (editedPage != null && editedPage != p) {
                                    SwingUtilities.invokeLater(() -> {
                                        try {
                                            getConsole().getPanelManager().close(c);
                                        } catch (IndexOutOfBoundsException x) {
                                        }
                                        openManagedPage(editedPage);
                                    });
                                }
                            } catch (RuntimeException x) {
                            }
                            break;
                        }
                    }
                };
                par.put(Panel.ON_EDIT, onEdit);
                Console.getConsole().getPanelManager().open(panel, par);
                page.getView().install();
                currentViews.add(page);
            }
        }
    }
    
    private void openDefaultAgentView(String agentName) {
        MonitorPage.Descriptor mpDesc = new MonitorPage.Descriptor();
        mpDesc.setCategory(MonitorPage.CATEGORY);
        mpDesc.setName(agentName);
        PersistableMonitorView.Descriptor viewDesc = new PersistableMonitorView.Descriptor();
        Creator.Descriptor creDesc = new Creator.Descriptor(MonitorView.CATEGORY, "Built-In/Default");
        viewDesc.setCreator(creDesc);
        mpDesc.setView(viewDesc);
        PersistableAgentChannelsFilter.Descriptor filterDesc = new PersistableAgentChannelsFilter.Descriptor();
        creDesc = new Creator.Descriptor(AgentChannelsFilter.CATEGORY, "Built-In/Generic", agentName, "[" + agentName + "]", Par.NULL, Par.NULL, "false", Par.NULL);
        filterDesc.setCreator(creDesc);
        mpDesc.setFilter(filterDesc);
        openManagedPage(mpDesc);
    }
    
    /**
     * Called when a managed page is closed (by the user or programmatically).
     * @param panel Graphical component that has been closed.
     */
    private void onClose(JComponent panel) {
        
        MonitorPage monPage = null;
        Iterator<MonitorPage> it = currentViews.iterator();
        while (it.hasNext()) {
            MonitorPage p = it.next();
            if (panel == p.getPanel()) {
                monPage = p;
                it.remove();
                break;
            }
        }
        if (monPage == null) return;
        monPage.getView().uninstall();
        
        MonitorPage.Descriptor pd = monPage.save();
        if (pd != null) {
            String name = pd.getName();
            if (name != null) {
                Iterator<MonitorPage.Descriptor> itt = recentViews.iterator();
                while (itt.hasNext()) {
                    if (itt.next().getName().equals(name)) {
                        itt.remove();
                        break;
                    }
                }
                if (recentViews.size() > 9) {
                    recentViews.pollLast();
                }
                recentViews.addFirst(pd);
                resetRecentMenu();
            }        
        }
    }
    
    private void resetRecentMenu() {
        Console console = getConsole();
        console.removeMenu(" CCS Tools ", MENU_NAME, "Recent");
        int i = 0;
        for (MonitorPage.Descriptor desc : recentViews) {
            Action act = new AbstractAction(desc.getName()) {
                @Override
                public void actionPerformed(ActionEvent e) {
                    openManagedPage(desc);
                }
            };
            getServices().addMenu(act, "400: CCS Tools :-1:15", MENU_NAME +":100:100", "Recent:" + i++);
        }
    }
    
    
// -- Saving and restoring sessions : ------------------------------------------

    @Override
    public ComponentDescriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        if (!currentViews.isEmpty()) {
            MonitorPage.Descriptor[] current = currentViews.stream().map(page -> page.save()).collect(Collectors.toList()).toArray(new MonitorPage.Descriptor[0]);
            desc.setCurrentViews(current);
        }
        if (!recentViews.isEmpty()) {
            MonitorPage.Descriptor[] recent = recentViews.toArray(new MonitorPage.Descriptor[0]);
            desc.setRecentViews(recent);
        }
        return desc;
    }

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        PanelManager pm = getConsole().getPanelManager();
        for (MonitorPage page : new ArrayList<>(currentViews)) {
            pm.close(page.getPanel());
        }
        currentViews.clear();
        recentViews.clear();
        if (storageBean instanceof Descriptor) {
            Descriptor desc = (Descriptor) storageBean;
            MonitorPage.Descriptor[] mvd = desc.getCurrentViews();
            if (mvd != null) {
                for (MonitorPage.Descriptor d : mvd) {
                    openManagedPage(d);
                }
            }
            if (desc.getRecentViews() != null) {
                recentViews.addAll(Arrays.asList(desc.getRecentViews()));
            }
        }
        resetRecentMenu();
        return true;
    }

    static public class Descriptor extends ComponentDescriptor {

        private MonitorPage.Descriptor[] currentViews;
        private MonitorPage.Descriptor[] recentViews;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public MonitorPage.Descriptor[] getCurrentViews() {
            return currentViews;
        }

        public void setCurrentViews(MonitorPage.Descriptor[] currentViews) {
            this.currentViews = currentViews;
        }

        public MonitorPage.Descriptor[] getRecentViews() {
            return recentViews;
        }

        public void setRecentViews(MonitorPage.Descriptor[] recentViews) {
            this.recentViews = recentViews;
        }
        
    }
    
    
// -- Temporary code for saving monitoring pages : -----------------------------
    
    private class Saver extends CommandProcessor implements PageListener {
        
        private boolean on;

        @Override
        public void pageChanged(PageEvent pe) {
            PageContext context = pe.getPageContext();
            if (context == null) return;
            JComponent panel = (JComponent) context.getPage();
            if (panel == null) return;
            switch (pe.getID()) {
                case PageEvent.PAGESELECTED:
                    MonitorDisplay md = findMonitorDisplay(panel, false);
                    if (md == null) {
                        if (on) {
                            app.getCommandTargetManager().remove(this);
                            on = false;
                        }
                    } else {
                        if (!on) {
                            app.getCommandTargetManager().add(this);
                            on = true;
                        }
                    }
                    break;
                case PageEvent.PAGEDESELECTED:
//                    if (on) {
//                        app.getCommandTargetManager().remove(this);
//                        on = false;
//                    }
                    break;
            }
        }
        
        private MonitorDisplay findMonitorDisplay(JComponent component, boolean visibleOnly) {
            if (visibleOnly && !component.isShowing()) {
                return null;
            }
            if (component instanceof MonitorDisplay) {
                return (MonitorDisplay) component;
            }
            Component[] cc = component.getComponents();
            for (Component c : cc) {
                if (c instanceof JComponent) {
                    MonitorDisplay md = findMonitorDisplay((JComponent)c, visibleOnly);
                    if (md != null) return md;
                }
            }
            return null;
        } 
        
        public void onSaveAs() {
            try {
                PageManager pm = app.getPageManager();
                PageContext context = pm.getSelectedPage();
                JComponent component = (JComponent) context.getPage();
                MonitorDisplay md = findMonitorDisplay(component, true);

                Properties user = app.getUserProperties();
                String creator = "creator";
                ExportDialog dlg = new ExportDialog(creator, false);
                dlg.addExportFileType(new MonitorExport());
                dlg.addExportFileType(new MonitorExport()); // working around the bug in ExportDialog line 160
                dlg.setUserProperties(user);
                dlg.showExportDialog(app, "Save As...", (Component)md, "monitor");
            } catch (NullPointerException|ClassCastException x) {
                app.error("No selected monitoring table", x);
            }
        }
        
    }
    
    static public void saveData(OutputStream out, String mimeType, List<AgentChannel> channels, List<MonitorField> fields) {
        
        PrintStream ps = new PrintStream(out);
        
        StringBuilder sb = new StringBuilder();
        for (MonitorField field : fields) {
            sb.append(field.getTitle()).append(",");
        }      
        ps.println(sb.substring(0, sb.length()-1));
        
        for (AgentChannel channel : channels) {
            sb = new StringBuilder();
            for (MonitorField field : fields) {
                sb.append(channel.<String>get(field.getKey())).append(",");
            }
            ps.println(sb.substring(0, sb.length()-1));
        }
        
    }

    
// -- Pre-defined monitor pages: -----------------------------------------------
    
    @Create(category = MonitorPage.CATEGORY,
            name = "Core States",
            path = "Built-In/Core States",
            description = "Core states of all subsystems")
    static public MonitorPage makeCoreStatesPage() {
        
        MonitorPage.Descriptor mpDesc = new MonitorPage.Descriptor();
        mpDesc.setCategory(MonitorPage.CATEGORY);
        
        LazyTreeView.Descriptor viewDesc = new LazyTreeView.Descriptor();
        Creator.Descriptor creDesc = new Creator.Descriptor(MonitorView.CATEGORY, "Built-In/Optimized Tree");
        viewDesc.setCreator(creDesc);
        TreeMap<String, Serializable> nodes = new TreeMap<>();
        LazyTreeView.InterNodeDescriptor root = new LazyTreeView.InterNodeDescriptor();
        root.setExpanded(true);
        root.setDisplayMode(LazyTreeView.InterNodeDescriptor.DisplayMode.MESH);
        root.setSort(LazyTreeView.InterNodeDescriptor.Sort.ALPHABETIC);
        nodes.put("", root);
        viewDesc.setNodes(nodes);
        mpDesc.setView(viewDesc);
        
        PersistableAgentChannelsFilter.Descriptor filterDesc = new PersistableAgentChannelsFilter.Descriptor();
        creDesc = new Creator.Descriptor(AgentChannelsFilter.CATEGORY, "Built-In/Core States", "Core States", Par.NULL);
        filterDesc.setCreator(creDesc);
        mpDesc.setFilter(filterDesc);
        
        return (MonitorPage) PersistenceService.getService().make(mpDesc);
    }

}
