
package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.*;
import javax.swing.Box;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.CommandState;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.bus.states.OperationalState;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusAggregator;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusEvent;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import static org.lsst.ccs.gconsole.plugins.commandbrowser.Browser.ICON_LOCKED;
import static org.lsst.ccs.gconsole.plugins.commandbrowser.Browser.ICON_UNAVAILABLE;
import static org.lsst.ccs.gconsole.plugins.commandbrowser.Browser.ICON_UNLOCKED;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;

/**
 * A {@link Browser} that works with one particular subsystem.
 *
 * @author onoprien
 */
public class BrowserOneSubsystem implements Browser {

// -- Fields : -----------------------------------------------------------------
    
    private Descriptor descriptor;
    private JPanel browserPanel;

    private AgentPanel agentPanel;
    private JLabel lockLabel, levelLabel, stateLabel;
    private JFormattedTextField levelField;
    
    private AgentStatus agent;
    private AgentStatusListener statusListener;
    private final AgentStatusAggregator statusAggregator = Console.getConsole().getStatusAggregator();
    
// -- Life cycle : -------------------------------------------------------------
    
    @Override
    public JComponent getPanel() {
        if (browserPanel == null) {
            
            browserPanel = new JPanel(new BorderLayout());
            
            // Agent dictionary panel:
            
            agentPanel = new AgentPanel();
            browserPanel.add(agentPanel, BorderLayout.CENTER);
            
            // Lock panel:
            
            Box lockPanel = Box.createHorizontalBox();
            browserPanel.add(lockPanel, BorderLayout.NORTH);
            
            lockPanel.add(Box.createRigidArea(Const.HDIM));
            lockLabel = new JLabel();
            lockPanel.add(lockLabel);
            lockLabel.setMinimumSize(new Dimension(ICON_UNLOCKED.getIconWidth(), ICON_UNLOCKED.getIconHeight()));
            lockLabel.addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    if (e.getClickCount() == 2) {
                        AgentLock lock = agent.getLock();
                        if (lock == null) {
                            agent.setLock(true);
                        } else if (lock.getOwnerAgentName().equals(Console.getConsole().getAgentInfo().getName())) {
                            agent.setLock(false);
                        }
                    }
                }
                
            });
            lockPanel.add(Box.createRigidArea(Const.HDIM));
            levelLabel = new JLabel("Level:");
            lockPanel.add(levelLabel);
            lockPanel.add(Box.createRigidArea(Const.HDIM));
            levelField = new JFormattedTextField(0);
            lockPanel.add(levelField);
            levelField.setColumns(3);
            levelField.setMaximumSize(new Dimension(levelField.getPreferredSize().width, Integer.MAX_VALUE));
            levelField.addActionListener(e -> {
                try {
                    int level = (Integer) levelField.getValue();
                    if (agent != null) agent.setLevel(level);
                } catch (NumberFormatException x) {
                }
            });
            lockPanel.add(Box.createRigidArea(Const.HDIM));
            stateLabel = new JLabel("OFFLINE");
            lockPanel.add(stateLabel);
            lockPanel.add(Box.createHorizontalGlue());
            
            // Start listening for agent connections:
            
            statusListener = new AgentStatusListener() {
                @Override
                public void connect(AgentStatusEvent event) {
                    SwingUtilities.invokeLater(() -> {
                        updateStateDisplay();
                    });
                }
                @Override
                public void disconnect(AgentStatusEvent event) {
                    SwingUtilities.invokeLater(() -> {
                        updateStateDisplay();
                    });
                }
                @Override
                public void statusChanged(AgentStatusEvent event) {
                    SwingUtilities.invokeLater(() -> {
                        updateStateDisplay();
                    });
                }
            };
            
            agent = new AgentStatus(descriptor.getAgent());
            agent.init();
            agent.addListener(e -> updateLockDisplay());
            updateLockDisplay();
            agentPanel.setAgent(agent);
            List<String> channels = Arrays.asList("/state/PhaseState", "/state/OperationalState", "/state/ConfigurationState",  "/state/AlertState",  "/state/CommandState");
            statusAggregator.addListener(statusListener, Collections.singleton(descriptor.getAgent()), channels);
            
        }
        return browserPanel;
    }

    @Override
    public void shutdown() {
        if (agent != null) {
            agent.shutdown();
        }
        statusAggregator.removeListener(statusListener);
        browserPanel = null;
    }
    

    @Override
    public String getName() {
        return descriptor.getName();
    }
    
// -- Updating : ---------------------------------------------------------------
    
    private void updateLockDisplay() {
        if (agent == null) {
            stateLabel.setText("OFFLINE");
        } else {
            AgentLock agentLock = agent.getLock();
            if (agentLock == null) {
                lockLabel.setIcon(ICON_UNLOCKED);
                lockLabel.setToolTipText("double-click to lock");
            } else if (agentLock.getOwnerAgentName().equals(Console.getConsole().getAgentInfo().getName())) {
                lockLabel.setIcon(ICON_LOCKED);
                lockLabel.setToolTipText("double-click to unlock");
            } else {
                lockLabel.setIcon(ICON_UNAVAILABLE);
                lockLabel.setToolTipText("locked by: "+ agentLock.getAgentName());
            }
        }
        levelField.setValue(agent.getLevel());
    }
    
    private void updateStateDisplay() {
        StateBundle state = statusAggregator.getAgentState(agent.getName());
        if (state == null) {
            stateLabel.setText("OFFLINE");
            return;
        }
        StringBuilder sb = new StringBuilder("<html>");
        PhaseState phase = state.getState(PhaseState.class);
        if (phase != null) sb.append(phase).append(" - ");
        OperationalState ops = state.getState(OperationalState.class);
        if (ops != null) {
            switch (ops) {
                case NORMAL:
                    sb.append("<font color=\"#008800\">NORMAL</font>"); break;
                case ENGINEERING_FAULT:
                    sb.append("<font color=\"#DD0000\">ENGINEERING_FAULT</font>"); break;
                default:
                    sb.append(ops);
            }
            sb.append(" - ");
        }
        ConfigurationState con = state.getState(ConfigurationState.class);
        if (con != null) {
            switch (con) {
                case DIRTY:
                    sb.append("<font color=\"#0000DD\">MODIFIED CONFIGURATION</font>"); break;
                default:
                    sb.append(con);
            }
            sb.append(" - ");
        }
        AlertState als = state.getState(AlertState.class);
        if (als != null && !als.equals(AlertState.NOMINAL)) {
            switch (als) {
                case WARNING:
                    sb.append("<font color=\"#0000DD\">WARNING</font>"); break;
                case ALARM:
                    sb.append("<font color=\"#DD0000\">ALARM</font>"); break;
            }
            sb.append(" - ");
        }
        CommandState com = state.getState(CommandState.class);
        if (com != null) {
            switch (com) {
                case READY:
                    sb.append("<font color=\"#008800\">READY</font>"); break;
                case ACTIVE:
                    sb.append("<font color=\"#DD0000\">ACTIVE</font>"); break;
            }
        }
        stateLabel.setText(sb.toString());
    }

// -- Saving/restoring : -------------------------------------------------------
    
    @Override
    public Descriptor save() {
        if (agentPanel == null) {
            descriptor.setAgentPanel(null);
        } else {
            descriptor.setAgentPanel(agentPanel.save());
        }
        if (browserPanel != null) {
            descriptor.setPage(DataPanelDescriptor.get(browserPanel));
        }
        return descriptor;
    }
    
    @Override
    public void restore(Serializable descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
            if (agentPanel != null) {
                AgentPanel.Descriptor apDesc = this.descriptor.getAgentPanel();
                if (apDesc != null) agentPanel.restore(apDesc);
            }
        } else {
        }
    }
    
    static public class Descriptor extends Browser.Descriptor {
        
    }
    
}
