package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.BorderLayout;
import java.util.*;
import java.util.AbstractMap.SimpleEntry;
import java.util.function.Function;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.base.filter.PageFilter;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.services.persist.Persistable;

/**
 * Monitoring view that displays several other views in tabs.
 *
 * @author onoprien
 */
public class TabbedView extends AbstractMonitorView {

// -- Fields : -----------------------------------------------------------------
    
    private Descriptor descriptor = new Descriptor();
    
    private final JComponent panel = new JPanel(new BorderLayout());
    private JTabbedPane tabbedPane;
    
    private final LinkedHashMap<String,AbstractMonitorView> views = new LinkedHashMap<>();
    
    private Function<String,AbstractMonitorView> viewFactory;
    

// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = "MonitorView",
            name = "Tabbed View",
            path = "Built-In/Tabbed",
            description = "Monitoring data view that displays its data in a set of tabs.")
    public TabbedView() {
        viewFactory = name -> {
            AbstractMonitorView view = new TreeView();
            if (formatter != null) {
                view.setFormatter(formatter);
            }
            return view;
        };
        panel.add(new MonitorDisplay.EMPTY());
    }
    
    
// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the view to be displayed in the tab {@code name}.
     * @param name
     * @param view 
     */
    public void setTab(String name, AbstractMonitorView view) {
        // FIXME
    }
    
    /**
     * Sets the factory used to create views for tabs for which the view has not been explicitly set.
     * @param factory Factory that returns view instance given the tab name.
     */
    public void setViewFactory(Function<String,AbstractMonitorView> factory) {
        viewFactory = s -> {
            AbstractMonitorView view = factory.apply(s);
            if (formatter != null) {
                view.setFormatter(formatter);
            }
            return view;
        };
    }
    
// -- Implementing MonitorView : -----------------------------------------------

    @Override
    public JComponent getPanel() {
        return panel;
    }


// -- Calling AbstractMonitorView2 hooks : --------------------------------------    

    @Override
    public void addChannels(List<AgentInfo> agents, List<Map.Entry<String,AgentChannel>> channels) {
        Map<String,List<Map.Entry<String,AgentChannel>>> allTabs = new LinkedHashMap<>();
        channels.forEach(e ->  {
            String[] ss = split(e.getKey());
            AgentChannel ch = e.getValue();
            List<Map.Entry<String,AgentChannel>> oneTab = allTabs.get(ss[0]);
            if (oneTab == null) {
                oneTab = new ArrayList<>(128);
                allTabs.put(ss[0], oneTab);
            }
            oneTab.add(new SimpleEntry(ss[1], ch));
        });
        allTabs.forEach((tab, cc) -> {
            AbstractMonitorView view = views.get(tab);
            if (view == null) {
                view = viewFactory.apply(tab);
                if (filter != null) {
                    view.setFilter(new PageFilter(filter, tab));
                }
                TreeMap<String,AbstractMonitorView.Descriptor> m = getDescriptor().getTabs();
                if (m != null) {
                    AbstractMonitorView.Descriptor d = m.get(tab);
                    if (d != null) {
                        view.restore(d);
                    }
                }
                views.put(tab, view);
                switch (views.size()) {
                    case 0:
                        tabbedPane = null;
                        panel.removeAll();
                        panel.add(new MonitorDisplay.EMPTY());
                        break;
                    case 1:
                        tabbedPane = null;
                        panel.removeAll();
                        panel.add(view.getPanel());
                        break;
                    case 2:
                        panel.removeAll();
                        tabbedPane = new JTabbedPane();
                        views.forEach((name,v) -> tabbedPane.add(name, v.getPanel()));
                        panel.add(tabbedPane);
                        break;
                    default:
                        tabbedPane.add(tab, view.getPanel());
                }
            }
            view.addChannels(agents, cc);
        });
    }

    @Override
    public void removeChannels(List<AgentInfo> agents, List<Map.Entry<String,AgentChannel>> channels) {
        Map<String,List<Map.Entry<String,AgentChannel>>> allTabs = new LinkedHashMap<>();
        channels.forEach(e ->  {
            String[] ss = split(e.getKey());
            AgentChannel ch = e.getValue();
            List<Map.Entry<String,AgentChannel>> oneTab = allTabs.get(ss[0]);
            if (oneTab == null) {
                oneTab = new ArrayList<>(128);
                allTabs.put(ss[0], oneTab);
            }
            oneTab.add(new SimpleEntry(ss[1], ch));
        });
        allTabs.forEach((tab, cc) -> {
            AbstractMonitorView view = views.get(tab);
            if (view != null) {
                view.removeChannels(agents, cc);
                if (view.isEmpty()) {
                    views.remove(tab);
                    switch (views.size()) {
                        case 0:
                            tabbedPane = null;
                            panel.removeAll();
                            panel.add(new MonitorDisplay.EMPTY());
                            break;
                        case 1:
                            tabbedPane = null;
                            panel.removeAll();
                            panel.add(views.values().iterator().next().getPanel());
                            break;
                        default:
                            tabbedPane.remove(view.getPanel());
                    }
                }
            }
        });
    }

    @Override
    public void updateChannels(List< Map.Entry<String,Map.Entry<AgentChannel,List<String>>> > channels) {
        Map<String, List<Map.Entry<String,Map.Entry<AgentChannel,List<String>>>>> allTabs = new LinkedHashMap<>();
        channels.forEach(e ->  {
            String[] ss = split(e.getKey());
            List< Map.Entry<String,Map.Entry<AgentChannel,List<String>>> > oneTab = allTabs.get(ss[0]);
            if (oneTab == null) {
                oneTab = new ArrayList<>(128);
                allTabs.put(ss[0], oneTab);
            }
            oneTab.add(new SimpleEntry(ss[1], e.getValue()));
        });
        allTabs.forEach((tab, cc) -> {
            AbstractMonitorView view = views.get(tab);
            if (view != null) {
                view.updateChannels(cc);
            }
        });
    }

    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * JavaBean that contains information required to re-create this view in its current state.
     */
    static public class Descriptor extends Persistable.Descriptor {

        private TreeMap<String, Persistable.Descriptor> tabs;

        public TreeMap<String, Persistable.Descriptor> getTabs() {
            return tabs;
        }

        public void setTabs(TreeMap<String, Persistable.Descriptor> tabs) {
            this.tabs = tabs;
        }

        @Override
        public Descriptor clone() {
            Descriptor desc = (Descriptor) super.clone();
            if (desc.tabs != null) {
                desc.tabs = new TreeMap<>(desc.tabs);
                desc.tabs.entrySet().forEach(e -> e.setValue(e.getValue().clone()));
            }
            return desc;
        }
        
    }
    
    /**
     * Returns a descriptor that contains information required to re-create this view in its current state.
     * @return View descriptor.
     */
    @Override
    public Descriptor save() {
        Descriptor desc = new Descriptor();
        if (!views.isEmpty()) {
            TreeMap<String, Persistable.Descriptor> dd = new TreeMap<>();
            desc.setTabs(dd);
            for (Map.Entry<String,AbstractMonitorView> e : views.entrySet()) {
                dd.put(e.getKey(), e.getValue().save());
            }
        }
        return desc;
    }
    
    /**
     * Restores this view to the state described by the provided descriptor, to the extent possible.
     * @param descriptor View descriptor.
     */
    @Override
    public void restore(PersistableMonitorView.Descriptor descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
            TreeMap<String, AbstractMonitorView.Descriptor> dd = this.descriptor.getTabs();
            if (dd != null) {
                for (Map.Entry<String, AbstractMonitorView.Descriptor> e : dd.entrySet()) {
                    AbstractMonitorView v = views.get(e.getKey());
                    if (v != null) {
                        v.restore(e.getValue());
                    }
                }
            }
        }
    }

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    /**
     * Splits display path produced by the filter onto [tab, path inside tab].
     * The implementation provided by this class splits around "//".
     * 
     * @param displayPath
     * @return 
     */
    protected String[] split(String displayPath) {
        String[] out = new String[2];
        int i = displayPath.indexOf("//");
        if (i == -1) {
            out[0] = "";
            out[1] = displayPath;
        } else {
            out[0] = displayPath.substring(0, i);
            out[1] = displayPath.substring(i+2);
        }
        return out;
    }

}
