package org.lsst.ccs.gconsole.services.aggregator;

/**
 * Classes that implement this interface can be registered with {@code AgentStatusAggregator}
 * to receive notifications when an agent connects or disconnects from the buses, or when 
 * there are changes in data channels published by one of the connected agents.
 * Typically, a listener is registered with an {@link AgentChannelsFilter}
 * or with explicit sets of agents and channels to limit events that trigger notifications.
 * See {@link AgentStatusAggregator} for details on how the notifications are filtered.
 * Note that due to asynchronous nature of communication over the buses, agents are
 * not guaranteed to remain in the reported state by the time listeners are notified.
 * <p>
 * For every remote agent, the listener will receive events in the following order:<ul>
 * <li>{@code connect(event)} - when the agent is first seen on the buses.
 * <li>{@code configure(event)} - when the agent is OPERATIONAL, and all its dictionaries are available.
 * <li>zero or more {@code statusChanged(event)} - whenever the data this listener is interested in changes.
 * <li>{@code disconnect(event)} - when the agent disconnects from the buses or goes into OFFLINE state.
 * </ul>
 * This sequence may be interrupted if the listener is removed from the status aggregator.
 * <p>
 * All methods are called consecutively on a dedicated thread. 
 * Implementations of these methods should return quickly. Any time consuming or
 * potentially blocking operations should be scheduled on separate threads.
 * <p>
 * Empty default implementations are provided for all methods.
 * 
 * @author onoprien
 */
public interface AgentStatusListener {
    
    /**
     * Called when a new agent connects to the buses.
     * Newly registered listeners are notified immediately if the agent is already connected.
     * <p>
     * The event passed to this method does not contain any information on channels
     * (corresponding methods return empty collections) or a reference to a status message.
     * The event can only be used to retrieve {@code AgentInfo} descriptor and a reference
     * to the {@link AgentStatusAggregator} instance.
     * 
     * @param event Event.
     */
    default public void connect(AgentStatusEvent event) {}
    
    /**
     * Called when the source agent reaches OPERATIONAL state and its all static information
     * (including channels dictionary) becomes available.
     * Newly registered listeners are notified immediately if the information is already available.
     * <p>
     * The event passed to this method contains a list of added channels. Methods accessing lists
     * of changed and removed channels will return empty collections. Since the configuration event
     * is not necessarily triggered by a status message, {@code getMessage()} method returns {@code null}.
     * 
     * @param event Event.
     */
    default public void configure(AgentStatusEvent event) {}
    
    /**
     * Called when an agent disconnects from the buses or goes into OFFLINE state.
     * <p>
     * The event passed to this method does not have any added or modified channels, but all previously existing 
     * channels are listed as removed. Since the disconnection event might or might not
     * be triggered by a status message, {@code getMessage()} method might return {@code null}.
     * 
     * @param event Event.
     */
    default public void disconnect(AgentStatusEvent event) {}
   
    
    /**
     * Called on data change.
     * 
     * @param event event Event.
     */
    default public void statusChanged(AgentStatusEvent event) {}
    
}
