package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;

/**
 * {@link DisplayChannel} that can be associated with multiple original data channels.
 *
 * @author onoprien
 */
public class DisplayChannelMulti implements DisplayChannel {

// -- Fields : -----------------------------------------------------------------
    
    private final String path;
    private ArrayList<AgentChannel> channels;
    private Updatable target;

// -- Life cycle : -------------------------------------------------------------
    
    public DisplayChannelMulti(String path) {
        this.path = path;
        channels = new ArrayList<>(1);
    }
    
// -- Implement DisplayChannel : -----------------------------------------------

    @Override
    public String getPath() {
        return path;
    }

    @Override
    public Updatable getTarget() {
        return target;
    }

    @Override
    public void setTarget(Updatable target) {
        this.target = target;
    }

    @Override
    public List<AgentChannel> getChannels() {
        return Collections.unmodifiableList(channels);
    }

    @Override
    public void setChannel(AgentChannel channel) {
        channels = new ArrayList<>(1);
        if (channel != null) {
            channels.add(channel);
        }
    }

    @Override
    public void setChannels(Collection<AgentChannel> channels) {
        this.channels = new ArrayList<>(channels);
    }

    @Override
    public void addChannel(AgentChannel channel) {
        String p = channel.getPath();
        ListIterator<AgentChannel> it = channels.listIterator();
        while (it.hasNext()) {
            AgentChannel ch = it.next();
            if (ch == channel) return;
            if (ch.getPath().equals(p)) {
                it.set(channel);
                return;
            }
        }
        channels.add(channel);
        channels.trimToSize();
    }

    @Override
    public boolean removeChannel(String path) {
        ListIterator<AgentChannel> it = channels.listIterator();
        while (it.hasNext()) {
            AgentChannel ch = it.next();
            if (ch.getPath().equals(path)) {
                it.remove();
                break;
            }
        }
        return channels.isEmpty();
    }

    @Override
    public boolean purgeOfflineChannels(List<String> agentNames) {
        Iterator<AgentChannel> it = channels.iterator();
        while (it.hasNext()) {
            AgentChannel channel = it.next();
            if (agentNames.contains(channel.getAgentName()) && DataProviderState.OFF_LINE.equals(channel.get(AgentChannel.Key.STATE))) {
                it.remove();
            }
        }
        channels.trimToSize();
        return channels.isEmpty();
    }

    
// -- Overriding Object : ------------------------------------------------------

    @Override
    public String toString() {
        return getPath();
    }

    @Override
    public boolean equals(Object obj) {
        return obj instanceof DisplayChannel && ((DisplayChannel)obj).getPath().equals(getPath());
    }

    @Override
    public int hashCode() {
        return getPath().hashCode();
    }

}
